<?php
/**
 * Holds font upload/download/delete functionality.
 *
 * @package BuddyBossApp\Admin\Appearance
 */

namespace BuddyBossApp\Admin\Appearance;

use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\Helpers\File;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;
use BuddyBossApp\Typography as BuddyBossAppTypography;
use FontLib\Exception\FontNotFoundException;

/**
 * Font typography class.
 */
class Typography {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Notice message.
	 *
	 * @var array $messages
	 */
	private $messages = array();

	/**
	 * Exception error message.
	 *
	 * @var string $error_exception_msg
	 */
	private static $error_exception_msg = '';

	/**
	 * If the current page to load HTML.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Get the instance of the class.
	 *
	 * @return Typography
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Typography constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		$this->messages = array(
			1 => array(
				'msg'   => __( 'Font uploaded successfully.', 'buddyboss-app' ),
				'class' => 'updated',
			),
			2 => array(
				'msg'   => __( 'Font deleted successfully.', 'buddyboss-app' ),
				'class' => 'updated',
			),
			3 => array(
				'msg'   => __( 'Error uploading, unable to fetch font name.', 'buddyboss-app' ),
				'class' => 'updated',
			),
			4 => array(
				'msg'   => __( 'The font you are trying to install is already installed.', 'buddyboss-app' ),
				'class' => 'updated',
			),
			5 => array(
				'msg'   => ! empty( self::$error_exception_msg ) ? self::$error_exception_msg : __( 'Something went wrong.', 'buddyboss-app' ),
				'class' => 'updated',
			),
		);

		add_action( 'admin_init', array( $this, 'process_upload_fonts' ) );
		add_action( 'wp_ajax_bbapp_typography_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_typography_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_bbapp_typography_gfonts', array( $this, 'json_fonts_response' ) );
		add_action( 'wp_ajax_nopriv_bbapp_typography_gfonts', array( $this, 'json_fonts_response' ) );
		add_action( 'wp_ajax_bbapp_font_previews', array( $this, 'font_previews' ) );
	}

	/**
	 * Functions tells & sets thats if current page is one where it's will render.
	 *
	 * @param bool $set True if need to load HTML.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Renders the typography screen
	 *
	 * @return bool|mixed
	 */
	public function render() {
		if ( ! current_user_can( 'manage_options' ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this settings.', 'buddyboss-app' ) );

			return false;
		}

		include bbapp()->plugin_dir . 'views/branding/typography.php';
	}

	/**
	 * Render screen fields.
	 *
	 * @param array $font_field Font field name.
	 * @param string $font_key   Font key.
	 */
	public function render_screen_fields_html( $font_field, $font_key ) {

		if ( ! isset( $font_field['family'] ) ) {
			$font_field['family'] = null;
		}

		if ( ! \BuddyBossApp\Typography::instance()->bbapp_get_installed_fonts_by_family( $font_field['family'] ) ) {
			$font_field['family'] = 'Default';
		}

		$fonts = \BuddyBossApp\Typography::instance()->get_installed_fonts();
		?>
		<table class="widefat bbapp-font-table" cellspacing="0" id="status">
			<?php

			$class_name = strtolower( str_replace( '.', '_', $font_key ) );
			?>
			<tr class="<?php echo esc_attr( $class_name ); ?> option_field entry-row">
				<td class="bbapp-font-table__label">
					<div class="option_field_label"><?php echo esc_html__( 'Font Family', 'buddyboss-app' ); ?></div>
					<span class="option_field_info"></span>
				</td>
				<td>
					<div class="bbapp-font-select-wrap">
						<select class="bbapp-app-fonts web" data-title="bbapp-style-<?php echo esc_attr( $font_key ); ?>" name="fonts[styles.fonts.<?php echo esc_attr( $font_key ); ?>][name]">
							<?php foreach ( $fonts as $font_name => $font ) { ?>
								<option data-font="<?php echo esc_attr( $font['name'] ); ?>" <?php selected( $font_field['family'], $font_name ); ?> value="<?php echo esc_attr( $font_name ); ?>"><?php echo esc_html( $font['name'] ); ?>
								</option>
							<?php } ?>
						</select>
					</div>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Render font list.
	 */
	public function render_front_list_html() {
		$have_fonts  = false;
		$font_scales = BuddyBossAppTypography::instance()->get_fonts_scales_opts();

		foreach ( \BuddyBossApp\Typography::instance()->get_installed_fonts() as $font_name => $font ) {
			$have_fonts = true;
			?>
			<div class="custom-font-block <?php echo ( 'Default' === $font_name ) ? 'custom-font-block--system' : 'custom-font-block--custom'; ?>">
				<div class="font-data">
					<div class="font-data-label">
						<?php echo esc_html( $font['name'] ); ?>
					</div>

				</div>
				<?php if ( isset( $font['fonts'] ) && ! empty( $font['fonts'] ) ) { ?>
					<div class="font-data">
						<div class="font-data-meta">
							<span><?php echo esc_attr( $font['fonts']['ext'] ); ?></span>
						</div>
					</div>
				<?php } ?>
				<div class="font-actions">
					<a href="#" data-href="<?php echo esc_url( admin_url( "admin-ajax.php?action=bbapp_font_previews&fontname={$font['name']}" ) ); ?>" data-fontname="<?php echo esc_attr( $font['name'] ); ?>" class="button-secondary button-secondary-preview" data-fancybox data-src="#bbapp-font-preview">
						<?php esc_attr_e( 'Preview', 'buddyboss-app' ); ?>
					</a>

					<?php
					if ( \BuddyBossApp\Typography::instance()->can_upload_fonts( get_current_user_id() ) ) {
						?>
						<a title="<?php esc_attr_e( 'Remove this font', 'buddyboss-app' ); ?>" href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin.php?page=bbapp-appearance&setting=typography&delete_font=' . $font_name ), 'bbapp_font_delete_nonce_' . $font_name, 'bbapp_font_delete_nonce' ) ); ?>" class="button-secondary button-secondary--delete" onclick="return confirm('<?php echo esc_attr__( 'Are you sure you want to delete this font?', 'buddyboss-app' ); ?>')">
							<?php esc_attr_e( 'Remove', 'buddyboss-app' ); ?>
						</a>
						<?php
					}
					?>
				</div>
				<div class="font-scale">
					<label>
						<span><?php esc_attr_e( 'Font Scale', 'buddyboss-app' ); ?></span>
						<div class="font-scale__text">
							<input type="number" class="font-scale__opt-text" min="<?php echo esc_attr( min( $font_scales ) ); ?>" max="<?php echo esc_attr( max( $font_scales ) ); ?>" value="<?php echo esc_attr( $font['scale'] ); ?>">
						</div>
						<div class="font-scale__range">
							<input type="range" min="<?php echo esc_attr( min( $font_scales ) ); ?>" max="<?php echo esc_attr( max( $font_scales ) ); ?>" value="<?php echo esc_attr( $font['scale'] ); ?>" class="font-scale__opt" name="font_scale[<?php echo esc_attr( $font_name ); ?>]"/>
						</div>
					</label>
				</div>
			</div>
			<?php
		}

		if ( empty( $have_fonts ) ) {
			printf( '<p class="bbapp-no-custom-fonts">%1$s</p>', esc_html__( 'You don\'t have any custom fonts installed yet.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Outputs the google fonts list to json on ajax endpoint for dropDown purpose.
	 */
	public function json_fonts_response() {
		if ( ! is_user_logged_in() ) {
			echo wp_json_encode( array(), JSON_PRETTY_PRINT );
			exit;
		}

		$fonts    = $this->get_google_fonts();
		$response = array();

		foreach ( $fonts as $font ) {
			$preview = array();

			if ( ! empty( $font['variants'] ) ) {
				foreach ( $font['variants'] as $font_variant ) {
					$preview[ $font_variant ] = __( 'Almost before we knew it, we had left the ground.', 'buddyboss-app' );
				}
			}
			$response[ $font['family'] ] = array(
				'family'   => $font['family'],
				'category' => $font['category'],
				'variants' => $font['variants'],
				'subsets'  => $font['subsets'],
				'preview'  => $preview,
			);
		}

		echo wp_json_encode( $response, JSON_PRETTY_PRINT );
		exit;
	}

	/**
	 * Returns the google fonts API key.
	 *
	 * @return string
	 */
	public function get_google_font_key() {
		$global_settings = ManageApp::instance()->get_settings();

		if ( isset( $global_settings['google.font_service_key'] ) ) {
			return $global_settings['google.font_service_key'];
		}

		return '';
	}

	/**
	 * Return the google fonts, from google server.
	 *
	 * @return array
	 */
	public function get_google_fonts() {
		$google_font_key = $this->get_google_font_key();

		if ( empty( $google_font_key ) ) {
			return array();
		}

		$cache_key = '_bbapp_google_typography_fonts';
		$cache     = get_transient( $cache_key );

		if ( is_array( $cache ) ) {
			return $cache;
		}

		$response = bbapp_remote_get( 'https://www.googleapis.com/webfonts/v1/webfonts?key=' . $google_font_key );

		if ( is_wp_error( $response ) ) {
			return array();
		}

		$response = json_decode( $response['body'], true );

		if ( ! isset( $response['items'] ) ) {
			return array();
		}

		set_transient( $cache_key, $response['items'], 1 * HOUR_IN_SECONDS );

		return $response['items'];
	}

	/**
	 * Output the font form nonce.
	 */
	public function upload_font_form_nonce() {
		wp_nonce_field(
			'bb_font_upload_form_nonce',
			'bb_font_upload_form_nonce'
		);
	}

	/**
	 * Process uploaded fonts.
	 *
	 * @return bool|void
	 * @throws FontNotFoundException Font not found exception message.
	 */
	public function process_upload_fonts() {
		if ( ! \BuddyBossApp\Typography::instance()->can_upload_fonts( get_current_user_id() ) ) {
			return false;
		}

		// Delete Font.
		$nonce_delete_font_get = ( ! empty( $_GET['bbapp_font_delete_nonce'] ) ) ? sanitize_text_field( wp_unslash( $_GET['bbapp_font_delete_nonce'] ) ) : '';
		$font_name             = ( ! empty( $_GET['delete_font'] ) ) ? sanitize_text_field( wp_unslash( $_GET['delete_font'] ) ) : '';
		if ( isset( $font_name ) && wp_verify_nonce( $nonce_delete_font_get, 'bbapp_font_delete_nonce_' . $font_name ) ) {
			$fonts    = \BuddyBossApp\Typography::instance()->bbapp_get_font_details( $font_name );
			$font_dir = \BuddyBossApp\Typography::instance()->get_fonts_upload_dir();

			// Delete file.
			foreach ( $fonts['fonts_family'] as $variant_name => $font_variant ) {
				$font_file_path = trailingslashit( $font_dir ) . $font_variant['file'];
				$base64_path    = trailingslashit( $font_dir ) . "{$font_variant['file']}.b64.txt";

				wp_delete_file( $font_file_path );
				wp_delete_file( $base64_path );
			}

			$this->update_fonts_information( $font_name, true );

			wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=2' ) );
			exit;
		}

		// Upload Custom Font.
		$nonce_post = ( ! empty( $_POST['bb_font_upload_form_nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bb_font_upload_form_nonce'] ) ) : '';
		if ( wp_verify_nonce( $nonce_post, 'bb_font_upload_form_nonce' ) ) {
			$custom_font_name = ( ! empty( $_POST['bbapp-custom-font-name'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp-custom-font-name'] ) ) : '';

			// Note used because custom font uploader disable/hide/remove.
			if ( ! empty( $custom_font_name ) ) {
				if ( ! isset( $_POST['bbapp_custom_font'] ) || empty( $_POST['bbapp_custom_font'] ) ) {
					wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=3' ) );
					exit;
				}

				if ( BuddyBossAppTypography::instance()->bbapp_get_installed_fonts_by_family( $custom_font_name ) ) {
					wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=4' ) );
					exit;
				}

				$update_fonts[ $custom_font_name ] = array(
					'name' => $custom_font_name,
				);

				if ( ! empty( $_POST['bbapp_custom_font'] ) ) {
					foreach ( map_deep( wp_unslash( $_POST['bbapp_custom_font'] ), 'sanitize_text_field' ) as $font_style => $bbapp_custom_font ) {
						$update_fonts[ $custom_font_name ]['fonts_family'][ $font_style ] = $this->save_custom_font( $font_style, $bbapp_custom_font );
					}
				}

				if ( ! empty( $update_fonts ) ) {
					$update = $this->update_fonts_information( $update_fonts );
				}

				if ( ! $update ) {
					wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=3' ) );
					exit;
				}

				// Add font into installed font list.
				if ( true === $update ) {
					wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=1' ) );
					exit;
				}
			}

			// Import Google Fonts.
			$google_font_post    = ( ! empty( $_POST['bbapp-google-font'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp-google-font'] ) ) : '';
			$google_font_variant = ( ! empty( $_POST['bbapp-google-font-variant'] ) ) ? map_deep( wp_unslash( $_POST['bbapp-google-font-variant'] ), 'sanitize_text_field' ) : '';

			if ( ! empty( $google_font_post ) ) {
				$this->bbapp_google_font_upload( $google_font_post, $google_font_variant );
			}
		}
	}


	/**
	 * Save branding settings.
	 */
	public function process_submit() {
		$nonce_post = ( ! empty( $_POST['bbapp_save_typography_options'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_save_typography_options'] ) ) : '';

		if ( wp_verify_nonce( $nonce_post, 'bbapp_save_typography_options' ) ) {

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( "You don't have permission to access this page." );
			}

			// Save Uploads.
			$upload_fields      = BuddyBossAppTypography::instance()->font_fields();
			$typography_options = BuddyBossAppTypography::instance()->get_options();

			foreach ( $upload_fields as $field_name => $field ) {
				$typography_options[ $field_name ]['family'] = ! empty( $_POST['fonts'][ "styles.fonts.{$field_name}" ]['name'] ) ? sanitize_text_field( wp_unslash( $_POST['fonts'][ "styles.fonts.{$field_name}" ]['name'] ) ) : '';

				$font = BuddyBossAppTypography::instance()->get_installed_font( $typography_options[ $field_name ]['family'] );

				if ( ! empty( $font ) ) {
					// Add isset condition for the weight and ext because if they not exists then its displaying error in debug log.
					$typography_options[ $field_name ]['weight'] = isset( $font['weight'] ) ? $font['weight'] : '';
					$typography_options[ $field_name ]['type']   = isset( $font['ext'] ) ? $font['ext'] : '';
				}
			}

			BuddyBossAppTypography::instance()->set_options( $typography_options );

			// Save Scales.
			$scales_opts         = BuddyBossAppTypography::instance()->get_fonts_scales_opts();
			$post_scales         = ( isset( $_POST['font_scale'] ) && is_array( $_POST['font_scale'] ) ) ? map_deep( wp_unslash( $_POST['font_scale'] ), 'sanitize_text_field' ) : array();
			$font_scales         = array();
			$get_installed_fonts = BuddyBossAppTypography::instance()->get_installed_fonts();

			foreach ( $get_installed_fonts as $font_name => $font ) {
				$font_scales[ $font_name ] = ( isset( $post_scales[ $font_name ] ) && in_array( (int) $post_scales[ $font_name ], $scales_opts, true ) ) ? $post_scales[ $font_name ] : 100;
			}

			BuddyBossAppTypography::instance()->set_fonts_scales( $font_scales );

			wp_send_json_success( __( 'Changes Saved', 'buddyboss-app' ) );
		} else {
			wp_send_json_error( __( 'Security Token Error', 'buddyboss-app' ) );
		}
	}

	/**
	 * Google font upload in our WordPress site.
	 *
	 * @param string       $font_name     Font name.
	 * @param array|string $font_variants Array of font variants.
	 *
	 * @throws FontNotFoundException Font libracry exception message.
	 */
	public function bbapp_google_font_upload( $font_name, $font_variants ) {
		$font_variants   = explode( ',', $font_variants );
		$upload_file_dir = BuddyBossAppTypography::instance()->get_fonts_upload_dir();
		$upload_files    = array();

		foreach ( $font_variants as $variant_name ) {
			$file_name                     = uniqid() . '-' . $font_name . '-' . $variant_name;
			$upload_files[ $variant_name ] = trailingslashit( $upload_file_dir ) . sanitize_title( $file_name );
		}

		$gfonts = $this->get_google_fonts();

		foreach ( $font_variants as $variant_name ) {
			// As it's single font we can name it as we want.
			$font_family = "{$font_name}-" . ucfirst( $variant_name );

			if ( BuddyBossAppTypography::instance()->bbapp_get_installed_fonts_by_family( $font_family ) ) {
				wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=4' ) );
				exit;
			}
		}

		// Find the Font to Download it.
		$selected_font = false;

		foreach ( $gfonts as $font ) {
			if ( $font_name === $font['family'] ) {
				$selected_font = $font;
				break;
			}
		}

		if ( ! $selected_font ) {
			wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=3' ) );
			exit;
		}

		$update                     = false;
		$update_fonts[ $font_name ] = array(
			'name' => $font_name,
		);

		// fix the keys of $selected_font.
		foreach ( $selected_font['files'] as $k => $v ) {
			unset( $selected_font['files'][ $k ] );

			if ( 'regular' === $k ) {
				$k = '400';
			}

			if ( 'italic' === $k ) {
				$k = '400italic';
			}

			$selected_font['files'][ $k ] = $v;
		}

		foreach ( $font_variants as $variant_name ) {
			$download_file = false;

			// get the download file.
			if ( isset( $selected_font['files'][ $variant_name ] ) ) {
				$download_file = $selected_font['files'][ $variant_name ];
			}

			if ( ! $download_file ) {
				wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=3' ) );
				exit;
			}

			$font_ext    = explode( '.', basename( $download_file ) );
			$font_ext    = end( $font_ext );
			$upload_file = $upload_files[ $variant_name ] . '.' . $font_ext;

			try {
				// Download Font.
				$response = wp_remote_get( $download_file );
				if ( ( ! is_wp_error( $response ) ) && ( 200 === wp_remote_retrieve_response_code( $response ) ) ) {
					$font_file_content = $response['body'];
					// Write the file using put_contents instead of fopen(), etc.
					BBAPP_File::file_handler( $upload_file, $font_file_content );
					$update_fonts[ $font_name ]['fonts_family'][ $variant_name ] = $upload_file;
				}
			} catch ( \Exception $ex ) {
				// Handle Exception.
				self::$error_exception_msg = $ex->getMessage();
				wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=5' ) );
				exit;
			}
		}

		if ( ! empty( $update_fonts ) ) {
			$update = $this->update_fonts_information( $update_fonts );

			if ( true === (bool) $update ) {
				wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=1' ) );
				exit;
			}
		}

		wp_safe_redirect( bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&msg=3' ) );
		exit;
	}

	/**
	 * Update fonts information on font json file.
	 *
	 * @param array|string $update_fonts Updated fonts array or font name.
	 * @param false        $delete       Fonts deleted or not.
	 *
	 * @return bool
	 * @throws FontNotFoundException Font exception message.
	 */
	public function update_fonts_information( $update_fonts, $delete = false ) {
		$font_dir   = BuddyBossAppTypography::instance()->get_fonts_upload_dir();
		$fonts_info = BBAPP_File::read_file( trailingslashit( $font_dir ) . 'fonts.json' );
		$fonts_info = (array) json_decode( $fonts_info, true );

		if ( false === $delete ) {
			foreach ( $update_fonts as $index => $update_font ) {
				$font_name = false;

				foreach ( $update_font['fonts_family'] as $font_variant => $font_path ) {
					if ( empty( $font_path ) ) {
						unset( $update_fonts[ $index ]['fonts_family'][ $font_variant ] ); // unset empty one.
						continue;
					}

					$file = basename( $font_path );

					try {
						// Get the font real name for font family.
						$font = \BuddyBossApp\Library\Composer::instance()->font_lib_instance()->font_lib( $font_path );

						if ( $font ) { // it's used for verify font.
							unset( $update_fonts[ $index ]['fonts_family'][ $font_variant ] ); // unset old one.

							if ( 'regular' === $font_variant ) {
								$font_variant = '400';
							}

							if ( 'italic' === $font_variant ) {
								$font_variant = '400italic';
							}

							if ( 'medium' === $font_variant ) {
								$font_variant = '500';
							}

							if ( 'medium_italic' === $font_variant ) {
								$font_variant = '500italic';
							}

							if ( 'semibold' === $font_variant ) {
								$font_variant = '600';
							}

							if ( 'semibold_italic' === $font_variant ) {
								$font_variant = '600italic';
							}

							if ( 'bold' === $font_variant ) {
								$font_variant = '700';
							}

							if ( 'bold_italic' === $font_variant ) {
								$font_variant = '700italic';
							}

							$font_name = $font->getFontPostscriptName();
							$update_fonts[ $index ]['fonts_family'][ $font_variant ] = array(
								'name' => $font_name,
								'file' => $file,
							);
						} else {
							wp_die( esc_html__( 'Sorry the provided font is invalid or not supported format.', 'buddyboss-app' ) );
						}
					} catch ( \Exception $msg ) {
						wp_die( esc_html__( 'Sorry the provided font is invalid or not supported format.', 'buddyboss-app' ) );
					}
				}
			}

			$fonts_info[] = $update_fonts;
		} else {
			foreach ( $fonts_info as $key => $font_info ) {
				if ( isset( $font_info[ $update_fonts ] ) ) {
					unset( $fonts_info[ $key ] );
				}
			}
		}

		$fonts_info = wp_json_encode( array_values( $fonts_info ) );
		BBAPP_File::write_file( trailingslashit( $font_dir ) . 'fonts.json', $fonts_info );

		return true;
	}

	/**
	 * Return the fonts preview.
	 */
	public function font_previews() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			die( esc_html__( "You don't have permission to preview fonts", 'buddyboss-app' ) );
		}

		$font_name = ( isset( $_GET['fontname'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['fontname'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$get_fonts = \BuddyBossApp\Typography::instance()->get_installed_font( $font_name );
		$font_url  = \BuddyBossApp\Typography::instance()->get_fonts_upload_url();

		$missing_variants = array();
		$allowed_variants = array(
			'400'       => __( 'Regular', 'buddyboss-app' ),
			'400italic' => __( 'Regular Italic', 'buddyboss-app' ),
			'500'       => __( 'Medium', 'buddyboss-app' ),
			'500italic' => __( 'Medium Italic', 'buddyboss-app' ),
			'600'       => __( 'SemiBold', 'buddyboss-app' ),
			'600italic' => __( 'SemiBold Italic', 'buddyboss-app' ),
			'700'       => __( 'Bold', 'buddyboss-app' ),
			'700italic' => __( 'Bold Italic', 'buddyboss-app' ),
		);

		?>
		<h3><?php echo esc_html( $font_name ); ?></h3>
		<div class="bbapp-font-item bbapp-font-item-preview" style="">
			<div class="bbapp-load-font-preview">
				<?php
				echo "<style>\n";
				foreach ( $get_fonts['fonts_family'] as $font_variant => $font ) {
					$parse_variant = \BuddyBossApp\Typography::instance()->parse_variant_name( $font_variant );
					?>
					@font-face {
					font-family: '<?php echo esc_attr( $font_name ); ?>';
					<?php if ( isset( $parse_variant['italic'] ) ) : ?>
						font-style: italic;
					<?php endif; ?>
					font-weight: <?php echo isset( $parse_variant['weight'] ) ? esc_attr( $parse_variant['weight'] ) : ''; ?>;
					src: url(<?php echo esc_url( $font_url ) . '/' . esc_attr( $font['base_name'] ); ?>) format('truetype');
					}
					.bbapp-font-item__type-face-<?php echo esc_attr( $font_variant ); ?> { font-family:<?php echo esc_attr( $font_name ); ?>; font-style:<?php echo ( $parse_variant['italic'] ) ? 'italic' : ''; ?>;font-weight:<?php echo esc_attr( $parse_variant['weight'] ); ?> }
					<?php
				}
				echo "\n</style>";

				foreach ( $allowed_variants as $variant => $label ) {
					$font_url    = \BuddyBossApp\Typography::instance()->get_fonts_upload_url();
					$font_style  = '';
					$font_weight = '400';

					if ( false !== strpos( $variant, 'italic' ) ) {
						$font_style = 'italic';

						if ( '' === str_replace( 'italic', '', $variant ) ) {
							$font_weight = '400';
						} else {
							$font_weight = str_replace( 'italic', '', $variant );
						}
					} else {
						$font_weight = $variant;
					}

					if ( isset( $get_fonts['fonts_family'][ $variant ] ) ) {
						?>
						<div class="font-block-heading <?php echo esc_attr( $variant ); ?>">
							<label><?php echo esc_html( $allowed_variants[ $variant ] ); ?></label>
							<span class="bbapp-font-item__type-face bbapp-font-item__type-face-<?php echo esc_attr( $variant ); ?>">
						<?php esc_html_e( 'Almost before we knew it, we had left the ground.', 'buddyboss-app' ); ?>
					</span>
						</div>
						<?php
					} else {
						$missing_variants[] = $variant;
					}
				}
				?>
			</div>
		</div>
		<?php
		$message = '';

		if ( count( $missing_variants ) > 1 ) {
			$first_var  = array();
			$second_var = '';
			$i          = 1;

			foreach ( $missing_variants as $v ) {
				$label = ( strpos( $v, '500' ) !== false || strpos( $v, '700' ) !== false ) ? "<span class='s{$v}'>{$allowed_variants[$v]}</span>" : $allowed_variants[ $v ];
				$label = ( strpos( $v, 'italic' ) !== false ) ? "<span class='s{$v}'>{$label}</span>" : $label;

				if ( count( $missing_variants ) === $i ) {
					$second_var = $label;
				} else {
					$first_var[] = $label;
				}
				$i ++;
			}

			/* translators: %1$s: First character, %2$s: Second character */
			$message = sprintf( __( 'This font does not support %1$s or %2$s styles. You will be unable to view these styles in your app if you use this font.', 'buddyboss-app' ), implode( ', ', $first_var ), $second_var );
		} elseif ( count( $missing_variants ) === 1 ) {
			$first_var = '';

			foreach ( $missing_variants as $v ) {
				$first_var = ( strpos( $v, '500' ) !== false || strpos( $v, '700' ) !== false ) ? "<span class='s{$v}'>{$allowed_variants[$v]}</span>" : $allowed_variants[ $v ];
				$first_var = ( strpos( $v, 'italic' ) !== false ) ? "<span class='s{$v}'>{$first_var}</span>" : $first_var;
			}

			/* translators: %s: First character. */
			$message = sprintf( __( 'This font does not support %s. You will be unable to view these styles in your app if you use this font.', 'buddyboss-app' ), $first_var );
		}

		if ( ! empty( $message ) ) {
			printf( '<div class="bbapp-font-err"><p>%1$s</p></div>', wp_kses_post( $message ) );
		}
		exit;
	}

	/**
	 * Function for custom font fields.
	 *
	 * @return array[]
	 */
	public function custom_font_fields() {
		return array(
			'regular'         => array(
				'label'       => __( 'Regular', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => true,
			),
			'italic'          => array(
				'label'       => __( 'Italic', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'medium'          => array(
				'label'       => __( 'Medium', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'medium_italic'   => array(
				'label'       => __( 'Medium Italic', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'semibold'        => array(
				'label'       => __( 'SemiBold', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'semibold_italic' => array(
				'label'       => __( 'SemiBold Italic', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'bold'            => array(
				'label'       => __( 'Bold', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
			'bold_italic'     => array(
				'label'       => __( 'Bold Italic', 'buddyboss-app' ),
				'allowed_ext' => array( 'ttf', 'otf', 'woff', 'woff2' ),
				'value'       => '',
				'required'    => false,
			),
		);
	}

	/**
	 * Function to get custom font field.
	 *
	 * @param string $font_style Font style name.
	 *
	 * @return array
	 */
	public function custom_font_field( $font_style ) {
		$custom_font_fields = $this->custom_font_fields();

		return isset( $custom_font_fields[ $font_style ] ) ? $custom_font_fields[ $font_style ] : array();
	}

	/**
	 * Save custom fonts.
	 *
	 * @param string $font_style Font style name.
	 * @param array  $post_value Posted values.
	 *
	 * @return string|string[]
	 */
	public function save_custom_font( $font_style, $post_value ) {
		$upload_name = $post_value['field_name'];
		$settings    = $this->custom_font_field( $font_style );

		if ( ! isset( $settings['allowed_ext'] ) ) {
			$settings['allowed_ext'] = array();
		}

		$upload    = ! empty( $_FILES[ $upload_name ] ) ? map_deep( wp_unslash( $_FILES[ $upload_name ] ), 'sanitize_text_field' ) : array();
		$old_value = $this->get_value( $upload_name, $font_style );

		if ( isset( $upload['tmp_name'] ) && ! empty( $upload['tmp_name'] ) ) {
			$upload_file_dir = BuddyBossAppTypography::instance()->get_fonts_upload_dir();

			// Delete the old file.
			$old_file = $old_value;

			if ( ! empty( $old_file ) && file_exists( trailingslashit( $upload_file_dir ) . '/' . $old_file ) ) {
				wp_delete_file( trailingslashit( $upload_file_dir ) . '/' . $old_file );
			}

			// File Extension Validation.
			$file_name   = $upload['name'];
			$upload_file = trailingslashit( $upload_file_dir ) . $file_name;
			$ext         = pathinfo( $file_name, PATHINFO_EXTENSION );

			if ( ! in_array( $ext, $settings['allowed_ext'], true ) ) {
				return $old_value;
			}

			if ( move_uploaded_file( $upload['tmp_name'], $upload_file ) ) {
				// keep only relative path to upload dir.
				return $upload_file;
			}
		}

		return $old_value; // return old value if new was nt uploaded.
	}

	/**
	 * Function get empty value.
	 *
	 * @param string $upload_name File upload name.
	 * @param string $font_style  Font style.
	 *
	 * @return string
	 */
	public function get_value( $upload_name, $font_style ) {
		return '';
	}
}
