<?php
/**
 * Holds styling functionality related branding.
 *
 * @package BuddyBossApp\Admin\Appearance
 */

namespace BuddyBossApp\Admin\Appearance;

use BuddyBossApp\Permissions;
use BuddyBossApp\Styling as BuddyBossAppStyling;

/**
 * Styling class.
 */
class Styling {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Check if current page to load the screen.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Get the instance of the class.
	 *
	 * @return Styling
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Styling constructor.
	 */
	public function __construct() {     }

	/**
	 * Load method.
	 */
	public function load() {
		add_action( 'wp_ajax_bbapp_styling_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_styling_submit', array( $this, 'process_submit' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ) );
	}

	/**
	 * Functions tells & sets thats if current page is one where it's will render.
	 *
	 * @param bool $set Detetermine if screen needs to be loaded.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {
		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Returns the fields to be used in template.
	 */
	public function get_fields() {
		$fields = BuddyBossAppStyling::instance()->get_app_styling_fields();
		$fields = (array) $fields;
		return $fields;
	}

	/**
	 * Renders the styling screen
	 *
	 * @return bool|mixed
	 */
	public function render() {
		if ( ! current_user_can( 'manage_options' ) ) {
			echo '<p>' . esc_html( "You don't have permission to access these settings." ) . '</p>';
			return false;
		}

		$styling_options = BuddyBossAppStyling::instance()->get_options();
		$fields          = BuddyBossAppStyling::instance()->get_app_styling_fields();
		$values          = isset( $styling_options['styles'] ) ? $styling_options['styles'] : array();

		$fields               = (array) $fields;
		$styling_fields_array = (array) $fields['fields'];
		include bbapp()->plugin_dir . 'views/branding/colors.php';
	}

	/**
	 * Render screen fields.
	 *
	 * @param array  $fields      Array of fields.
	 * @param string $screen_name Screen name.
	 * @param array  $values      Array of filed values.
	 */
	public function render_screen_fields_html( $fields, $screen_name, $values ) {
		?>
		<table class="widefat" cellspacing="0" id="status">
			<?php
			$i = 0;
			foreach ( $fields as $field ) {
				$field = (array) $field;

				if ( $field['screen'] !== $screen_name ) {
					continue;
				}

				$i ++;
				$class_name = strtolower( str_replace( '.', '_', $field['name'] ) );
				?>
				<tr class="<?php echo ( ( $i % 2 ) ? 'alternate' : '' ) . ' ' . esc_attr( $class_name ) . ' ' . esc_attr( $field['type'] ); ?> option_field entry-row">
					<td>
						<div class="option_field_label"><?php echo esc_html( $field['label'] ); ?></div>
						<?php if ( isset( $field['description'] ) ) { ?>
							<span class="option_field_info"><?php echo wp_kses_post( $field['description'] ); ?></span>
						<?php } ?>
					</td>
					<td>
						<?php $this->render_field( $field, $values ); ?>
					</td>
				</tr>
				<?php
			}
			?>
		</table>
		<?php
	}

	/**
	 * Renders the styling singular field.
	 *
	 * @param array $field  Field data.
	 * @param array $values Field values.
	 */
	public function render_field( $field, $values ) {
		$field_name          = $field['name'];
		$field_value         = ( isset( $values[ $field_name ] ) ) ? $values[ $field_name ] : $field['value'];
		$class               = array( 'bbapp-field-item' );
		$class[]             = 'quick-field';
		$quick_field_default = 'data-default="' . esc_attr( $field['value'] ) . '"';

		if ( 'styles.colors.authFieldBgColor' === $field['name'] || 'styles.colors.regFieldBgColor' === $field['name'] ) {
			$class[]                 = 'color-picker';
			$quick_field_alpha       = 'data-alpha-enabled="true"';
			$quick_field_alpha_width = 'data-alpha-custom-width="1"';
			$quick_field_alpha_type  = 'data-alpha-color-type="rgba"';
		} else {
			$quick_field_alpha       = 'data-alpha-enabled="false"';
			$quick_field_alpha_width = 'data-alpha-custom-width="false"';
			$quick_field_alpha_type  = 'data-alpha-color-type="hex"';
		}

		$input_link = md5( $field_name . 'link' );
		$class[]    = $input_link;
		if ( 'color' === $field['type'] ) {
			$class[] = 'bbapp-color-picker';
		}$class = 'class="' . esc_attr( implode( ' ', $class ) ) . '"';

		if ( 'select' === $field['type'] ) {
			$options = (array) $field['options'];

			echo '<select name="styles[' . esc_attr( $field_name ) . ']" data-link="' . esc_attr( $input_link ) . '">';
			foreach ( $options as $option ) {
				$option = (array) $option;
				echo sprintf( '<option value="%1$s" %2$s %3$s>%4$s</option>', esc_attr( $option['value'] ), wp_kses_post( $class ), wp_kses_post( selected( $field_value, $option['value'], false ) ), esc_html( $option['label'] ) );
			}
			echo '</select>';
		}

		if ( 'color' === $field['type'] ) {
			echo sprintf( '<input name="styles[%1$s]" type="text" value="%2$s" data-link="%3$s" %4$s %5$s %6$s %7$s %8$s/>', esc_attr( $field_name ), esc_attr( $field_value ), esc_attr( $input_link ), wp_kses_post( $class ), wp_kses_post( $quick_field_alpha ), wp_kses_post( $quick_field_alpha_type ), wp_kses_post( $quick_field_alpha_width ), wp_kses_post( $quick_field_default ) );
		}
	}

	/**
	 * Save styling settings.
	 */
	public function process_submit() {
		$post_nonce = ( ! empty( $_POST['bbapp_save_styling_options'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_save_styling_options'] ) ) : '';

		if ( wp_verify_nonce( $post_nonce, 'bbapp_save_styling_options' ) ) {

			if ( ! Permissions::instance()->can_manage_app() ) {
				wp_send_json_error( __( "You don't have permission to access this page.", 'buddyboss-app' ) );
			}

			$styling_options = BuddyBossAppStyling::instance()->get_options();
			if ( ! empty( $_POST['styles'] ) ) {
				$styles = isset( $_POST['styles'] ) ? map_deep( wp_unslash( $_POST['styles'] ), 'sanitize_text_field' ) : array();
				// Convert RGB color codes to hex color codes.
				$styling_options['styles'] = map_deep( $styles, array( $this, 'convert_rgb_or_hsl_to_hex' ) );
				BuddyBossAppStyling::instance()->set_options( $styling_options );
			}

			wp_send_json_success( __( 'Changes Saved', 'buddyboss-app' ) );
		} else {
			wp_send_json_error( __( 'Security Token Error', 'buddyboss-app' ) );
		}
	}

	/**
	 * Helper function to convert RGB OR HSL to hex color codes.
	 *
	 * @param string $value The value to convert.
	 *
	 * @since 1.8.20
	 *
	 * @return mixed|string
	 */
	public function convert_rgb_or_hsl_to_hex( $value ) {

		// Check if the value is an RGB color code.
		if ( preg_match( '/^rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)$/', $value, $matches ) ) {
			// Extract RGB values from the color code.
			$red   = intval( $matches[1] );
			$green = intval( $matches[2] );
			$blue  = intval( $matches[3] );

			// Convert RGB to hex color code and return.
			$hex = sprintf( '#%02x%02x%02x', $red, $green, $blue );

			// Return the hex color code.
			return $hex;
		}

		// Check if the value is an HSL color code and convert it to HEX.
		if ( preg_match( '/^hsl\(\s*(\d+)\s*,\s*(\d+%)\s*,\s*(\d+%)\s*\)$/', $value, $matches ) ) {
			// Extract HSL values from the color code.
			$hue        = intval( $matches[1] );
			$saturation = intval( $matches[2] );
			$lightness  = intval( $matches[3] );

			// Normalize the HSL values
			$hue        = $hue % 360;
			$saturation = $saturation > 100 ? 100 : ( $saturation < 0 ? 0 : $saturation );
			$lightness  = $lightness > 100 ? 100 : ( $lightness < 0 ? 0 : $lightness );

			// Convert HSL to RGB.
			$saturation = $saturation / 100;
			$lightness  = $lightness / 100;
			$chroma     = ( 1 - abs( 2 * $lightness - 1 ) ) * $saturation;
			$hue_prime  = $hue / 60;
			$x          = $chroma * ( 1 - abs( $hue_prime % 2 - 1 ) );
			$rgb        = [ 0, 0, 0 ];

			if ( $hue_prime >= 0 && $hue_prime < 1 ) {
				$rgb = [ $chroma, $x, 0 ];
			} elseif ( $hue_prime >= 1 && $hue_prime < 2 ) {
				$rgb = [ $x, $chroma, 0 ];
			} elseif ( $hue_prime >= 2 && $hue_prime < 3 ) {
				$rgb = [ 0, $chroma, $x ];
			} elseif ( $hue_prime >= 3 && $hue_prime < 4 ) {
				$rgb = [ 0, $x, $chroma ];
			} elseif ( $hue_prime >= 4 && $hue_prime < 5 ) {
				$rgb = [ $x, 0, $chroma ];
			} elseif ( $hue_prime >= 5 && $hue_prime < 6 ) {
				$rgb = [ $chroma, 0, $x ];
			}

			$lightness_match = $lightness - $chroma / 2;

			// Scale RGB values and convert to hex.
			$to_hex = function ( $value ) {
				return str_pad( dechex( round( $value * 255 ) ), 2, '0', STR_PAD_LEFT );
			};

			$red   = $to_hex( $rgb[0] + $lightness_match );
			$green = $to_hex( $rgb[1] + $lightness_match );
			$blue  = $to_hex( $rgb[2] + $lightness_match );

			// Concatenate the hex values to form the final hex color code.
			$hex = '#' . $red . $green . $blue;

			// Return the hex color code.
			return $hex;
		}

		// Return the original value if it's not an RGB color code.
		return $value;
	}

	/**
	 * Enqueue CSS/JS.
	 */
	public function admin_enqueue_assets() {
		global $pagenow;

		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Just load it on pages we are using it.
		if ( 'admin.php' === $pagenow && 'bbapp-appearance' === $page_get ) {
			$styling_fields = BuddyBossAppStyling::instance()->get_app_styling_fields();
			// Localize.
			wp_localize_script(
				'bbapp-script',
				'styling_fields',
				array(
					'fields' => $styling_fields->fields,
				)
			);
		}
	}
}
