<?php
/**
 * Holds branding class functionality.
 *
 * @package BuddyBossApp\Admin\Appearance
 */

namespace BuddyBossApp\Admin\Appearance;

use BuddyBossApp\Branding as BuddyBossAppBranding;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\Helpers\File;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;

/**
 * Branding classs.
 */
class Branding {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Check if current page to load HTML.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Get the instance of the class.
	 *
	 * @return Branding
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Branding constructor.
	 */
	public function __construct() {
	}

	/**
	 * Load method.
	 */
	public function load() {
		add_action( 'wp_ajax_bbapp_branding_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_branding_submit', array( $this, 'process_submit' ) );
	}

	/**
	 * Functions tells & sets thats if current page is one where it's will render.
	 *
	 * @param bool $set Determine if screen needs to render.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Returns the fields to be used in template.
	 */
	public function get_fields() {
		$fields = (array) \BuddyBossApp\Branding::instance()->get_app_branding_fields();

		return $fields;
	}

	/**
	 * Renders the branding screen
	 *
	 * @return bool|mixed
	 */
	public function render() {

		if ( ! current_user_can( 'manage_options' ) ) {
			echo '<p>' . esc_html( "You don't have permission to access this settings." ) . '</p>';
			return false;
		}

		include bbapp()->plugin_dir . '/views/branding/images.php';
	}

	/**
	 * Get single field.
	 *
	 * @param string $field_name Field name.
	 *
	 * @return mixed|string
	 */
	public function get_single_field( $field_name ) {
		$fields = (array) $this->get_fields();

		return isset( $fields[ $field_name ] ) ? $fields[ $field_name ] : '';
	}

	/**
	 * Function to render the field HTML.
	 *
	 * @param string $field_name Field name.
	 * @param array  $branding_options Array of branding options.
	 *
	 * @return void
	 */
	public function render_field_html( $field_name, $branding_options ) {
		$field = $this->get_single_field( $field_name );
		$value = ( isset( $branding_options[ $field_name ] ) ) ? $branding_options[ $field_name ] : '';
		switch ( $field['type'] ) {
			case 'html':
				?>
				<div class="<?php echo esc_attr( $field_name ); ?> upload_field">
					<?php echo wp_kses_post( $field['html'] ); ?>
				</div>
				<?php
				break;
			case 'upload':
				$filename = $field_name . '.' . $field['format'];
				$fileinfo = $this->get_upload_file_info( $filename );
				$value    = '';

				if ( $fileinfo ) {
					$value = $fileinfo['fileurl'];
				}

				$fixedratio = (int) $field['fixedratio'];
				?>
				<div class="<?php echo esc_attr( $field_name ); ?> upload_field">
					<h3><?php echo esc_html( $field['label'] ); ?></h3>
					<?php
					if ( isset( $field['desc'] ) ) {
						echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
					}

					if ( isset( $field['notice'] ) ) {
						echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
					}
					?>
					<!--<div class="bbapp-upload-img-wrap">-->
						<div class="bbapp_upload_field_file_container">
							<span class="bbapp-upload-text"><?php esc_html_e( 'Upload Image', 'buddyboss-app' ); ?></span>
							<img src="<?php echo esc_url( $value ); ?>" style="<?php echo( empty( $value ) ? 'display:none' : '' ); ?>" class="bbapp_upload_preview"/>
							<input type="file" name="<?php echo esc_attr( $field_name ); ?>" class="bbapp_upload_field button" accept="image/*" data-height="<?php echo esc_attr( $field['height'] ); ?>" data-width="<?php echo esc_attr( $field['width'] ); ?>" data-class="<?php echo esc_attr( $field['class'] ); ?>" data-fixedratio="<?php echo esc_attr( $fixedratio ); ?>">
						</div>
						<div class="btn-finish-wrap">
							<button type="button" style="display:none" class="done_crop button-secondary" data-name="<?php echo esc_attr( $field_name ); ?>">
								<?php esc_html_e( 'Finish', 'buddyboss-app' ); ?>
							</button>
							<input name="<?php echo esc_attr( $field_name ); ?>_delete" id="<?php echo esc_attr( $field_name ); ?>_delete" class="delete_field" type="hidden" value=""/>
						</div>
					<!--</div>-->
					<div class="bbapp-img-button-wrap">
						<button style="<?php echo( ( empty( $value ) ) ? 'display:none' : '' ); ?>" type="button" class="replace button bbapp-img-button">
							<?php esc_html_e( 'Replace Image', 'buddyboss-app' ); ?>
						</button>
						<a style="<?php echo( ( empty( $value ) ) ? 'display:none' : '' ); ?>" href="#" data-id="<?php echo esc_attr( ManageApp::instance()->get_app_id() ); ?>" data-image="<?php echo esc_attr( $field_name ); ?>" class="delete button button-link">
							<?php esc_html_e( 'Remove Image', 'buddyboss-app' ); ?>
						</a>
					</div>
				</div>
				<?php
				break;
			case 'checkbox':
				?>
				<div class="<?php echo esc_attr( $field_name ); ?> checkbox_field">
				<input id="bbapp_field_<?php echo esc_attr( $field_name ); ?>" class="regular-text <?php echo esc_attr( $field_name ) . ' checkbox_field'; ?>" name="<?php echo esc_attr( $field_name ); ?>" type="checkbox" value="1" <?php checked( true, $value ); ?>>

				<?php
				if ( isset( $field['label_checkbox'] ) ) {
					?>
					<label for="bbapp_field_<?php echo esc_attr( $field_name ); ?>">
						<?php echo esc_html( $field['label_checkbox'] ); ?>
					</label>
					<?php
				}

				if ( isset( $field['desc'] ) ) {
					echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
				}

				if ( isset( $field['notice'] ) ) {
					echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
				}

				echo '</div>';
				break;
			case 'colorpicker':
				echo '<div class="' . esc_attr( $field_name ) . ' colorpicker_field">';

				if ( isset( $field['label'] ) && ! empty( $field['label'] ) ) {
					echo '<h3>' . esc_html( $field['label'] ) . '</h3>';
				}
				?>
				<input id="bbapp_field_<?php echo esc_attr( $field_name ); ?>" class="regular-text wp-color-picker bbapp-color-picker <?php echo esc_attr( $field_name ) . ' colorpicker_field'; ?>" name="<?php echo esc_attr( $field_name ); ?>" type="text" value="<?php echo esc_attr( $value ); ?>">

				<?php if ( isset( $field['label_checkbox'] ) ) { ?>
				<label for="bbapp_field_<?php echo esc_attr( $field_name ); ?>">
					<?php echo esc_html( $field['label_checkbox'] ); ?>
				</label>
					<?php
				}

				if ( isset( $field['desc'] ) ) {
					echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
				}

				if ( isset( $field['notice'] ) ) {
					echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
				}

				echo '</div>';
				break;
		}

		// This style used for cropping js to crop image box.
		if ( isset( $field['class'] ) && isset( $field['boundary_style_override'] ) ) {
			?>
		<style>
		.<?php echo esc_attr( $field['class'] ); ?> .cr-viewport.cr-vp-square {
			width: <?php echo esc_attr( $field['boundary_style_override']['width'] ); ?> !important;
			height: <?php echo esc_attr( $field['boundary_style_override']['height'] ); ?> !important;
		}
		</style>
			<?php
		}
	}

	/**
	 * Renders the uploads fields
	 */
	public function render_branding_fields() {
		$fields = $this->get_fields();

		// Render only required fields.
		foreach ( $fields as $k => $v ) {
			$v['super_admin'] = ( isset( $v['super_admin'] ) ) ? $v['super_admin'] : false;

			if ( $v['super_admin'] && ! bbapp_is_super_admin_page() ) {
				unset( $fields[ $k ] );
			}

			if ( ! $v['super_admin'] && bbapp_is_super_admin_page() && ! bbapp_is_admin_page() ) {
				unset( $fields[ $k ] );
			}
		}

		$branding_options = BuddyBossAppBranding::instance()->get_options();

		foreach ( $fields as $field_name => $field ) {

			$value = ( isset( $branding_options[ $field_name ] ) ) ? $branding_options[ $field_name ] : '';

			switch ( $field['type'] ) {
				case 'html':
					echo '<div class="' . esc_html( $field_name ) . ' upload_field">';
					echo wp_kses_post( $field['html'] );
					echo '</div>';
					break;
				case 'upload':
					$filename = $field_name . '.' . $field['format'];
					$fileinfo = $this->get_upload_file_info( $filename );
					$value    = '';

					if ( $fileinfo ) {
						$value = $fileinfo['fileurl'];
					}

					$fixedratio = (int) $field['fixedratio'];

					echo '<div class="' . esc_attr( $field_name ) . ' upload_field">';
					echo '<h3>' . esc_html( $field['label'] ) . '</h3>';
					echo '<div class="bbapp_upload_field_file_container">' . esc_html__( 'Upload Image', 'buddyboss-app' ) . '<input type="file" name="' . esc_attr( $field_name ) . '" class="bbapp_upload_field button" data-height="' . esc_attr( $field['height'] ) . '" data-width="' . esc_attr( $field['width'] ) . '" data-fixedratio="' . esc_attr( $fixedratio ) . '"></div>';
					echo '<button type="button" style="display:none" class="done_crop" data-name="' . esc_attr( $field_name ) . '">' . esc_html__( 'Finish', 'buddyboss-app' ) . '</button>';
					echo '<input name="' . esc_attr( $field_name ) . '_delete" id="' . esc_attr( $field_name ) . '_delete" class="delete_field" type="hidden" value="">';
					echo '<a style="' . ( ( empty( $value ) ) ? 'display:none' : '' ) . '" href="#" data-id="' . esc_attr( ManageApp::instance()->get_app_id() ) . '" data-image="' . esc_attr( $field_name ) . '" title="' . esc_html__( 'Delete Image', 'buddyboss-app' ) . '" class="delete">X</a>';
					echo '<img src="' . esc_url( $value ) . '" style="' . ( empty( $value ) ? 'display:none' : '' ) . '" class="bbapp_upload_preview" />';

					if ( isset( $field['desc'] ) ) {
						echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
					}

					if ( isset( $field['notice'] ) ) {
						echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
					}

					echo '</div>';
					break;
				case 'checkbox':
					echo '<div class="' . esc_attr( $field_name ) . ' checkbox_field">';
					?>
					<input id="bbapp_field_<?php echo esc_attr( $field_name ); ?>" class="regular-text <?php echo esc_attr( $field_name ) . ' checkbox_field'; ?>" name="<?php echo esc_attr( $field_name ); ?>" type="checkbox" value="1" <?php checked( true, $value ); ?>>

					<?php if ( isset( $field['label_checkbox'] ) ) { ?>
					<label for="bbapp_field_<?php echo esc_attr( $field_name ); ?>">
						<?php echo esc_html( $field['label_checkbox'] ); ?>
					</label>
						<?php
					}

					if ( isset( $field['desc'] ) ) {
						echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
					}

					if ( isset( $field['notice'] ) ) {
						echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
					}

					echo '</div>';
					break;
				case 'colorpicker':
					echo '<div class="' . esc_attr( $field_name ) . ' colorpicker_field">';

					if ( isset( $field['label'] ) && ! empty( $field['label'] ) ) :
						echo '<h3>' . esc_html( $field['label'] ) . '</h3>';
					endif;
					?>

					<input id="bbapp_field_<?php echo esc_attr( $field_name ); ?>" class="regular-text wp-color-picker <?php echo esc_attr( $field_name ); ?> colorpicker_field" name="<?php echo esc_attr( $field_name ); ?>" type="text" value="<?php echo esc_attr( $value ); ?>">

					<?php if ( isset( $field['label_checkbox'] ) ) { ?>
					<label for="bbapp_field_<?php echo esc_attr( $field_name ); ?>">
						<?php echo esc_html( $field['label_checkbox'] ); ?>
					</label>
						<?php
					}

					if ( isset( $field['desc'] ) ) {
						echo '<p class="description">' . wp_kses_post( $field['desc'] ) . '</p>';
					}

					if ( isset( $field['notice'] ) ) {
						echo '<div class="bbapp_notice"><p>' . wp_kses_post( $field['notice'] ) . '</p></div>';
					}

					echo '</div>';
					break;
			}
		}
	}

	/**
	 * Return uploaded file information.
	 *
	 * @param string $filename Field name.
	 *
	 * @return array|bool
	 */
	public function get_upload_file_info( $filename ) {
		$branding_options = BuddyBossAppBranding::instance()->get_options();
		$files_dir        = BuddyBossAppBranding::instance()->get_branding_upload_dir();
		$files_url        = BuddyBossAppBranding::instance()->get_branding_upload_url();
		$file_hash        = ( isset( $branding_options['uploads_hash'][ $filename ] ) ) ? $branding_options['uploads_hash'][ $filename ] : '';
		$value            = ( file_exists( $files_dir . '/' . $filename ) ) ? $files_url . '/' . $filename . '?' . $file_hash : '';

		if ( empty( $value ) ) {
			return false;
		} else {
			return array(
				'filepath' => $files_dir . '/' . $filename,
				'fileurl'  => $files_url . '/' . $filename . '?' . $file_hash,
			);
		}
	}

	/**
	 * Renders the branding singular field.
	 *
	 * @param array $field Field data.
	 * @param array $values Field values.
	 */
	public function render_field( $field, $values ) {
		$field_name  = $field['name'];
		$field_value = ( isset( $values[ $field_name ] ) ) ? $values[ $field_name ] : $field['value'];
		$class       = array( 'bbapp-field-item' );
		$input_link  = md5( $field_name . 'link' );
		$class[]     = $input_link;
		$class       = 'class="' . esc_attr( implode( ' ', $class ) ) . '"';

		if ( 'select' === $field['type'] ) {
			$options = (array) $field['options'];

			echo sprintf( '<select name="styles[%1$s]" data-link="%2$s">', esc_attr( $field_name ), esc_attr( $input_link ) );

			foreach ( $options as $option ) {
				$option = (array) $option;
				echo sprintf( '<option value="%1$s" %2$s %3$s>%4$s</option>', esc_attr( $option['value'] ), wp_kses_post( $class ), wp_kses_post( selected( $field_value, $option['value'], false ) ), esc_html( $option['label'] ) );
			}
			echo '</select>';
		}

		if ( 'color' === $field['type'] ) {
			echo sprintf( '<input name="styles[%1$s]" type="text" value="%2$s" data-link="%3$s" %4$s />', esc_attr( $field_name ), esc_attr( $field_value ), esc_attr( $input_link ), wp_kses_post( $class ) );
		}
	}

	/**
	 * Save branding settings.
	 */
	public function process_submit() {
		$nonce_post = ( ! empty( $_POST['bbapp_save_branding_options'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_save_branding_options'] ) ) : '';

		if ( wp_verify_nonce( $nonce_post, 'bbapp_save_branding_options' ) ) {
			if ( ! Permissions::instance()->can_manage_app() ) {
				wp_send_json_error( "You don't have permission to access this page." );
			}

			// Save Uploads.
			$upload_fields    = BuddyBossAppBranding::instance()->get_app_branding_fields();
			$branding_options = BuddyBossAppBranding::instance()->get_options();

			foreach ( $upload_fields as $field_name => $field ) {
				// Permission to upload super admin fields.
				if ( isset( $field['super_admin'] ) && $field['super_admin'] && ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
					continue;
				}

				switch ( $field['type'] ) {
					case 'upload':
						$out_location = BuddyBossAppBranding::instance()->get_branding_upload_dir();

						// If it's a delete request than delete it.
						if ( isset( $_POST[ $field_name . '_delete' ] ) && '1' === sanitize_text_field( $_POST[ $field_name . '_delete' ] ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							$file_path = '';

							if ( 'png' === $field['format'] ) {
								$file_path = $out_location . '/' . $field_name . '.png';
							}

							if ( 'jpg' === $field['format'] ) {
								$file_path = $out_location . '/' . $field_name . '.jpg';
							}

							if ( file_exists( $file_path ) && is_file( $file_path ) ) {
								wp_delete_file( $file_path );
							}

							break; // avoid perform below steps.
						}

						if ( isset( $_FILES[ $field_name . '_raw' ] ) ) {
							$file_data = array_map( 'sanitize_text_field', $_FILES[ $field_name . '_raw' ] ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							$file_data = BBAPP_File::read_file( $file_data['tmp_name'] );

							if ( empty( $file_data ) ) {
								break; // avoid perform below steps.
							}

							$file_hash = time();
							$im        = imagecreatefromstring( $file_data );

							if ( ! $im ) {
								break; // avoid perform below steps.
							}

							$width  = imagesx( $im );
							$height = imagesy( $im );

							if ( $field['fixedratio'] ) {
								if ( $width !== (int) $field['width'] || $height !== (int) $field['height'] ) {
									break; // avoid perform below steps.
								}
							}

							$filename = $field_name . '.' . $field['format'];
							// When exact size is provided don't manipulate images.
							$file_tmp_name = ! empty( $_FILES[ $field_name . '_raw' ]['tmp_name'] ) ? sanitize_text_field( wp_unslash( $_FILES[ $field_name . '_raw' ]['tmp_name'] ) ) : '';

							if ( $width === (int) $field['width'] && $height === (int) $field['height'] && IMAGETYPE_PNG === exif_imagetype( $file_tmp_name ) && 'png' === $field['format'] ) {
									BBAPP_File::write_file( $out_location . '/' . $field_name . '.png', $file_data );
							} elseif ( $width === (int) $field['width'] && $height === (int) $field['height'] && IMAGETYPE_JPEG === exif_imagetype( $file_tmp_name ) && 'jpg' === $field['format'] ) {
									BBAPP_File::write_file( $out_location . '/' . $field_name . '.jpg', $file_data );
							} else {
								// Save the image.
								if ( 'png' === $field['format'] ) {
									$new = imagecreatetruecolor( $width, $height );

									imagecolortransparent( $new, imagecolorallocatealpha( $new, 0, 0, 0, 127 ) );
									imagealphablending( $new, false );
									imagesavealpha( $new, true );
									imagecopyresampled( $new, $im, 0, 0, 0, 0, $width, $height, $width, $height );

									$file_path = $out_location . '/' . $field_name . '.png';

									if ( file_exists( $file_path ) ) {
										wp_delete_file( $file_path );
									}

									imagepng( $new, $file_path );
								}

								if ( 'jpg' === $field['format'] ) {
									// incase image source is png and background is transparent turn it into white.
									$new   = imagecreatetruecolor( $width, $height );
									$white = imagecolorallocate( $new, 255, 255, 255 );

									imagefilledrectangle( $new, 0, 0, $width, $height, $white );
									imagecopy( $new, $im, 0, 0, 0, 0, $width, $height );

									$file_path = $out_location . '/' . $field_name . '.jpg';

									if ( file_exists( $file_path ) ) {
										wp_delete_file( $file_path );
									}

									imagejpeg( $new, $file_path );
								}

								imagedestroy( $im );
							}
							$branding_options['uploads_hash'][ $filename ] = $file_hash;
						}
						break;
					case 'checkbox':
					case 'colorpicker':
						$branding_options[ $field_name ] = ( isset( $_POST[ $field_name ] ) ) ? sanitize_text_field( wp_unslash( $_POST[ $field_name ] ) ) : false;
						break;
				}
			}

			$styles = ( ! empty( $_POST['styles'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['styles'] ) ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			// Sanitize Styles.
			if ( ! empty( $styles ) ) {
				foreach ( (array) $styles as $k => $v ) {
					if ( ! is_string( $styles[ $k ] ) ) {
						continue;
					}

					$styles[ $k ] = sanitize_text_field( $v );
				}
			}

			$branding_options['styles'] = ! empty( $styles ) ? $styles : array();

			BuddyBossAppBranding::instance()->set_options( $branding_options );

			wp_send_json_success( __( 'Changes Saved', 'buddyboss-app' ) );
		} else {
			wp_send_json_error( __( 'Security Token Error', 'buddyboss-app' ) );
		}
	}
}
