<?php
/**
 * Holds all the branding class load.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

/**
 * Appearance class for branding.
 */
class Appearance {

	/**
	 * Class instance,
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Appearance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Appearance constructor.
	 */
	public function __construct() {
	}

	/**
	 * Load method.
	 *
	 * @return void
	 */
	public function load() {
		Appearance\Branding::instance();
		Appearance\Styling::instance();
		Appearance\Typography::instance();
		Appearance\Preview::instance();

		add_action( 'admin_init', array( $this, 'load_init' ) );
	}

	/**
	 * Init method.
	 *
	 * @return void
	 */
	public function load_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-appearance' === $page_get ) {
			if ( 'branding' === $this->get_current_sub_tab() ) {
				Appearance\Branding::instance()->will_render( true ); // tell the class that it will render.
			}

			if ( 'styling' === $this->get_current_sub_tab() ) {
				Appearance\Styling::instance()->will_render( true ); // tell the class that it will render.
			}

			if ( 'typography' === $this->get_current_sub_tab() ) {
				Appearance\Typography::instance()->will_render( true ); // tell the class that it will render.
			}
		}
	}

	/**
	 * Saved branding platform preview.
	 *
	 * @param string $platform Olatform name.
	 */
	public function save_branding_platform_preview( $platform = 'ios' ) {
		$platform       = isset( $_GET['platform'] ) ? bbapp_input_clean( wp_unslash( $_GET['platform'] ) ) : $platform; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$saved_platform = get_option( 'bbapp_branding_platform_preview' );

		if ( empty( $saved_platform ) || ( ! empty( $platform ) && $saved_platform !== $platform ) ) {
			update_option( 'bbapp_branding_platform_preview', $platform );
		}
	}

	/**
	 * Get the branding platform preview.
	 *
	 * @return false|mixed|string|void
	 */
	public function get_branding_platform_preview() {
		$bbapp_branding_platform_preview = get_option( 'bbapp_branding_platform_preview' );
		$bbapp_branding_platform_preview = empty( $bbapp_branding_platform_preview ) ? 'ios' : $bbapp_branding_platform_preview;

		return isset( $_GET['platform'] ) ? bbapp_input_clean( wp_unslash( $_GET['platform'] ) ) : $bbapp_branding_platform_preview; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}

	/**
	 * Function to localize JS variables.
	 *
	 * @return void
	 */
	public function footer() {
		wp_localize_script(
			'bbapp-script',
			'iconsPicker',
			array(
				'bbapp_icons_path'       => \BuddyBossApp\Common\IconPicker::instance()->icon_picker_app_icon_url(),
				'bbapp_custom_icons_url' => \BuddyBossApp\Common\IconPicker::instance()->icon_picker_custom_icon_url(),
			)
		);
	}

	/**
	 * Saves the settings
	 *
	 * @param array $settings Array of settings.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {

		$app_settings = \BuddyBossApp\ManageApp::instance()->get_app_settings();

		\BuddyBossApp\ManageApp::instance()->update_app_settings( $app_settings );

		return true;
	}

	/**
	 * Gets the settings
	 */
	public function get_settings() {

		$settings = \BuddyBossApp\ManageApp::instance()->get_app_settings();

		return $settings;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {

		$return = array(
			'branding'   => __( 'Images', 'buddyboss-app' ),
			'styling'    => __( 'Colors', 'buddyboss-app' ),
			'typography' => __( 'Typography', 'buddyboss-app' ),
		);

		// Hide the Tab Bar(app_menu) from super admin dashboard(so called "Network Admin Area").
		if ( ! bbapp_is_admin_page() && bbapp_is_super_admin_page() ) {
			unset( $return['styling'] );
		}

		return $return;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $page_get ) || 'bbapp-build' === $page_get ) {
			return null;
		}

		$default      = 'branding';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {

		add_action( 'admin_footer', array( $this, 'footer' ), 50 );

		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( 'branding' === $this->get_current_sub_tab() ) {
			Appearance\Branding::instance()->render();
		}

		if ( 'styling' === $this->get_current_sub_tab() ) {
			Appearance\Styling::instance()->render();
		}

		if ( 'typography' === $this->get_current_sub_tab() ) {
			Appearance\Typography::instance()->render();
		}
	}
}
