<?php
/**
 * Holds App languages functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\AppLanguages as BuddyBossAppLanguages;
use BuddyBossApp\Helpers\BBAPP_File;
use Exception;

/**
 * App languages class.
 */
class AppLanguages {

	/**
	 * Class instance.
	 *
	 * @var AppLanguages $instance
	 */
	private static $instance;

	/**
	 * Notice messages.
	 *
	 * @var array $messages
	 */
	private $messages = array();

	/**
	 * Get the instance of the class.
	 *
	 * @return AppLanguages
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * AppLanguages constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'wp_ajax_bbapp_update_languages', array( $this, 'ajax_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_update_languages', array( $this, 'ajax_submit' ) );
		add_action( 'wp_ajax_bbapp_get_translations', array( $this, 'ajax_get_translations' ) );
		add_action( 'wp_ajax_bbapp_switch_language', array( $this, 'ajax_switch_language' ) );
		add_action( 'wp_ajax_bbapp_add_language', array( $this, 'ajax_add_language' ) );
		add_action( 'wp_ajax_bbapp_delete_language', array( $this, 'ajax_delete_language' ) );
		add_action( 'wp_ajax_bbapp_export_translations', array( $this, 'ajax_export_translations' ) );
		add_action( 'wp_ajax_bbapp_import_translations', array( $this, 'ajax_import_translations' ) );

		// Register CRON job for cleaning up orphaned translations.
		add_action( 'bbapp_every_week', array( $this, 'handle_cleanup_cron' ) );
	}

	/**
	 * Renders the branding screen
	 *
	 * @since 1.0.0
	 *
	 * @return bool|mixed|void
	 */
	public function render() {
		if ( ! current_user_can( 'manage_options' ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		$this->show_messages();

		include bbapp()->plugin_dir . 'views/settings/app_languages.php';
	}

	/**
	 * Output the error or update messages to render.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), wp_kses_post( $message['message'] ) );
			}
		}
	}

	/**
	 * Ajax Submit for App Languages Save Changes.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function ajax_submit() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_languages_nonce', 'bbapp_languages_nonce', false ) ) {
			wp_send_json_error( __( 'Invalid nonce', 'buddyboss-app' ) );

			return;
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( "You don't have permission", 'buddyboss-app' ) );

			return;
		}

		// Get submitted translations.
		$languages_post_data = isset( $_POST['lang_data'] ) ? sanitize_text_field( wp_unslash( $_POST['lang_data'] ) ) : '{}';
		$languages_post_data = json_decode( $languages_post_data, true );

		if ( ! is_array( $languages_post_data ) ) {
			$languages_post_data = array();
		}

		// Process CSV data if submitted with form.
		if ( ! empty( $_POST['lang_data_csv'] ) ) {
			$csv_translations = json_decode( wp_unslash( $_POST['lang_data_csv'] ), true ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$lang_name_csv    = ! empty( $_POST['lang_name_csv'] ) ? sanitize_text_field( wp_unslash( $_POST['lang_name_csv'] ) ) : 'en';

			// If valid CSV data found.
			if ( is_array( $csv_translations ) && ! empty( $csv_translations ) ) {
				// Create or update the language array.
				if ( ! isset( $languages_post_data[ $lang_name_csv ] ) ) {
					$languages_post_data[ $lang_name_csv ] = array();
				}

				// Add each CSV translation to the languages data.
				foreach ( $csv_translations as $translation ) {
					if ( isset( $translation['string_handle'] ) && isset( $translation['updated_string'] ) ) {
						$languages_post_data[ $lang_name_csv ][ $translation['string_handle'] ] = $translation['updated_string'];
					}
				}
			}
		}

		global $wpdb;
		$updated_count = 0;

		// Begin transaction for database operations.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		try {
			// Process each language.
			foreach ( $languages_post_data as $language_code => $translations ) {
				if ( ! is_array( $translations ) ) {
					continue;
				}

				// Process each translation.
				foreach ( $translations as $string_handle => $translation_text ) {
					// Update the translation in the database.
					$result = $this->update_string_translation(
						$language_code,
						$string_handle,
						$translation_text
					);

					if ( $result ) {
						$updated_count ++; // phpcs:ignore
					}
				}
			}

			// Commit the transaction if everything went well.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

			// translators: placeholder: %d number of strings updated.
			wp_send_json_success( sprintf( __( 'Translations saved successfully. %d strings updated.', 'buddyboss-app' ), $updated_count ) );
		} catch ( Exception $e ) {
			// Rollback the transaction if there was an error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
			wp_send_json_error( __( 'Error saving translations: ', 'buddyboss-app' ) . $e->getMessage() );
		}
	}

	/**
	 * AJAX handler for getting translations with pagination and search
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_get_translations() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_languages_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Invalid nonce', 'buddyboss-app' ) );

			return;
		}

		$page          = isset( $_POST['page'] ) ? (int) $_POST['page'] : 1;
		$per_page      = isset( $_POST['per_page'] ) ? (int) $_POST['per_page'] : BuddyBossAppLanguages::instance()->per_page;
		$search        = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';
		$search_by     = isset( $_POST['search_by'] ) ? sanitize_text_field( wp_unslash( $_POST['search_by'] ) ) : 'lang-source';
		$language_code = isset( $_POST['language_code'] ) ? sanitize_text_field( wp_unslash( $_POST['language_code'] ) ) : bbapp_wp_locale_to_app_locale();
		$show_empty    = isset( $_POST['show_empty'] ) && sanitize_text_field( wp_unslash( $_POST['show_empty'] ) ) === 'true';

		// Convert API language code to database format for querying.
		$db_language_code = BuddyBossAppLanguages::instance()->get_db_language_code( $language_code );

		// Parse the search_by parameter into a column name.
		if ( ! empty( $search ) ) {
			switch ( $search_by ) {
				case 'lang-id':
					$search_column = 'string_handle';
					break;
				case 'lang-translate':
					$search_column = 'updated_string';
					break;
				default:
					$search_column = 'default_string';
					break;
			}
		}

		// Get total count first.
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		$where = $wpdb->prepare( 'WHERE language_code = %s', $db_language_code ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		if ( ! empty( $search ) ) {
			$where .= $wpdb->prepare(
				" AND {$search_column} LIKE %s", // phpcs:ignore
				'%' . $wpdb->esc_like( $search ) . '%'
			); // phpcs:ignore
		}

		if ( true === $show_empty ) {
			$where .= " AND (updated_string IS NULL OR updated_string = '')";
		}

		$total_items = $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name} {$where}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		// Get paginated results.
		$offset       = ( $page - 1 ) * $per_page;
		$translations = $wpdb->get_results( "SELECT * FROM {$table_name} {$where} ORDER BY string_handle LIMIT {$per_page} OFFSET {$offset}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		$total_pages = ceil( $total_items / $per_page );

		$response = array(
			'success' => true,
			'data'    => array(
				'translations' => $translations,
				'pagination'   => array(
					'current_page' => $page,
					'per_page'     => $per_page,
					'total_items'  => $total_items,
					'total_pages'  => $total_pages,
				),
			),
		);

		wp_send_json( $response );
	}

	/**
	 * AJAX handler for switching language
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_switch_language() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_languages_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Invalid nonce', 'buddyboss-app' ) );

			return;
		}

		$language_code = isset( $_POST['language_code'] ) ? sanitize_text_field( wp_unslash( $_POST['language_code'] ) ) : '';

		if ( empty( $language_code ) ) {
			wp_send_json_error( array( 'message' => 'Language code is required' ) );
		}

		// Save the selected language.
		update_option( 'bbapp_selected_language', $language_code );

		$page       = isset( $_POST['page'] ) ? (int) $_POST['page'] : 1;
		$per_page   = isset( $_POST['per_page'] ) ? (int) $_POST['per_page'] : BuddyBossAppLanguages::instance()->per_page;
		$search     = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';
		$search_by  = isset( $_POST['search_by'] ) ? sanitize_text_field( wp_unslash( $_POST['search_by'] ) ) : 'lang-source';
		$show_empty = isset( $_POST['show_empty'] ) && sanitize_text_field( wp_unslash( $_POST['show_empty'] ) ) === 'true';

		// Convert API language code to database format for querying.
		$db_language_code = BuddyBossAppLanguages::instance()->get_db_language_code( $language_code );

		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Build the WHERE clause.
		$where = $wpdb->prepare( 'WHERE language_code = %s', $db_language_code ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		if ( ! empty( $search ) ) {
			switch ( $search_by ) {
				case 'lang-source':
					$where .= $wpdb->prepare(  // phpcs:ignore
						' AND default_string LIKE %s',
						'%' . $wpdb->esc_like( $search ) . '%' // phpcs:ignore
					);
					break;
				case 'lang-id':
					$where .= $wpdb->prepare(  // phpcs:ignore
						' AND string_handle LIKE %s',
						'%' . $wpdb->esc_like( $search ) . '%' // phpcs:ignore
					);
					break;
				case 'lang-translate':
					$where .= $wpdb->prepare(  // phpcs:ignore
						' AND updated_string LIKE %s',
						'%' . $wpdb->esc_like( $search ) . '%'  // phpcs:ignore
					);
					break;
			}
		}

		// Handle the "Show Empty" checkbox.
		if ( true === $show_empty ) {
			// When "Show Empty" IS checked, show only empty translations.
			$where .= " AND (updated_string IS NULL OR updated_string = '')";
		}

		// Get total count.
		$total_items = $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name} {$where}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		// Get paginated results.
		$offset       = ( $page - 1 ) * $per_page;
		$translations = $wpdb->get_results( "SELECT * FROM {$table_name} {$where} ORDER BY string_handle LIMIT {$per_page} OFFSET {$offset}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		$total_pages   = ceil( $total_items / $per_page );
		$language_name = BuddyBossAppLanguages::instance()->get_language_name( $language_code );
		$response      = array(
			'success' => true,
			'data'    => array(
				// translators: %s is the language code.
				'message'      => sprintf( __( 'Successfully switched to %s', 'buddyboss-app' ), $language_name ),
				'translations' => $translations,
				'pagination'   => array(
					'current_page' => $page,
					'per_page'     => $per_page,
					'total_items'  => $total_items,
					'total_pages'  => $total_pages,
				),
			),
		);

		wp_send_json( $response );
	}

	/**
	 * AJAX handler for adding a new language
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_add_language() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_add_language_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'buddyboss-app' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action', 'buddyboss-app' ) ) );
		}

		// Get language code.
		$language = isset( $_POST['language'] ) ? sanitize_text_field( wp_unslash( $_POST['language'] ) ) : '';
		if ( empty( $language ) ) {
			wp_send_json_error( array( 'message' => __( 'Language code is required', 'buddyboss-app' ) ) );
		}

		// Check if language is supported.
		$app_supported_languages = BuddyBossAppLanguages::instance()->get_app_supported_languages();

		if ( ! isset( $app_supported_languages[ $language ] ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid language code', 'buddyboss-app' ) ) );
		}

		// Get current active languages.
		$active_languages = get_option( 'bbapp_active_languages', array() );
		if ( ! is_array( $active_languages ) ) {
			$active_languages = array();
		}

		// Check if language is already active.
		if ( in_array( $language, $active_languages, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Language is already active', 'buddyboss-app' ) ) );
		}

		// Add new language.
		$active_languages[] = $language;

		// Update option.
		update_option( 'bbapp_active_languages', array_unique( $active_languages ) );

		// Set this as the selected language.
		update_option( 'bbapp_selected_language', $language );

		wp_send_json_success();
	}

	/**
	 * AJAX handler for deleting a language
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_delete_language() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_delete_language_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'buddyboss-app' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action', 'buddyboss-app' ) ) );
		}

		// Get language code.
		$language = isset( $_POST['language'] ) ? sanitize_text_field( wp_unslash( $_POST['language'] ) ) : '';
		if ( empty( $language ) ) {
			wp_send_json_error( array( 'message' => __( 'Language code is required', 'buddyboss-app' ) ) );
		}

		// Get WordPress default language.
		$wp_locale = bbapp_wp_locale_to_app_locale();

		// Get current active languages.
		$active_languages = get_option( 'bbapp_active_languages', array() );
		if ( ! is_array( $active_languages ) ) {
			$active_languages = array();
		}

		// Always ensure English is in the active languages.
		if ( ! in_array( $wp_locale, $active_languages, true ) ) {
			array_unshift( $active_languages, $wp_locale );
		}

		// Check if language exists in active languages.
		if ( ! in_array( $language, $active_languages, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Language is not active', 'buddyboss-app' ) ) );
		}

		// Check if it's the WordPress default language.
		if ( $language === $wp_locale ) {
			// translators: %s is the WordPress default language.
			wp_send_json_error( array( 'message' => sprintf( __( 'Cannot delete the WordPress default language (%s)', 'buddyboss-app' ), $wp_locale ) ) );
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Begin transaction.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		try {
			// Reset all updated_string values for the language.
			$wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$table_name,
				array(
					'updated_string' => '',
					'updated_at'     => null,
				),
				array(
					'language_code' => $language,
				)
			);

			// Remove language from active languages.
			$active_languages = array_diff( $active_languages, array( $language ) );

			// If we're deleting English and it's not the WordPress default, we need to ensure another language is set as default.
			if ( $language === $wp_locale ) {
				// If there are no other languages, add English back as it's required.
				if ( empty( $active_languages ) ) {
					$active_languages[] = $wp_locale;
				}
			}

			update_option( 'bbapp_active_languages', $active_languages );

			// Check if the deleted language is the currently selected language.
			$selected_language = get_option( 'bbapp_selected_language', $wp_locale );
			if ( $selected_language === $language ) {
				// Update the selected language to WordPress default.
				update_option( 'bbapp_selected_language', $wp_locale );
			}

			// Commit transaction.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

			wp_send_json_success( array( 'message' => __( 'Language deleted successfully', 'buddyboss-app' ) ) );
		} catch ( \Exception $e ) {
			// Rollback on error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
			wp_send_json_error( array( 'message' => __( 'Error deleting language', 'buddyboss-app' ) ) );
		}
	}

	/**
	 * Cleanup orphaned translations from the database
	 * Removes translations that have been marked as deleted for more than one year
	 *
	 * @since 2.4.10
	 *
	 * @return int Number of rows deleted
	 */
	public function cleanup_orphaned_translations() {
		global $wpdb;
		$table_name    = $wpdb->prefix . 'bbapp_string_translations';
		$app_languages = \BuddyBossApp\AppLanguages::instance();

		// Calculate date one year ago.
		$one_year_ago = gmdate( 'Y-m-d H:i:s', strtotime( '-1 year' ) );

		// Find orphaned translations (deleted more than one year ago).
		$orphaned_translations = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT id FROM {$table_name}  WHERE deleted_at IS NOT NULL AND deleted_at < %s",
				$one_year_ago
			)
		); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		// If no orphaned translations found, return 0.
		if ( empty( $orphaned_translations ) ) {
			return 0;
		}

		// Delete them permanently using our batch delete function.
		$result = $app_languages->batch_delete_translations( $orphaned_translations, true );

		// Log the cleanup for monitoring purposes.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// translators: %d is the number of orphaned translations cleaned up.
			error_log( sprintf( 'BuddyBoss App: Cleaned up %d orphaned translations', $result['success'] ) );
		}

		return $result['success'];
	}

	/**
	 * Handle the CRON event for cleaning up orphaned translations
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function handle_cleanup_cron() {
		$this->cleanup_orphaned_translations();
	}

	/**
	 * AJAX handler for exporting translations to CSV
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_export_translations() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_languages_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Invalid nonce', 'buddyboss-app' ) );

			return;
		}

		$language_code = isset( $_POST['language_code'] ) ? sanitize_text_field( wp_unslash( $_POST['language_code'] ) ) : '';
		if ( empty( $language_code ) ) {
			wp_send_json_error( array( 'message' => __( 'Language code is required', 'buddyboss-app' ) ) );
		}

		// Convert API language code to database format for querying and exporting.
		$db_language_code = \BuddyBossApp\AppLanguages::instance()->get_db_language_code( $language_code );

		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Get all translations for the selected language.
		$translations = $wpdb->get_results( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT string_handle, default_string, updated_string FROM {$table_name} WHERE language_code = %s ORDER BY string_handle",
				$db_language_code
			)
		); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		if ( empty( $translations ) ) {
			wp_send_json_error( array( 'message' => __( 'No translations found for this language', 'buddyboss-app' ) ) );
		}

		// Create CSV content with header row only (no metadata).
		$csv_content = "String ID,Source Text,Translation\n";

		foreach ( $translations as $translation ) {
			$string_handle  = $this->escape_csv_value( $translation->string_handle );
			$default_string = $this->escape_csv_value( $translation->default_string );
			$updated_string = $this->escape_csv_value( $translation->updated_string );

			$csv_content .= "{$string_handle},{$default_string},{$updated_string}\n";
		}

		wp_send_json_success(
			array(
				'csv_content'   => $csv_content,
				'language_code' => $language_code,
			)
		);
	}

	/**
	 * Escapes a value for a CSV file
	 *
	 * @param string $value The value to escape.
	 *
	 * @since 2.4.10
	 *
	 * @return string The escaped value
	 */
	private function escape_csv_value( $value ) {
		// Replace double quotes with two double quotes.
		$value = str_replace( '"', '""', $value );

		// If the value contains commas, newlines, or double quotes, enclose it in double quotes.
		if ( false !== strpos( $value, ',' ) || false !== strpos( $value, '"' ) || false !== strpos( $value, "\n" ) ) {
			$value = '"' . $value . '"';
		}

		return $value;
	}

	/**
	 * Handle AJAX import of translations from CSV
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function ajax_import_translations() {
		// Check nonce.
		if ( ! check_ajax_referer( 'bbapp_languages_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Invalid nonce', 'buddyboss-app' ) );

			return;
		}

		// Check capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buddyboss-app' ) ) );
		}

		// Get the uploaded file.
		if ( ! isset( $_FILES['translations_file'] ) || ( ! empty( $_FILES['translations_file']['error'] ) && UPLOAD_ERR_OK !== $_FILES['translations_file']['error'] ) ) {
			wp_send_json_error( array( 'message' => __( 'No file uploaded or upload error.', 'buddyboss-app' ) ) );
		}

		// Validate file type (extension check).
		$file_info = wp_check_filetype( $_FILES['translations_file']['name'] ); // phpcs:ignore

		if ( 'csv' !== $file_info['ext'] ) {
			wp_send_json_error( array( 'message' => __( 'Invalid file type. Please upload a CSV..', 'buddyboss-app' ) ) );
		}

		// Get selected language code.
		$selected_language_code = isset( $_POST['language_code'] ) ? sanitize_text_field( wp_unslash( $_POST['language_code'] ) ) : '';
		if ( empty( $selected_language_code ) ) {
			wp_send_json_error( array( 'message' => __( 'No language selected.', 'buddyboss-app' ) ) );
		}

		// Get language code from filename if already extracted in JavaScript.
		$filename_language = isset( $_POST['filename_language'] ) ? sanitize_text_field( wp_unslash( $_POST['filename_language'] ) ) : '';

		$buddyboss_app_languages = BuddyBossAppLanguages::instance();

		// check the supported filename language.
		$supported_filename_language = array_keys( $buddyboss_app_languages->get_app_supported_languages() );
		if ( ! in_array( $filename_language, $supported_filename_language, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid file name language.', 'buddyboss-app' ) ) );
		}

		// Get active languages.
		$active_languages = $buddyboss_app_languages->get_active_languages();

		// Check if the file is a valid CSV.
		if ( empty( $_FILES['translations_file']['tmp_name'] ) ) {
			wp_send_json_error( array( 'message' => __( 'No file uploaded.', 'buddyboss-app' ) ) );
		}

		// Try to detect language code from filename if not already provided by JS.
		$imported_language_code = $filename_language;
		if ( empty( $imported_language_code ) ) {
			$filename = sanitize_text_field( wp_unslash( $_FILES['translations_file']['name'] ) ); // phpcs:ignore

			// Our export format uses language code at beginning of filename (e.g. de_DE-translations.csv).
			if ( preg_match( '/^([a-zA-Z]{2}[_-][a-zA-Z]{2})/', $filename, $matches ) ) {
				$imported_language_code = $matches[1];
			} elseif ( preg_match( '/[-_]([a-zA-Z]{2}[_-][a-zA-Z]{2})[-_\.]/', $filename, $matches ) ) { // Fallback to extract from middle of filename (e.g. translations-de_DE.csv).
				$imported_language_code = $matches[1];
			} elseif ( preg_match( '/[-_]([a-zA-Z]{2}[_-][a-zA-Z]{2})$/', $filename, $matches ) ) { // Try end of filename without extension.
				$imported_language_code = $matches[1];
			} elseif ( preg_match( '/^([a-zA-Z]{2}[_-][a-zA-Z]{2})\.csv$/', $filename, $matches ) ) { // Try filename with just language code.
				$imported_language_code = $matches[1];
			}
		}

		// Load WP_Filesystem.
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		global $wp_filesystem;
		WP_Filesystem();

		$csv_path = $_FILES['translations_file']['tmp_name']; // phpcs:ignore

		if ( ! $wp_filesystem->exists( $csv_path ) ) {
			wp_send_json_error( __( 'Uploaded file not found.', 'buddyboss-app' ) );
		}

		$csv_contents = $wp_filesystem->get_contents( $csv_path );

		if ( false === $csv_contents ) {
			wp_send_json_error( __( 'Could not read the uploaded CSV file.', 'buddyboss-app' ) );
		}

		$csv_data  = array();
		$csv_lines = explode( "\n", $csv_contents );
		$header    = null;

		foreach ( $csv_lines as $line ) {
			if ( empty( trim( $line ) ) ) {
				continue;
			}

			$row = str_getcsv( $line );

			if ( null === $header ) {
				$header = $row;
				continue;
			}

			if ( count( $header ) === count( $row ) ) {
				$sanitized_row = array_map( 'sanitize_text_field', $row );
				$csv_data[]    = array_combine( $header, $sanitized_row );
			}
		}

		// Check if the detected language is in the available languages list.
		$app_supported_languages = $buddyboss_app_languages->get_app_supported_languages();

		$detected_language_supported = false;

		if ( ! empty( $imported_language_code ) ) {
			// Normalize language code for comparison.
			$normalized_imported = strtolower( str_replace( '-', '_', $imported_language_code ) );
			foreach ( $app_supported_languages as $code => $name ) {
				$normalized_code = strtolower( str_replace( '-', '_', $code ) );
				if ( $normalized_code === $normalized_imported ) {
					$detected_language_supported = true;
					$imported_language_code      = $code; // Use the properly formatted code.
					break;
				}
			}
		}

		// Check if a supported language was detected but not active.
		if ( $detected_language_supported && ! in_array( $imported_language_code, array_keys( $active_languages ), true ) ) {
			wp_send_json_error(
				array(
					'message'                => sprintf(
					// translators: %1$s is the detected language code.
						__( 'The CSV file appears to contain translations for "%1$s" which is not in your active languages list.', 'buddyboss-app' ),
						$buddyboss_app_languages->get_language_name( $imported_language_code )
					),
					'detected_language'      => $imported_language_code,
					'detected_language_name' => $buddyboss_app_languages->get_language_name( $imported_language_code ),
					'code'                   => 'language_not_exists',
				)
			);
		}

		// Language mismatch warning - normalized comparison for better UX.
		if ( ! empty( $imported_language_code ) && $selected_language_code !== $imported_language_code ) {
			// Normalize both language codes for comparison.
			$normalized_imported = strtolower( str_replace( '-', '_', $imported_language_code ) );
			$normalized_selected = strtolower( str_replace( '-', '_', $selected_language_code ) );

			// If they don't match after normalization.
			if ( $normalized_imported !== $normalized_selected ) {
				// Show warning and include the detected language in the response.
				wp_send_json_error(
					array(
						'message'                => sprintf(
						// translators: %1$s is the detected language code, %2$s is the selected language code.
							__(
								'The CSV filename suggests language "%1$s" but you have "%2$s" selected. Please select the correct language or upload the correct file.',
								'buddyboss-app'
							),
							$buddyboss_app_languages->get_language_name( $imported_language_code ),
							$buddyboss_app_languages->get_language_name( $selected_language_code )
						),
						'detected_language'      => $imported_language_code,
						'detected_language_name' => $buddyboss_app_languages->get_language_name( $imported_language_code ),
						'code'                   => 'language_mismatch',
					)
				);
			}
		}

		// Check if the selected language exists in active languages.
		if ( ! in_array( $selected_language_code, array_keys( $active_languages ), true ) ) {
			wp_send_json_error(
				array(
					'message' => sprintf(
					// translators: %s is the selected language code.
						__( 'The selected language "%s" is not in your active languages list. Please add it first.', 'buddyboss-app' ),
						$selected_language_code
					),
					'code'    => 'language_not_exists',
				)
			);
		}

		// Reset array keys.
		$csv_array = $csv_data;

		// Validate CSV structure - should have at least 3 columns.
		if ( empty( $csv_array ) || ! isset( $csv_array[0] ) || count( $csv_array[0] ) < 3 ) {
			wp_send_json_error( array( 'message' => __( 'Invalid CSV format. Please ensure it has at least 3 columns.', 'buddyboss-app' ) ) );
		}

		$translations = array();
		$count        = 0;

		foreach ( $csv_array as $row ) {
			// Skip empty rows.
			if ( empty( $row ) ) {
				continue;
			}

			// Get values using associative array keys.
			$string_handle     = isset( $row['String ID'] ) ? sanitize_text_field( trim( $row['String ID'] ) ) : '';
			$default_string    = isset( $row['Source Text'] ) ? sanitize_text_field( trim( $row['Source Text'] ) ) : '';
			$translated_string = isset( $row['Translation'] ) ? wp_kses_post( trim( $row['Translation'] ) ) : '';

			// Verify we have a valid row with at least string_handle.
			if ( ! empty( $string_handle ) && ! empty( $translated_string ) ) {
				$translations[] = array(
					'string_handle'  => $string_handle,
					'default_string' => $default_string,
					'updated_string' => $translated_string,
				);
				++$count;
			}
		}

		if ( 0 === $count ) {
			wp_send_json_error( array( 'message' => __( 'No valid translations found in the CSV file.', 'buddyboss-app' ) ) );
		}

		// Return the parsed translations to be displayed in the interface
		// The user will need to click Save to apply them.
		wp_send_json_success(
			array(
				'message'       => sprintf(
				// translators: %d is the number of translations loaded.
					__( '%d translations loaded from CSV. Click "Save Changes" to apply them.', 'buddyboss-app' ),
					$count
				),
				'translations'  => $translations,
				'language_code' => $selected_language_code,
			)
		);
	}

	/**
	 * Update a string translation or create it if it doesn't exist
	 *
	 * @param string $language_code The language code.
	 * @param string $string_handle The string handle/ID.
	 * @param string $new_value     The new translation value.
	 *
	 * @since 2.4.10
	 *
	 * @return bool|int False on failure, true on no change needed, or number of rows affected.
	 */
	public function update_string_translation( $language_code, $string_handle, $new_value ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Convert API language code to database format for querying and storing translations.
		$db_language_code = \BuddyBossApp\AppLanguages::instance()->get_db_language_code( $language_code );

		// Get existing translation.
		$existing = $wpdb->get_row( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
			$wpdb->prepare(
				"SELECT id, updated_string, default_string, is_build_string FROM {$table_name} WHERE language_code = %s AND string_handle = %s",
				$db_language_code,
				$string_handle
			)
		); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		// Determine if this is a build string based on the prefix.
		$is_build_string = ( 0 === strpos( $string_handle, 'bbAppBuildTime.' ) ) ? 1 : 0;

		if ( $existing ) {
			// Only update if value actually changed.
			if ( $existing->updated_string !== $new_value ) {
				$current_time = current_time( 'mysql' );
				$update_data  = array(
					'updated_string' => $new_value,
					'updated_at'     => $current_time,
				);

				// Update is_build_string if it doesn't match the expected value.
				if ( (int) $existing->is_build_string !== $is_build_string ) {
					$update_data['is_build_string'] = $is_build_string;
				}

				return $wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
					$table_name,
					$update_data,
					array(
						'id' => $existing->id,
					)
				);
			}

			return true; // No change needed.
		} else {
			// Get the default string from any language.
			$default_record = $wpdb->get_row( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$wpdb->prepare(
					"SELECT default_string, is_build_string 
					FROM {$table_name} 
					WHERE string_handle = %s 
					LIMIT 1",
					$string_handle
				)
			);

			$default_string = '';
			if ( $default_record ) {
				$default_string = $default_record->default_string;
				// Use existing is_build_string value if available.
				if ( null !== $default_record->is_build_string ) {
					$is_build_string = $default_record->is_build_string;
				}
			} else {
				// If we can't find the default string, use the new value as default.
				$default_string = $new_value;
			}

			// Insert new translation.
			$current_time = current_time( 'mysql' );
			$site_id      = get_current_blog_id();

			return $wpdb->insert( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$table_name,
				array(
					'site_id'          => $site_id,
					'string_handle'    => $string_handle,
					'default_string'   => $default_string,
					'language_code'    => $db_language_code,
					'updated_string'   => $new_value,
					'is_build_string'  => $is_build_string,
					'is_custom_string' => 1,
					'created_at'       => $current_time,
					'updated_at'       => $current_time,
				)
			);
		}
	}
}
