<?php
/**
 * Holds advanced features functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\Tools\Telemetry;
use BuddyBossApp\ManageApp;

/**
 * Advanced class.
 */
class Advanced {

	/**
	 * Class instance.
	 *
	 * @since 2.2.70
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Form variable.
	 *
	 * @since 2.2.70
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @since 2.2.70
	 *
	 * @return Advanced
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Render Setting Form.
	 *
	 * @since 2.2.70
	 */
	public function render_settings() {
		$this->settings_form->render_form();
	}

	/**
	 * Load Settings.
	 *
	 * @since 2.2.70
	 */
	public function load_settings() {
		$this->settings_form = new FormProvider( 'settings' );
		// holds app settings storage handle.
		$app_settings_provider = new SettingProvider( 'app_settings' );

		// Register app settings storing & receiving.
		$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
		$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

		// Link Provider In Form Provider.
		$this->settings_form->link_setting_provider( $app_settings_provider );

		add_action( 'admin_init', array( $this->settings_form, 'save' ) );

		$this->settings_form->add_field(
			$app_settings_provider->field(
				false,
				array(
					'type'    => 'table-open',
					'classes' => 'buddyboss-app-admin-card bbapp-admin-advanced',
				)
			)
		);

		$this->settings_form->add_field(
			$app_settings_provider->field(
				'app_advanced_heading',
				array(
					'type'  => 'heading',
					'value' => __( 'Telemetry', 'buddyboss-app' ),
					'icon'  => 'bb-icon-f bb-icon-box',
					'tutorial_label'  => __( 'About Telemetry', 'buddyboss-app' ),
					'tutorial_url' => 'https://www.buddyboss.com/usage-tracking/?utm_source=product&utm_medium=app-plugin&utm_campaign=telemetry',
				)
			)
		);

		$bbapp_db_telemetry_option = ManageApp::instance()->get_app_setting( 'app_telemetry' );
		$bbapp_db_telemetry_option = empty( $bbapp_db_telemetry_option ) ? 'anonymous' : $bbapp_db_telemetry_option;

		// Generate html for telemetry.
		// phpcs:ignore.
		$telemetry_reporting = isset( $_POST['app_telemetry'] ) ? sanitize_text_field( wp_unslash( $_POST['app_telemetry'] ) ) : $bbapp_db_telemetry_option;

		$telemetry_modes = array(
			'complete'  => array(
				'label'    => esc_html__( 'Complete reporting', 'buddyboss-app' ),
				'name'     => 'app_telemetry',
				'value'    => 'complete',
				'id'       => 'complete_reporting',
				'checked'  => 'complete' === $telemetry_reporting,
				'notice'   => sprintf(
					/* translators: %1$s and %2$s wrap the phrase "disable telemetry" in a clickable link. */
					esc_html__( 'Telemetry helps us gather usage statistics and information about your configuration and the features and functionality you use. We aggregate this information to help us improve our product and associate it with your customer record to help us serve you better. We do not gather or send any of your users\' personally identifiable information. To stop contributing towards improving the product, you can %1$sdisable telemetry%2$s.', 'buddyboss-app' ),
					'<a class="bbapp-disable-telemetry-link" href="#">',
					'</a>'
				),
				'disabled' => false,
			),
			'anonymous' => array(
				'label'    => esc_html__( 'Anonymous reporting', 'buddyboss-app' ),
				'name'     => 'app_telemetry',
				'value'    => 'anonymous',
				'id'       => 'anonymous_reporting',
				'checked'  => 'anonymous' === $telemetry_reporting,
				'notice'   => sprintf(
					/* translators: %1$s and %2$s wrap the phrase "disable telemetry" in a clickable link. */
					esc_html__( 'Telemetry helps us gather usage statistics and information about your configuration and the features and functionality you use. We aggregate this information to help us improve our product. By choosing anonymous reporting, your data will not be associated with your customer record, and the way we serve you will be less relevant to you. We do not gather or send any of your users\' personally identifiable information. If you stop contributing towards improving the product, you can %1$sdisable telemetry%2$s.', 'buddyboss-app' ),
					'<a class="bbapp-disable-telemetry-link" href="#">',
					'</a>'
				),
				'disabled' => false,
			),
		);
		ob_start();
		if ( ! empty( $telemetry_modes ) && is_array( $telemetry_modes ) ) {
			$notice_text = '';
			foreach ( $telemetry_modes as $telemetry_mode ) {
				?>
				<label for="<?php echo esc_attr( $telemetry_mode['id'] ); ?>" class="<?php echo esc_attr( ! empty( $telemetry_mode['disabled'] ) ? 'disabled' : '' ); ?>">
					<input name="<?php echo esc_attr( $telemetry_mode['name'] ); ?>"
						id="<?php echo esc_attr( $telemetry_mode['id'] ); ?>"
						type="radio"
						value="<?php echo esc_attr( $telemetry_mode['value'] ); ?>"
						data-current-val="<?php echo esc_attr( $telemetry_mode['value'] ); ?>"
						data-notice="<?php /* phpcs:ignore */ echo ! empty( $telemetry_mode['notice'] ) ? htmlspecialchars( $telemetry_mode['notice'], ENT_QUOTES, 'UTF-8' ) : ''; ?>"
						<?php
						checked( $telemetry_mode['checked'] );
						?>
					/>
					<?php echo esc_html( $telemetry_mode['label'] ); ?>
				</label>
				<?php
				if ( ! empty( $telemetry_mode['checked'] ) || ( 'disable' === $telemetry_reporting && 'complete' === $telemetry_mode['value'] ) ) {
					$notice_text = $telemetry_mode['notice'];
				}
			}

			if ( ! empty( $notice_text ) ) {
				?>
				<p class="description bbapp-telemetry-mode-description">
					<?php echo wp_kses_post( $notice_text ); ?>
				</p>
				<?php
			}
		}
		?>
		<div class='bbapp-setting-telemetry-no-reporting <?php echo ( 'disable' !== $telemetry_reporting ) ? esc_attr( 'bbapp-hide' ) : ''; ?>'>
			<br>
			<label for="no_reporting">
				<input name="app_telemetry" id="disable_reporting" type="radio" value="disable" <?php checked( $telemetry_reporting, 'disable' ); ?>/>
				<?php esc_html_e( 'Disable telemetry', 'buddyboss-app' ); ?>
			</label>
			<p class="description">
				<?php
				esc_html_e(
					'Disabling telemetry will stop gathering and reporting usage statistics 
				about your configuration and the features and functionality you use. By disabling telemetry, 
				you will not be contributing towards the improvement of the product and the way we serve 
				you will be less relevant to you.',
					'buddyboss-app'
				);
				?>
			</p>
		</div>
		<?php
		$telemetry_html = ob_get_clean();

		$this->settings_form->add_field(
			$app_settings_provider->field(
				'app_telemetry',
				array(
					'type'  => 'html',
					'label' => __( 'Telemetry', 'buddyboss-app' ),
					'icon'  => 'bb-icon-f bb-icon-box',
					'value' => $telemetry_html,
				)
			)
		);
	}

	/**
	 * Saves the settings which are app id specific.
	 *
	 * @param array $settings Settings.
	 *
	 * @since 2.2.70
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {

		// Get old telemetry setting value.
		$old_telemetry_reporting = $settings['app_telemetry'];

		// Add the new value before save.
		// phpcs:ignore
		$settings['app_telemetry'] = isset( $_POST['app_telemetry'] ) ? sanitize_text_field( wp_unslash( $_POST['app_telemetry'] ) ) : '';

		ManageApp::instance()->update_app_settings( $settings );

		// After settings saved.
		$this->bbapp_admin_send_immediate_telemetry_on_complete( $old_telemetry_reporting );

		return true;
	}

	/**
	 * Get settings.
	 *
	 * @since 2.2.70
	 */
	public function get_settings() {
		return ManageApp::instance()->get_app_settings();
	}

	/**
	 * Checks old values and compares if it changed to complete reporting then send telemetry immediately.
	 *
	 * @param string $old_telemetry_reporting Old telemetry setting value.
	 *
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_admin_send_immediate_telemetry_on_complete( $old_telemetry_reporting ) {
		// Check if it changed to complete reporting then send telemetry immediately.
		// phpcs:ignore
		$new_telemetry_reporting = isset( $_POST['app_telemetry'] ) ? sanitize_text_field( wp_unslash( $_POST['app_telemetry'] ) ) : '';

		// Check if telemetry reporting has changed.
		if ( $old_telemetry_reporting === $new_telemetry_reporting ) {
			return;
		}

		// Dismiss telemetry notice if reporting status has changed.
		update_option( 'bbapp_telemetry_notice_dismissed', 1 );

		if ( 'complete' === $new_telemetry_reporting && class_exists( '\BuddyBossApp\Admin\Tools\Telemetry' ) ) {
			// Clear single scheduled cron.
			if ( wp_next_scheduled( 'bbapp_telemetry_report_single_cron_event' ) ) {
				wp_clear_scheduled_hook( 'bbapp_telemetry_report_single_cron_event' );
			}

			// Send telemetry.
			$bbapp_telemetry = Telemetry::instance();
			$bbapp_telemetry->send_telemetry_data();
		}
	}
}
