<?php
/**
 * Used by Access settings functions.
 *
 * @package BuddyBossApp\Admin\AccessControls
 */

namespace BuddyBossApp\Admin\AccessControls;

use BuddyBossApp\AccessControls\AccessRule;

/**
 * Access controls class.
 *
 * @since 1.5.2.1
 */
class Settings {

	/**
	 * Class instance.
	 *
	 * @since 1.5.2.1
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Admin menu page name
	 *
	 * @since 1.5.2.1
	 * @var string
	 */
	public $menu_page = 'bbapp-settings';

	/**
	 * Tab url slug
	 *
	 * @since 1.5.2.1
	 * @var string
	 */
	public $tab_name = 'bb_access_controls';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return Settings
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {}

	/**
	 * Load additional hooks such as : admin_init
	 * NOTE : Method is similar to concept of init
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function init() {
		$this->hooks();
	}

	/**
	 * Load all hooks.
	 *
	 * @since 1.5.2.1
	 */
	public function hooks() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );

		add_action( 'wp_ajax_get_setting_dropdown', array( $this, 'ajax_get_setting_dropdown' ) );
		add_action( 'wp_ajax_nopriv_get_setting_dropdown', array( $this, 'ajax_get_setting_dropdown' ) );

		add_filter( 'bbapp_settings_rest_response_access_controls', array( $this, 'rest_get_default_restrict_message' ), 10, 2 );
	}

	/**
	 * Register all section and fields.
	 *
	 * @since 1.5.2.1
	 */
	public function admin_init() {

		$register_settings = $this->register_settings();
		foreach ( $register_settings as $section_id => $section ) {

			$section_title     = ! empty( $section['title'] ) ? $section['title'] : '';
			$section_callback  = ! empty( $section['callback'] ) ? $section['callback'] : false;
			$tutorial_callback = ! empty( $section['tutorial_callback'] ) ? $section['tutorial_callback'] : false;
			$notice            = ! empty( $section['notice'] ) ? $section['notice'] : false;
			$icon              = ! empty( $section['icon'] ) ? $section['icon'] : '';
			$this->add_section( $section_id, $section_title, $section_callback, $tutorial_callback, $notice, $icon );

			if ( ! empty( $section['fields'] ) ) {

				foreach ( $section['fields'] as $field_name => $field ) {
					$field_label   = ! empty( $field['label'] ) ? $field['label'] : '';
					$callback      = ! empty( $field['callback'] ) ? $field['callback'] : '';
					$callback_args = ! empty( $field['callback_args'] ) ? $field['callback_args'] : array();
					$field_args    = ! empty( $field['field_args'] ) ? $field['field_args'] : array();
					$this->add_field( $field_name, $field_label, $callback, $field_args, $callback_args, $section_id );
				}
			}
		}

		// save the settings.
		$this->save_settings();
	}

	/**
	 * Admin int.
	 *
	 * @since 1.5.2.1
	 *
	 * @return false|void
	 */
	public function save_settings() {
		if ( ! $this->is_saving() ) {
			return false;
		}

		check_admin_referer( $this->tab_name . '-options' );

		$this->settings_save();
		do_action( 'bbapp_admin_tab_setting_save', $this->tab_name, $this );

		$args = array(
			'page'    => 'bbapp-settings',
			'setting' => 'access_controls',
			'updated' => true,
		);

		$redirect_to = bbapp_get_admin_url( add_query_arg( $args, 'admin.php' ) );
		wp_safe_redirect( $redirect_to );
	}

	/**
	 * Determine whether current request is saving on the current tab
	 *
	 * @since 1.5.2.1
	 */
	public function is_saving() {
		$page_get    = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$submit_post = ( ! empty( $_POST['submit'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['submit'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
		$setting_get = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $page_get ) || empty( $submit_post ) ) {
			return false;
		}

		if ( $this->menu_page !== $page_get ) {
			return false;
		}

		$active_tab = ! empty( $setting_get ) ? $setting_get : '';

		if ( "bb_{$active_tab}" !== $this->tab_name ) {
			return false;
		}

		return true;
	}

	/**
	 * Method to save the fields
	 *
	 * By default it'll loop throught the setting group's fields, but allow
	 * extended classes to have their own logic if needed
	 *
	 * @since 1.5.2.1
	 */
	public function settings_save() {
		global $wp_settings_fields;

		$fields                   = isset( $wp_settings_fields[ $this->tab_name ] ) ? (array) $wp_settings_fields[ $this->tab_name ] : array();
		$access_controls_settings = $this->bbapp_get_access_control_settings();

		foreach ( $fields as $section => $settings ) {
			foreach ( $settings as $setting_name => $setting ) {
				$value                                     = isset( $_POST[ $setting_name ] ) ? ( is_array( $_POST[ $setting_name ] ) ? map_deep( wp_unslash( $_POST[ $setting_name ] ), 'sanitize_text_field' ) : sanitize_text_field( wp_unslash( $_POST[ $setting_name ] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
				$access_controls_settings[ $setting_name ] = $value;
			}
		}

		$this->bbapp_update_access_control_settings( $access_controls_settings );
	}

	/**
	 * Get all access controls settings
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function bbapp_get_access_control_settings() {
		return get_option( 'bb_access_controls_default_settings', array() );
	}

	/**
	 * Get access controls settings by key
	 *
	 * @param string $key key of access controls settings.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function bbapp_get_access_control_setting( $key ) {
		$settings = $this->bbapp_get_access_control_settings();

		return ! empty( $settings[ $key ] ) ? $settings[ $key ] : '';
	}

	/**
	 * Update access control settings.
	 *
	 * @param array $settings list of access controls settings.
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_update_access_control_settings( $settings ) {
		update_option( 'bb_access_controls_default_settings', $settings );
	}

	/**
	 * Add a wp setting section into current tab. Chainable
	 *
	 * @param string $id                Section id.
	 * @param string $title             Section title.
	 * @param string $callback          Section html callback.
	 * @param string $tutorial_callback Tutorial link callback.
	 * @param string $notice            Section notice.
	 * @param string $icon              Section icon.
	 *
	 * @since 1.5.2.1
	 *
	 * @return $this
	 */
	public function add_section( $id, $title, $callback = '__return_null', $tutorial_callback = '', $notice = '', $icon = '' ) {
		global $wp_settings_sections;
		add_settings_section( $id, $title, $callback, $this->tab_name );
		if ( ! empty( $tutorial_callback ) ) {
			$wp_settings_sections[ $this->tab_name ][ $id ]['tutorial_callback'] = $tutorial_callback; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		if ( ! empty( $notice ) ) {
			$wp_settings_sections[ $this->tab_name ][ $id ]['notice'] = $notice; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		if ( ! empty( $icon ) ) {
			$wp_settings_sections[ $this->tab_name ][ $id ]['icon'] = $icon; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		return $this;
	}

	/**
	 * Add a wp setting field to a wp setting section. Chainable
	 *
	 * @param string $name          Field name.
	 * @param string $label         Field label.
	 * @param string $callback      Field callback.
	 * @param array  $field_args    Field arguments.
	 * @param array  $callback_args Field callback arguments.
	 * @param string $id            Field id.
	 *
	 * @since 1.5.2.1
	 *
	 * @return $this
	 */
	public function add_field( $name, $label, $callback, $field_args = array(), $callback_args = array(), $id = '' ) {

		add_settings_field( $name, $label, $callback, $this->tab_name, $id, $callback_args );
		register_setting( $this->tab_name, $name, $field_args );

		return $this;
	}

	/**
	 * Output the form html on the setting page (not including tab and page title)
	 *
	 * @since 1.5.2.1
	 */
	public function form_html() {
		settings_fields( $this->tab_name );
		$this->bbapp_custom_do_settings_sections( $this->tab_name );
		printf(
			'<p class="submit"><input type="submit" name="submit" class="button-primary access-control-settings-submit" value="%s" /></p>',
			esc_attr_x( 'Save Changes', 'bb-access-control', 'buddyboss-app' )
		);
	}

	/**
	 * Prints out all settings sections added to a particular settings page
	 *
	 * Part of the Settings API. Use this in a settings page callback function
	 * to output all the sections and fields that were added to that $page with
	 * add_settings_section() and add_settings_field()
	 *
	 * @param string $page                 The slug name of the page whose settings sections you want to output.
	 *
	 * @since 1.5.2.1
	 *
	 * @global  array $wp_settings_fields   Storage array of settings fields and info about their pages/sections
	 * @global array  $wp_settings_sections Storage array of all settings sections added to admin pages
	 */
	public function bbapp_custom_do_settings_sections( $page ) {
		global $wp_settings_sections, $wp_settings_fields;

		if ( ! isset( $wp_settings_sections[ $page ] ) ) {
			return;
		}

		foreach ( (array) $wp_settings_sections[ $page ] as $section ) {
			echo sprintf( '<div id="%1$s" class="%2$s">', esc_html( $section['id'] ), esc_html( "buddyboss-app-admin-card section-{$section['id']}" ) );

			$has_tutorial_btn = ( isset( $section['tutorial_callback'] ) && ! empty( $section['tutorial_callback'] ) ) ? 'has_tutorial_btn' : '';
			if ( $section['title'] ) {

				echo '<h2 class=' . esc_attr( $has_tutorial_btn ) . '>';
				// Show icon if it's set.
				if ( ! empty( $section['icon'] ) ) {
					echo '<i class="' . esc_attr( $section['icon'] ) . '"></i>';
				}
				echo wp_kses(
					$section['title'],
					array(
						'a' => array(
							'href' => array(),
							'rel'  => array(),
						),
					)
				);

				if ( isset( $section['tutorial_callback'] ) && ! empty( $section['tutorial_callback'] ) ) {
					?>
					<div class="bbapp-tutorial-btn">
						<?php call_user_func( $section['tutorial_callback'], $section ); ?>
					</div>
					<?php
				}

				echo "</h2>\n";
			}

			if ( $section['callback'] ) {
				call_user_func( $section['callback'], $section );
			}

			if ( ! isset( $wp_settings_fields ) || ! isset( $wp_settings_fields[ $page ] ) || ! isset( $wp_settings_fields[ $page ][ $section['id'] ] ) ) {
				continue;
			}

			echo '<table class="form-table">';
			$this->bp_custom_do_settings_fields( $page, $section['id'] );
			echo '</table>';

			if ( isset( $section['notice'] ) && ! empty( $section['notice'] ) ) {
				?>
				<div class="display-notice bb-bottom-notice">
					<?php echo wp_kses_post( $section['notice'] ); ?>
				</div>
				<?php
			}
			echo '</table></div>';
		}
	}

	/**
	 * Print out the settings fields for a particular settings section
	 *
	 * Part of the Settings API. Use this in a settings page to output
	 * a specific section. Should normally be called by do_settings_sections()
	 * rather than directly.
	 *
	 * @param string $page               Slug title of the admin page who's settings fields you want to show.
	 * @param string $section            Slug title of the settings section who's fields you want to show.
	 *
	 * @global array $wp_settings_fields Storage array of settings fields and their pages/sections
	 *
	 * @since 1.5.2.1
	 */
	public function bp_custom_do_settings_fields( $page, $section ) {
		global $wp_settings_fields;
		if ( ! isset( $wp_settings_fields[ $page ][ $section ] ) ) {
			return;
		}

		foreach ( (array) $wp_settings_fields[ $page ][ $section ] as $field ) {
			$class = '';

			if ( ! empty( $field['args']['class'] ) ) {
				$class = $field['args']['class'];
				if ( is_callable( $field['args']['class'] ) ) {
					$class = call_user_func( $field['args']['class'] );
				}

				$class = ' class=' . esc_attr( $class ) . '';
			}

			echo sprintf( '<tr %1$s>', esc_html( $class ) );

			if ( ! empty( $field['args']['label_for'] ) ) {
				echo '<th scope="row"><label for="' . esc_attr( $field['args']['label_for'] ) . '">' . esc_html( $field['title'] ) . '</label></th>';
			} else {
				echo '<th scope="row">' . esc_html( $field['title'] ) . '</th>';
			}

			echo '<td>';
			if ( ! empty( $field['callback'] ) ) {
				call_user_func( $field['callback'], $field['args'] );
			}
			echo '</td>';
			echo '</tr>';
		}
	}

	/**
	 * Function to get the relevant dropdown.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_get_setting_dropdown() {
		$result            = array();
		$result['message'] = __( 'Something went wrong please try again later.', 'buddyboss-app' );
		$nonce             = ( ! empty( $_POST['nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp-get-dropdown' ) ) {
			$result['message'] = __( 'Security check failed.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$selection = ( ! empty( $_POST['value'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['value'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$item_ids  = array();
		if ( ! empty( $_POST['item_ids'] ) ) {
			$_POST['item_ids'] = stripslashes_from_strings_only( bbapp_input_clean( array_map( 'absint', $_POST['item_ids'] ) ) );
			$item_ids          = filter_input( INPUT_POST, 'item_ids', FILTER_SANITIZE_NUMBER_INT, FILTER_FORCE_ARRAY );
			$item_ids          = array_values( array_filter( $item_ids ) );
		}
		switch ( $selection ) {
			case 'app_page':
				ob_start();
				$this->create_custom_dropdown( 'app_page', '', $item_ids );
				$html              = ob_get_clean();
				$result['message'] = __( 'App page found.', 'buddyboss-app' );
				$result['html']    = $html;
				wp_send_json_success( $result );
				break;
			case 'iap_products':
				$result['message'] = __( 'Product screen found.', 'buddyboss-app' );
				$result['html']    = '';
				wp_send_json_success( $result );
				break;
			case 'iap_product':
				$this->create_custom_dropdown( 'iap_product' );
				$html              = ob_get_clean();
				$result['message'] = __( 'In app purchase found.', 'buddyboss-app' );
				$result['html']    = $html;
				wp_send_json_success( $result );
				break;
		}
		wp_send_json_error( $result );
	}


	/**
	 * Render Setting Form.
	 *
	 * @since 1.5.2.1
	 */
	public function render_settings() {
		$args = array(
			'page'    => 'bbapp-settings',
			'setting' => 'access_controls',
		);

		$form_action = bbapp_get_admin_url( add_query_arg( $args, 'admin.php' ) );
		$messages    = '';
		if ( isset( $_REQUEST['updated'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$messages = sprintf(
				'<div id="message" class="updated"><p>%s</p></div>',
				__( 'Your settings have been saved.', 'buddyboss-app' )
			);
		}
		if ( ! empty( $messages ) ) {
			printf( '<div class="bbapp-err-msgs">%s</div>', wp_kses_post( $messages ) );
		}
		?>
		<form action="<?php echo esc_url( $form_action ); ?>" id="bbapp-aac-form" method="post">
			<?php $this->form_html(); ?>
		</form>
		<script>
			jQuery( document ).ready( function( $ ) {
				$( document ).on( 'submit', '#bbapp-aac-form', function() {
					$( '.bbapp-field-err-cmn' ).remove();

					const form = jQuery( this );
					let validate = true;
					const selectCmn = form.find( '.bb-validate-field-cmn' );
					const errMsgEl = jQuery( '.bbapp-err-msgs' );

					errMsgEl.empty();

					const selectCmnIds = jQuery( selectCmn ).map( function() {
						return jQuery( this ).attr( 'id' );
					} );

					jQuery( selectCmnIds ).each( function( index, value ) {
						const selectEl = jQuery( '#' + value );
						const selectVal = selectEl.val();

						if ( 'undefined' === typeof selectVal || '' === selectVal || 0 === parseInt( selectVal ) ) {
							validate = false;
							selectEl.addClass( 'bbapp-has-err' );
							if ( ' ' !== selectEl.attr( 'data-err-msg' ) ) {
								if ( selectEl.hasClass( 'bbapp-select2-cmn-wrp' ) ) {
									selectEl.parent().append( '<p class="bbapp-field-err-cmn bbapp-field-err bbapp-error-msg">' + selectEl.attr( 'data-err-msg' ) + '</p>' );
								} else {
									selectEl.after( '<p class="bbapp-field-err-cmn bbapp-field-err bbapp-error-msg">' + selectEl.attr( 'data-err-msg' ) + '</p>' );
								}
							} else {
								selectEl.parent().find( '.description' ).addClass( 'bbapp-error-msg' );
							}
						} else if ( selectEl.hasClass( 'bbapp-has-err' ) ) {
							selectEl.removeClass( 'bbapp-has-err' );
							selectEl.parent().find( '.description' ).removeClass( 'bbapp-error-msg' );
							selectEl.parent().find( '.bbapp-field-err-cmn' ).remove();
						}
					} );

					if ( false === validate ) {
						errMsgEl.html( '<div id="message" class="bbapp-error"><p>Please resolve the errors below to save your settings.</p></div>' );

						return false;
					}
				} );
			} );

		</script>
		<?php
	}

	/**
	 * Register access controls settings sections and fields.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function register_settings() {

		$settings['bb_access_controls'] = array(
			'title'             => _x( 'Default Restricted Message', 'bb-access-control', 'buddyboss-app' ),
			'callback'          => array( $this, 'bb_access_controls_info_section_callback' ),
			'tutorial_callback' => array( $this, 'bbapp_access_control_tutorial' ),
			'icon'              => 'bb-icon-bf bb-icon-lock',
		);

		$settings['bb_access_controls']['fields']['title'] = array(
			'label'         => _x( 'Default Title', 'bb-access-control', 'buddyboss-app' ),
			'callback'      => array( $this, 'bb_access_controls_text_box' ),
			'callback_args' => array(
				'value' => function () {
					$value = $this->bbapp_get_access_control_setting( 'title' );

					return ! empty( $value ) ? $value : _x( 'Access Denied', 'bb-access-control', 'buddyboss-app' );
				},
				'name'  => 'title',
				'id'    => 'bb_access_controls_default_title',
				'desc'  => __( 'Enter the default title for restricted messages.', 'buddyboss-app' ),
			),
			'field_args'    => array(),
		);

		$settings['bb_access_controls']['fields']['content'] = array(
			'label'         => _x( 'Default Content', 'bb-access-control', 'buddyboss-app' ),
			'callback'      => array( $this, 'bb_access_controls_textarea' ),
			'callback_args' => array(
				'value'          => function () {
					$value = $this->bbapp_get_access_control_setting( 'content' );

					return ! empty( $value ) ? $value : __( 'You do not have permission to access to this content.', 'buddyboss-app' );
				},
				'selector_class' => array(
					'bb-validate-field-cmn',
				),
				'error_msg'      => ' ',
				'name'           => 'content',
				'id'             => 'bb_access_controls_default_content',
				'desc'           => __( 'Enter the default content for restricted messages.', 'buddyboss-app' ),
			),
			'field_args'    => array(),
		);

		$options = array(
			'no_button'    => _x( 'No Button', 'bb-access-control', 'buddyboss-app' ),
			'app_page'     => _x( 'Open App Page', 'bb-access-control', 'buddyboss-app' ),
			'iap_products' => _x( 'Open Product Screen', 'bb-access-control', 'buddyboss-app' ),
			'iap_product'  => _x( 'Open In-App Product', 'bb-access-control', 'buddyboss-app' ),
		);

		if ( ! bbapp_is_active( 'iap' ) ) {
			unset( $options['iap_products'] );
			unset( $options['iap_product'] );
		}

		$settings['bb_access_controls']['fields']['button_action'] = array(
			'label'         => _x( 'Default Button Action', 'bb-access-control', 'buddyboss-app' ),
			'callback'      => array( $this, 'bb_access_controls_button_action' ),
			'callback_args' => array(
				'value'             => function () {
					$value = $this->bbapp_get_access_control_setting( 'button_action' );

					return ! empty( $value ) ? $value : 'no_button';
				},
				'options'           => $options,
				'option_item_value' => function () {
					$value = $this->bbapp_get_access_control_setting( 'button_action' );

					return $this->bbapp_get_access_control_setting( "{$value}_id" );
				},
				'name'              => 'button_action',
				'id'                => 'bb_access_controls_default_button_action',
				'desc'              => __( 'Enter the default button action for the restricted messages.', 'buddyboss-app' ),
			),
			'field_args'    => array(),
		);

		$settings['bb_access_controls']['fields']['app_page_id']    = array(
			'label'         => '',
			'callback'      => array(),
			'callback_args' => array(
				'class' => 'hidden',
			),
			'field_args'    => array(),
		);
		$settings['bb_access_controls']['fields']['iap_product_id'] = array(
			'label'         => '',
			'callback'      => array(),
			'callback_args' => array(
				'class' => 'hidden',
			),
			'field_args'    => array(),
		);

		$settings['bb_access_controls']['fields']['button_text'] = array(
			'label'         => _x( 'Default Button Text', 'bb-access-control', 'buddyboss-app' ),
			'callback'      => array( $this, 'bb_access_controls_text_box' ),
			'callback_args' => array(
				'value'          => function () {
					$value = $this->bbapp_get_access_control_setting( 'button_text' );

					return ! empty( $value ) ? $value : _x( 'Learn More', 'bb-access-control', 'buddyboss-app' );
				},
				'selector_class' => array(
					'regular-text',
					'bb-validate-field-cmn',
				),
				'error_msg'      => ' ',
				'name'           => 'button_text',
				'id'             => 'bb_access_controls_default_button_text',
				'desc'           => __( 'Enter the default button title for restricted messages.', 'buddyboss-app' ),
				'class'          => function () {
					$class[]               = 'access_controls_default_button_text_tr';
					$default_button_action = $this->bbapp_get_access_control_setting( 'default_button_action' );
					if ( 'no_button' === $default_button_action ) {
						$class[] = 'hidden';
					}

					return implode( ' ', $class );
				},
			),
			'field_args'    => array(),
		);

		$settings['bb_access_controls']['fields']['show_teaser'] = array(
			'label'         => _x( 'Default Settings', 'bb-access-control', 'buddyboss-app' ),
			'callback'      => array( $this, 'bb_access_controls_checkbox' ),
			'callback_args' => array(
				'value'          => function () {
					$value = $this->bbapp_get_access_control_setting( 'show_teaser' );

					return ! empty( $value ) ? $value : 0;
				},
				'name'           => 'show_teaser',
				'id'             => 'bb_access_controls_show_teaser',
				'label_checkbox' => __( 'Show a teaser of the content behind the message', 'buddyboss-app' ),
				'desc'           => __( 'When enabled, by default members will see a teaser of the content behind the message, but will not be able to interact with it. ', 'buddyboss-app' ),
			),
			'field_args'    => array(),
		);

		return $settings;
	}

	/**
	 * Access controls settings section callback.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_access_controls_info_section_callback() {
		?>
		<p>
			<?php
			printf(
				'<p class="description">%s</p>',
				esc_html__( 'This is the default message that will show to members without access to a restricted content. These settings can be overwritten when setting up individual access rules.', 'buddyboss-app' )
			);
			?>
		</p>
		<?php
	}

	/**
	 * Link to Access controls tutorial
	 *
	 * @since 1.5.2.1
	 */
	public function bbapp_access_control_tutorial() {
		?>
		<p>
			<a class="button" href="
			<?php
			echo esc_url(
				bbapp_get_admin_url(
					add_query_arg(
						array(
							'page'    => 'bbapp-help',
							'article' => '125684',
						),
						'admin.php'
					)
				)
			);
			?>
			" target="_blank">
				<?php echo esc_html_x( 'View Tutorial', 'bb-access-control', 'buddyboss-app' ); ?>
			</a>
		</p>
		<?php
	}

	/**
	 * Textbox setting field
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_access_controls_text_box( $args ) {
		$default_args = array(
			'value'          => '',
			'selector_class' => array(
				'regular-text',
			),
			'desc'           => '',
			'error_msg'      => '',
		);
		$args         = wp_parse_args( $args, $default_args );
		$class        = implode( ' ', $args['selector_class'] );
		$value        = ! empty( $args['value'] ) && is_callable( $args['value'] ) ? call_user_func( $args['value'] ) : $args['value'];
		printf(
			'<input name="%1$s" id="%2$s" type="text" class="%3$s" value="%4$s" data-err-msg="%5$s">',
			esc_attr( $args['name'] ),
			esc_attr( $args['id'] ),
			esc_attr( $class ),
			esc_attr( $value ),
			esc_attr( $args['error_msg'] )
		);
		if ( ! empty( $args['desc'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['desc'] ) );
		}
	}

	/**
	 * Default content setting field.
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_access_controls_textarea( $args ) {
		$default_args = array(
			'value'          => '',
			'desc'           => '',
			'selector_class' => array(),
			'error_msg'      => '',
		);
		$args         = wp_parse_args( $args, $default_args );
		$value        = ! empty( $args['value'] ) && is_callable( $args['value'] ) ? call_user_func( $args['value'] ) : $args['value'];
		$class        = implode( ' ', $args['selector_class'] );
		printf(
			'<textarea name="%1$s" id="%2$s" rows="3" cols="70" class="%3$s" data-err-msg="%4$s">%5$s</textarea>',
			esc_attr( $args['name'] ),
			esc_attr( $args['id'] ),
			esc_attr( $class ),
			esc_attr( $args['error_msg'] ),
			wp_kses_post( $value )
		);
		if ( ! empty( $args['desc'] ) ) {
			printf( '<p class="description">%s</p>', wp_kses_post( $args['desc'] ) );
		}
	}

	/**
	 * Default button actions setting field.
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_access_controls_button_action( $args ) {
		require bbapp()->plugin_dir . 'views/access-controls/fields/button-action.php';
	}

	/**
	 * Default settings field.
	 *
	 * @param array $args Arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_access_controls_checkbox( $args ) {
		$default_args = array(
			'name'           => '',
			'label_checkbox' => '',
			'desc'           => '',
			'value'          => 0,
		);
		$args         = wp_parse_args( $args, $default_args );
		$value        = ! empty( $args['value'] ) && is_callable( $args['value'] ) ? call_user_func( $args['value'] ) : $args['value'];

		printf(
			'<input name="%1$s" id="%2$s" type="checkbox" value="1" %3$s>',
			esc_attr( $args['name'] ),
			esc_attr( $args['id'] ),
			checked( $value, 1, false )
		);
		printf( '<label for="%1$s">%2$s</label>', esc_attr( $args['id'] ), esc_html( $args['label_checkbox'] ) );

		if ( ! empty( $args['desc'] ) ) {
			printf( '<p class="description">%s</p>', wp_kses_post( $args['desc'] ) );
		}
	}

	/**
	 * Creating custom dropdown.
	 *
	 * @param string       $action      Form action.
	 * @param string       $value       Form dropdown value.
	 * @param array|string $exclude_item_ids       Form dropdown value.
	 *
	 * @since 1.5.2.1
	 */
	public function create_custom_dropdown( $action, $value = '', $exclude_item_ids = array() ) {
		switch ( $action ) {
			case 'app_page':
				$app_pages = get_posts(
					array(
						'post_type'   => 'app_page',
						'numberposts' => - 1,
						'exclude'     => $exclude_item_ids,
					)
				);

				printf( '<div class="bbapp_access_control_setting_dropdown_wrap"> <select name="app_page_id" class="bbapp_access_control_setting_dropdown bbapp-select2-cmn-wrp" id="%1$s" for="%1$s" data-err-msg="%2$s">', esc_html( "bb_access_controls_{$action}" ), esc_attr_x( 'Select an app page.', 'bb-access-control', 'buddyboss-app' ) );
				printf(
					'<option value="%s">%s</option>',
					'',
					esc_html_x( '-- Select App Page --', 'bb-access-control', 'buddyboss-app' )
				);
				foreach ( $app_pages as $app_page ) {
					$post_title = strlen( $app_page->post_title ) > 50 ? substr( $app_page->post_title, 0, 50 ) . '...' : $app_page->post_title;
					printf(
						'<option value="%s" %s>%s</option>',
						esc_attr( $app_page->ID ),
						selected( $value, $app_page->ID, false ),
						esc_html( $post_title )
					);
				}
				printf( '</select></div>' );
				$is_hidden = ( ! empty( AccessRule::instance()->get_access_rule( $value, 'app_page' ) ) && 'publish' === get_post_status( $value ) ) ? '' : 'hidden';
				printf(
					'<p class="bbapp-notice-box bbapp-notice-box--warning %s">%s</p>',
					esc_attr( $is_hidden ),
					esc_html_x( 'This app page is restricted to certain members. We recommend removing any access rules from a page used in this setting.', 'access controls general settings', 'buddyboss-app' )
				);
				break;
			case 'iap_product':
				if ( bbapp_is_active( 'iap' ) ) {
					$products = bbapp_iap_get_products( array() );
					printf( '<div class="bbapp_access_control_setting_dropdown_wrap"> <select name="iap_product_id" class="bbapp_access_control_setting_dropdown bbapp-select2-cmn-wrp" id="%1$s" for="%1$s" data-err-msg="%2$s">', esc_html( "bb_access_controls_{$action}" ), esc_attr_x( 'Select an app product.', 'bb-access-control', 'buddyboss-app' ) );
					printf(
						'<option value="%s">%s</option>',
						'',
						esc_html_x( '-- Select In-App Product --', 'bb-access-control', 'buddyboss-app' )
					);
					foreach ( $products as $product ) {
						$product_name = strlen( $product['name'] ) > 50 ? substr( $product['name'], 0, 50 ) . '...' : $product['name'];
						printf(
							'<option value="%s" %s>%s</option>',
							esc_attr( $product['id'] ),
							selected( $value, $product['id'], false ),
							esc_html( $product_name )
						);
					}
					printf( '</select></div>' );
				}
				break;
		}
	}

	/**
	 * Function to get access settings on rest.
	 *
	 * @param array        $access_controls Access control array data.
	 * @param object|array $request         Rest request.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function rest_get_default_restrict_message( $access_controls, $request ) {
		$settings         = $this->bbapp_get_access_control_settings();
		$default_settings = array(
			'title'          => _x( 'Access Denied', 'bb-access-control', 'buddyboss-app' ),
			'content'        => __( 'You do not have permission to access to this content.', 'buddyboss-app' ),
			'button_action'  => 'no_button',
			'button_text'    => _x( 'Learn More', 'bb-access-control', 'buddyboss-app' ),
			'iap_product_id' => 0,
			'app_page_id'    => 0,
			'show_teaser'    => false,
		);

		$settings['iap_product_id'] = isset( $settings['iap_product_id'] ) ? (int) $settings['iap_product_id'] : 0;
		$settings['app_page_id']    = isset( $settings['app_page_id'] ) ? (int) $settings['app_page_id'] : 0;
		$settings['show_teaser']    = ! empty( $settings['show_teaser'] );

		$settings = wp_parse_args( $settings, $default_settings );

		$access_controls['default_restrict_message'] = $settings;

		return $access_controls;
	}
}
