<?php
/**
 * Used by Access Rules functions.
 *
 * @package BuddyBossApp\Admin\AccessControls
 */

namespace BuddyBossApp\Admin\AccessControls;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\Admin\InAppPurchases\ProductHelper;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Access Rules class.
 */
class AccessRules {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Access Rules constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return AccessRules
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'load_init' ) );
	}

	/**
	 * Load init.
	 *
	 * @since 1.5.2.1
	 */
	public function load_init() {
		add_action( 'wp_ajax_bb_access_controls_get_rule_template', array( $this, 'ajax_rule_template' ) );
		add_action( 'wp_ajax_bb_access_controls_update_rule_data', array( $this, 'ajax_update_rule_data' ) );
		add_action( 'wp_ajax_bb_access_controls_check_rule_exists', array( $this, 'check_rule_exists' ) );
	}

	/**
	 * Renders screen based on action variable in url
	 *
	 * @since 1.5.2.1
	 */
	public function render() {

		if ( ! current_user_can( 'manage_options' ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permissions to access this page.', 'buddyboss-app' ) . '</p>';
		}

		$settings_screen = bb_access_controls_get_settings_screen();
		$tab_get         = ( ! empty( $_GET['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$current_tab     = ! empty( $tab_get ) ? $tab_get : 'general';
		?>

		<div class="wrap wrap--ctrl">
			<?php if ( ! empty( $settings_screen ) ) { ?>
				<div class="bbapp-tab">
					<?php foreach ( $settings_screen as $key => $setting_screen ) { ?>
						<?php $tab_url = admin_url( 'admin.php?page=bbapp-access-controls&tab=' . $key ); ?>
						<a href="<?php echo esc_url( $tab_url ); ?>" class="tablinks <?php echo ( $current_tab === $key ) ? 'active' : ''; ?>" id="defaultOpen">
							<?php echo esc_html( $setting_screen['menu_title'] ); ?>
						</a>
					<?php } ?>
				</div>
				<?php
				if ( ! empty( $settings_screen[ $current_tab ] ) ) {
					$setting_screen = $settings_screen[ $current_tab ];
					?>
					<div id="<?php echo esc_attr( $setting_screen['id'] ); ?>" class="bbapp-tabcontent bbapp-active-content">
						<div class="buddyboss-app-admin-card full-width <?php echo ( ! empty( $setting_screen['options'] ) && ! empty( $setting_screen['options']['sub_tab'] ) ) ? 'buddyboss-app-admin-card--subnav' : ''; ?>">
							<div class="bbapp-ie">
								<div class="bbapp-ie__heading bbapp-ie__sub">
									<h2 class="title">
										<?php echo esc_html( $setting_screen['page_title'] ); ?>
										<?php if ( ! empty( $setting_screen['options'] ) && ! empty( $setting_screen['options']['tutorial_link'] ) ) { ?>
											<a class="button button-secondary bbapp-tutorial-btn" target="_blank" href="<?php echo esc_url( $setting_screen['options']['tutorial_link'] ); ?>"> <?php esc_html_e( 'View Tutorial', 'buddyboss-app' ); ?></a>
										<?php } ?>
									</h2>
									<?php
									if ( ! empty( $setting_screen['options'] ) && ! empty( $setting_screen['options']['sub_tab'] ) ) {
										$current_sub_tab = ! empty( $_GET['sub_tab'] ) ? bbapp_input_clean( wp_unslash( $_GET['sub_tab'] ) ) : $current_tab; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
										?>
										<div class="bbapp-tablinks-sub-tab">
											<?php
											foreach ( $setting_screen['options']['sub_tab'] as $sub_tab_key => $sub_tab_val ) {
												$sub_tab_url = admin_url( "admin.php?page=bbapp-access-controls&tab={$current_tab}&sub_tab={$sub_tab_key}" );
												?>
												<a class="tablinks <?php echo ( $current_sub_tab === $sub_tab_key ) ? 'active' : ''; ?>" href="<?php echo esc_url( $sub_tab_url ); ?>">
													<?php echo esc_html( $sub_tab_val ); ?>
												</a>
											<?php } ?>
										</div>
									<?php } ?>
								</div>
								<div class="bbapp-ie__block">
									<div id="manageapp-container">
										<?php
										if ( ! empty( $setting_screen['render_callback'] ) && is_callable( $setting_screen['render_callback'] ) ) {
											call_user_func( $setting_screen['render_callback'] );
										}
										?>
									</div>
								</div>
							</div>
						</div>
					</div>
					<?php
				}
			}
			?>
		</div>
		<?php
	}

	/**
	 * Function to get access control rule template.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_rule_template() {

		$result            = array();
		$result['message'] = __( 'Something went wrong please try again later.', 'buddyboss-app' );
		$nonce             = ( ! empty( $_POST['nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp_sort_nonce_' . get_current_user_id() ) ) {
			$result['message'] = __( 'Security check failed.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$item_ids = filter_input( INPUT_POST, 'item_ids', FILTER_SANITIZE_NUMBER_INT, FILTER_FORCE_ARRAY );
		$item_ids = array_values( array_filter( $item_ids ) );

		$item_type = ( ! empty( $_POST['item_type'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['item_type'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$extra     = filter_input( INPUT_POST, 'extra', FILTER_DEFAULT, FILTER_FORCE_ARRAY );

		$is_bulk = ( 1 < count( $item_ids ) );
		$item_id = ( ! $is_bulk ) ? $item_ids[0] : 0;

		/**
		 * Get rule data.
		 *
		 * @param array  $default   Default data.
		 * @param array  $item_id   ID of current item.
		 * @param string $item_type Type of item.
		 *
		 * @since 1.5.2.1
		 */
		$settings = apply_filters( "bb_access_controls_get_rule_data_{$item_type}", array(), $item_id, $item_type );

		if ( empty( $settings ) ) {
			$settings = AccessRule::instance()->get_access_rule( $item_id, $item_type );
			$settings = ! empty( $settings['rule_data'] ) ? $settings['rule_data'] : array();
		}

		if ( ! empty( $settings['restricted_message']['content']['button_action'] ) && 'app_page' === $settings['restricted_message']['content']['button_action'] && 'publish' !== get_post_status( $settings['restricted_message']['content']['app_page_id'] ) ) {
			$settings['restricted_message']['content']['button_action'] = 'no_button';
		}

		if ( bbapp_is_active( 'iap' ) && ! empty( $settings['restricted_message']['content']['button_action'] ) && 'iap_product' === $settings['restricted_message']['content']['button_action'] && 'published' !== ProductHelper::instance()->get_iap_product_status( $settings['restricted_message']['content']['iap_product_id'] ) ) {
			$settings['restricted_message']['content']['button_action'] = 'no_button';
		}

		if ( ! empty( $item_ids ) && is_array( $item_ids ) ) {
			$item_ids = htmlspecialchars( wp_json_encode( $item_ids ), ENT_QUOTES, 'UTF-8' );
		}

		$restrict_to_visible        = isset( $extra['restrict_to_visible'] ) ? $extra['restrict_to_visible'] : false;
		$restricted_message_visible = isset( $extra['restricted_message_visible'] ) ? $extra['restricted_message_visible'] : false;

		ob_start();
		include bbapp()->plugin_dir . 'views/access-controls/access-rules/rule-modal.php';
		$html = ob_get_clean();
		if ( ! empty( $html ) ) {
			$result['message'] = '';
			$result['content'] = $html;
			$result['type']    = 'html';
			wp_send_json_success( $result );
		}

		wp_send_json_error( $result );
	}

	/**
	 * Function to update Access rule data.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_update_rule_data() {
		$result            = array();
		$result['is_edit'] = false;
		$_post             = filter_input_array( INPUT_POST, FILTER_DEFAULT );
		$item_ids          = ! empty( $_post['item-ids'] ) ? $_post['item-ids'] : 0;
		$item_ids          = json_decode( htmlspecialchars_decode( $item_ids ) );
		$item_type         = ! empty( $_post['item-type'] ) ? $_post['item-type'] : '';
		$is_bulk           = ! empty( $_post['is_bulk'] ) ? $_post['is_bulk'] : false;

		$data = array(
			'_wpnonce'                => $_post['_wpnonce'],
			'access_control_settings' => $_post['access_control_settings'],
		);

		$validate = true;
		$messages = array();

		/**
		 * Validations.
		 */
		if ( empty( $data['_wpnonce'] ) || ! wp_verify_nonce( $data['_wpnonce'], "bbapp-access-{$item_type}-edit" ) ) {
			$validate   = false;
			$messages[] = __( 'Security check failed.', 'buddyboss-app' );
		}

		/**
		 * Validate Item Type.
		 */
		if ( ! bb_access_controls_is_rules_item_type_registered( $item_type ) ) {
			$validate   = false;
			$messages[] = __( 'Provided item type is not registered.', 'buddyboss-app' );
		}

		if ( ! empty( $data['access_control_settings']['rule_condition'] ) && 'all-logged-in-members' !== $data['access_control_settings']['rule_condition'] ) {
			if ( ! empty( $data['access_control_settings']['group_ids'] ) && ! array_filter( $data['access_control_settings']['group_ids'] ) ) {
				$validate   = false;
				$messages[] = __( 'Select at least one access group.', 'buddyboss-app' );
			}
		}

		if ( 'custom' === $data['access_control_settings']['restricted_message']['source'] ) {
			if ( empty( $data['access_control_settings']['restricted_message']['content']['content'] ) ) {
				$validate   = false;
				$messages[] = __( 'Enter the content for the restricted message.', 'buddyboss-app' );
			}
			if ( 'no_button' !== $data['access_control_settings']['restricted_message']['content']['button_action'] ) {
				if ( empty( $data['access_control_settings']['restricted_message']['content']['button_text'] ) ) {
					$validate   = false;
					$messages[] = __( 'Enter the text for button.', 'buddyboss-app' );
				}
				if ( 'app_page_id' === $data['access_control_settings']['restricted_message']['content']['button_action'] && empty( $_post[ $data['access_control_settings']['restricted_message']['content']['button_action'] ] ) ) {
					$validate   = false;
					$messages[] = __( 'Select an app page.', 'buddyboss-app' );
				}
				if ( 'iap_product_id' === $data['access_control_settings']['restricted_message']['content']['button_action'] && empty( $_post[ $data['access_control_settings']['restricted_message']['content']['button_action'] ] ) ) {
					$validate   = false;
					$messages[] = __( 'Select an app product.', 'buddyboss-app' );
				}
			}
		}

		if ( ! $validate ) {
			$result['messages'] = $messages;
			if ( 1 === count( $item_ids ) ) {
				$result['message'] = __( 'There was a problem saving the access rule: ', 'buddyboss-app' );
			} else {
				$result['message'] = __( 'There was a problem saving the access rules: ', 'buddyboss-app' );
			}
			wp_send_json_error( $result );
		} else {
			$settings = $this->bb_prepare_access_control_settings( $data['access_control_settings'], $_post );
			/**
			 * This filter is for internal use only.
			 *
			 * @param array|int $item_ids  Item id.
			 * @param string    $item_type Item type.
			 * @param array     $settings  Setting array.
			 *
			 * @since 1.5.2.1
			 */
			$saved = apply_filters( "bb_access_controls_save_rule_data_{$item_type}", null, $item_ids, $item_type, $settings );

			if ( null === $saved ) {
				$saved = AccessRule::instance()->save_access_rules( $item_ids, $item_type, $settings );
			}

			if ( $saved ) {
				$result['rule_preview']    = bb_access_controls_rule_preview( $settings ); // html entry for admin access rules list.
				$result['message']         = true === $is_bulk ? __( 'All access rules has been successfully saved.', 'buddyboss-app' ) : __( 'The access rule has been successfully saved.', 'buddyboss-app' );
				$result['rule']['item_id'] = $item_ids; // html entry for admin access rules list.

				/**
				 * This filter is for internal use only.
				 *
				 * @param array  $result    Result data.
				 * @param int    $item_id   Item id.
				 * @param string $item_type Item type.
				 * @param array  $_post     Post data.
				 *
				 * @since 1.5.2.1
				 */
				$result = apply_filters( 'bb_access_controls_update_rule_data_result', $result, $item_ids, $item_type, $_post );

				wp_send_json_success( $result );
			} else {
				if ( 1 === count( $item_ids ) ) {
					$result['message'] = __( 'There was a problem saving the access rule: [DB error].', 'buddyboss-app' );
				} else {
					$result['message'] = __( 'There was a problem saving the access rules: [DB error].', 'buddyboss-app' );
				}
				wp_send_json_error( $result );
			}
		}
	}

	/**
	 * Function to prepare access control settings.
	 *
	 * @param array $data  prepared data.
	 * @param array $_post post data.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function bb_prepare_access_control_settings( $data, $_post ) {

		$settings                    = array();
		$settings['group_condition'] = '';
		$settings['group_ids']       = array();
		$settings['rule_condition']  = ! empty( $data['rule_condition'] ) ? $data['rule_condition'] : '';
		if ( ! empty( $settings['rule_condition'] ) && 'members-in-specific-groups' === $settings['rule_condition'] || bbapp_is_private_app_enabled() ) {
			$settings['group_condition'] = ! empty( $data['group_condition'] ) ? $data['group_condition'] : 'all';
			$settings['group_ids']       = ! empty( $data['group_ids'] ) ? array_filter( array_unique( $data['group_ids'] ) ) : array();
		}
		if ( bbapp_is_private_app_enabled() ) {
			$settings['rule_condition'] = 'members-in-specific-groups';
		}

		$restricted_message['source'] = ! empty( $data['restricted_message']['source'] ) ? $data['restricted_message']['source'] : 'default';
		if ( 'custom' === $restricted_message['source'] ) {
			$restricted_message_content = array(
				'title'         => ! empty( $data['restricted_message']['content']['title'] ) ? $data['restricted_message']['content']['title'] : '',
				'content'       => ! empty( $data['restricted_message']['content']['content'] ) ? $data['restricted_message']['content']['content'] : '',
				'button_action' => ! empty( $data['restricted_message']['content']['button_action'] ) ? $data['restricted_message']['content']['button_action'] : 'no_button',
				'show_teaser'   => ! empty( $data['restricted_message']['content']['show_teaser'] ) ? $data['restricted_message']['content']['show_teaser'] : '',
			);

			if ( 'no_button' !== $restricted_message_content['button_action'] ) {
				$restricted_message_content['button_text']    = ! empty( $data['restricted_message']['content']['button_text'] ) ? $data['restricted_message']['content']['button_text'] : '';
				$restricted_message_content['app_page_id']    = ! empty( $_post['app_page_id'] ) ? $_post['app_page_id'] : 0;
				$restricted_message_content['iap_product_id'] = ! empty( $_post['iap_product_id'] ) ? $_post['iap_product_id'] : 0;
			}
			$restricted_message['content'] = $restricted_message_content;
		}
		$settings['data']['restricted_message'] = $restricted_message;

		return $settings;
	}

	/**
	 * Function to check rule set or not.
	 *
	 * @since 1.5.2.1
	 */
	public function check_rule_exists() {
		$id     = filter_input( INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT );
		$result = array(
			'is_rule_set' => false,
		);

		if ( ! empty( AccessRule::instance()->get_access_rule( $id, 'app_page' ) ) ) {
			$result['is_rule_set'] = true;
		}

		wp_send_json_success( $result );
	}
}
