<?php
/**
 * Holds access group memberlist table HTML.
 *
 * @package BuddyBossApp\Admin\AccessControls\AccessGroups
 */

namespace BuddyBossApp\Admin\AccessControls\AccessGroups;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Access Group Members List class.
 */
class MembersList extends \WP_List_Table {

	/**
	 * Class constructor
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {

		parent::__construct(
			array(
				// singular name of the listed records.
				'singular' => _x( 'Access Group Member', 'bb-access-control', 'buddyboss-app' ),
				// plural name of the listed records.
				'plural'   => _x( 'Access Group Members', 'bb-access-control', 'buddyboss-app' ),
				// should this table support ajax?
				'ajax'     => true,
				'screen'   => 'access-group-members',
			)
		);
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'name' => _x( 'Name', 'bb-access-control', 'buddyboss-app' ),
		);
	}

	/**
	 * No groups found
	 *
	 * @since 1.5.2.1
	 */
	public function no_items() {
		esc_html_e( 'No access group members found.', 'buddyboss-app' );
	}

	/**
	 * Access Groups column default values.
	 *
	 * @param object $item        Group item.
	 * @param string $column_name Column name.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {

		switch ( $column_name ) {
			case 'name':
				$user_link = add_query_arg(
					array(
						'page'    => 'bbapp-user-access-groups',
						'user_id' => $item->user_id,
					),
					self_admin_url( 'users.php' )
				);
				printf( '<a href="%1$s" target="_blank" data-id="%2$s">%3$s<strong>%4$s</strong></a>', esc_url( $user_link ), esc_attr( $item->user_id ), get_avatar( $item->user_id, 40 ), esc_html( bbaap_get_user_display_name( $item->user_id ) ) );
				break;
			default:
				echo 'N/A';
				break;
		}
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 *
	 * @since 1.5.2.1
	 */
	public function prepare_items() {
		$group_id              = filter_input( INPUT_POST, 'group_id', FILTER_SANITIZE_NUMBER_INT );
		$group_id              = ( ! empty( $group_id ) ) ? $group_id : 0;
		$this->_column_headers = array(
			$this->get_columns(),
			array(),
			$this->get_sortable_columns(),
		); // Init column headers.

		/**
		 * Set per page.
		 *
		 * @type int Members per page.
		 *
		 * @since 1.5.2.1
		 */
		$per_page     = apply_filters( 'bb_access_controls_groups_per_page', 10 );
		$current_page = $this->get_pagenum();
		$search       = ! empty( $_REQUEST['s'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$group_members = bb_access_controls_get_group_members(
			array(
				'per_page'      => $per_page,
				'page'          => $current_page,
				'group_include' => $group_id,
				'search'        => $search,
			)
		);

		$this->items = $group_members['result'];
		$total_items = $group_members['count'];
		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);
	}

	/**
	 * Displays the search box.
	 *
	 * @param string $text     The 'submit' button label.
	 * @param string $input_id ID attribute value for the search input field.
	 *
	 * @since 3.1.0
	 */
	public function search_box( $text, $input_id ) {
		$search_term = ( ! empty( $_REQUEST['s'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $search_term ) && ! $this->has_items() ) {
			return;
		}

		$page    = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$setting = ( ! empty( $_REQUEST['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$action  = ( ! empty( $_REQUEST['action'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$id      = ( ! empty( $_REQUEST['id'] ) ) ? absint( $_REQUEST['id'] ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( ! empty( $page ) ) {
			echo '<input type="hidden" name="page" value="' . esc_attr( $page ) . '" />';
		}
		if ( ! empty( $setting ) ) {
			echo '<input type="hidden" name="setting" value="' . esc_attr( $setting ) . '" />';
		}
		if ( ! empty( $action ) ) {
			echo '<input type="hidden" name="action" value="' . esc_attr( $action ) . '" />';
		}
		if ( ! empty( $id ) ) {
			echo '<input type="hidden" name="id" value="' . esc_attr( $id ) . '" />';
		}
		?>
		<p class="search-box">
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_html( $text ); ?>:</label>
			<input type="search" placeholder="<?php echo esc_html_x( 'Search Members', 'bb-access-control', 'buddyboss-app' ); ?>" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>"/>
		</p>
		<?php
	}

	/**
	 * Function to override the access group members display.
	 *
	 * @Override display method
	 * @since    1.5.2.1
	 */
	public function display() {
		$singular = $this->_args['singular'];
		/**
		 * Adds a nonce field
		 */
		wp_nonce_field( 'ajax-access-group-member-list-nonce', '_ajax_access_group_member_list_nonce' );

		$this->screen->render_screen_reader_content( 'heading_list' );
		?>
		<table class="wp-list-table <?php echo implode( ' ', array_map( 'esc_attr', $this->get_table_classes() ) ); ?>">
			<thead>
			<tr>
				<?php $this->print_column_headers(); ?>
			</tr>
			</thead>

			<tbody id="the-list"
				<?php
				if ( $singular ) {
					echo " data-wp-lists='list:'" . esc_attr( $singular ) . "'";
				}
				?>
			>
			<?php $this->display_rows_or_placeholder(); ?>
			</tbody>

			<tfoot>
			<tr>
				<?php $this->print_column_headers( false ); ?>
			</tr>
			</tfoot>

		</table>
		<?php
		$this->display_tablenav( 'bottom' );
	}

	/**
	 * Function to override Ajax response of access group members.
	 *
	 * @Override ajax_response method.
	 *
	 * @since    1.5.2.1
	 */
	public function ajax_response() {

		check_ajax_referer( 'ajax-access-group-member-list-nonce', '_ajax_access_group_member_list_nonce' );

		$this->prepare_items();

		$total_items = ( ! empty( $this->_pagination_args['total_items'] ) ) ? $this->_pagination_args['total_items'] : 0;
		$total_pages = ( ! empty( $this->_pagination_args['total_pages'] ) ) ? $this->_pagination_args['total_pages'] : 0;

		ob_start();
		if ( ! empty( $_REQUEST['no_placeholder'] ) ) {
			$this->display_rows();
		} else {
			$this->display_rows_or_placeholder();
		}
		$rows = ob_get_clean();

		ob_start();
		$this->pagination( 'top' );
		$pagination_top = ob_get_clean();

		ob_start();
		$this->pagination( 'bottom' );
		$pagination_bottom = ob_get_clean();

		$response                         = array( 'rows' => $rows );
		$response['pagination']['top']    = $pagination_top;
		$response['pagination']['bottom'] = $pagination_bottom;

		if ( isset( $total_items ) ) {
			/* translators: %s Number of total items. */
			$response['total_items_i18n'] = sprintf( _n( '%s item', '%s items', $total_items, 'buddyboss-app' ), number_format_i18n( $total_items ) );
		}

		if ( isset( $total_pages ) ) {
			$response['total_pages']      = $total_pages;
			$response['total_pages_i18n'] = number_format_i18n( $total_pages );
		}

		die( wp_json_encode( $response ) );
	}
}
