<?php
/**
 * Used by Access Groups List class.
 *
 * @package BuddyBossApp\Admin\AccessControls\AccessGroups
 */

namespace BuddyBossApp\Admin\AccessControls\AccessGroups;

use BuddyBossApp\Admin\AccessControls\AccessGroups;
use BuddyBossApp\Jobs;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Access Group Helper class.
 */
class Helper {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance = null;

	/**
	 * Class construct.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {}

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return Helper
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Initialize actions/filters
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function init() {
	}

	/**
	 * Function to start members recalculation job.
	 *
	 * @param int  $group_id     Access Group ID.
	 * @param bool $re_calculate Start the calculation from start.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function bb_start_members_calculation_job( $group_id, $re_calculate = false ) {

		// Add the queue job to run on background.
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add(
			'bb_ac_members_calc',
			array(
				'group_id'     => $group_id,
				're_calculate' => $re_calculate,
			),
			1
		); // job type allows only 20 char.

		return $bbapp_queue->start();
	}

	/**
	 * Keep the last known status of conditions and changes the groups status based on that.
	 *
	 * @since 1.5.2.1
	 */
	public function enable_disable_groups_based_on_condition_availability() {
		$db_conditions_keys    = get_option( 'bb_access_controls_last_active_conditions' );
		$registered_conditions = bb_access_controls_get_conditions();
		$bb_conditions_keys    = array_keys( $registered_conditions );
		if ( ! empty( $db_conditions_keys ) ) {
			$bb_diff_conditions_keys = array_diff( $bb_conditions_keys, $db_conditions_keys );
			$db_diff_conditions_keys = array_diff( $db_conditions_keys, $bb_conditions_keys );
			if ( ! empty( $bb_diff_conditions_keys ) ) {
				$access_groups = bb_access_controls_get_groups(
					array(
						'include_conditions' => $bb_diff_conditions_keys,
						'status'             => array( 'disabled' ),
					)
				);
				if ( ! empty( $access_groups['result'] ) ) {
					foreach ( $access_groups['result'] as $access_group ) {
						// Only enable group which are not disabled manually or with any other errors then condition not registered.
						if ( isset( $access_group['data']['error_code'] ) && 'condition_not_registered' === $access_group['data']['error_code'] ) {
							bb_access_controls_enable_group( $access_group['id'] );
						}
					}
				}
				update_option( 'bb_access_controls_last_active_conditions', $bb_conditions_keys, true );
			} elseif ( ! empty( $db_diff_conditions_keys ) ) {
				$access_groups = bb_access_controls_get_groups(
					array(
						'include_conditions' => $db_diff_conditions_keys,
						'status'             => array( 'enabled' ),
					)
				);
				if ( ! empty( $access_groups['result'] ) ) {
					foreach ( $access_groups['result'] as $access_group ) {
						bb_access_controls_update_group_error( $access_group['id'], 'condition_not_registered', '' );
						bb_access_controls_disable_group( $access_group['id'] );
					}
				}
				update_option( 'bb_access_controls_last_active_conditions', $bb_conditions_keys, true );
			}
		} else {
			update_option( 'bb_access_controls_last_active_conditions', $bb_conditions_keys, true );
		}
	}

	/**
	 * Function to generate the Group list row.
	 *
	 * @param int     $id      Access group id.
	 * @param boolean $is_edit If action is edit.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string
	 */
	public function prepare_group_table_row( $id, $is_edit = false ) {
		$group_data = bb_access_controls_get_group( $id );
		$html       = '';

		if ( empty( $id ) ) {
			return $html;
		}

		$access_groups_list = new GroupsList();
		$item               = $group_data;
		ob_start();
		if ( $is_edit ) {
			$allowed_html                 = wp_kses_allowed_html( 'post' );
			$allowed_html['a']['onclick'] = true;
			echo wp_kses( $access_groups_list->single_row_columns( $item ), $allowed_html, array( 'javascript' ) );
		} else {
			$access_groups_list->single_row( $item );
		}
		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Function to generate the Group Tab le nav.
	 *
	 * @since 1.5.5
	 *
	 * @return array
	 */
	public function prepare_list_tablenav() {
		$html = array(
			'top'    => '',
			'bottom' => '',
		);

		$access_groups_list = new GroupsList();
		$access_groups_list->prepare_items();
		ob_start();
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		echo wp_kses( $access_groups_list->display_tablenav( 'top' ), $allowed_html, array( 'javascript' ) );
		$html['top'] = ob_get_clean();
		ob_start();
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		echo wp_kses( $access_groups_list->display_tablenav( 'bottom' ), $allowed_html, array( 'javascript' ) );
		$html['bottom'] = ob_get_clean();

		return $html;
	}

	/**
	 * Function to validate Access Group member data.
	 *
	 * @param array $data_to_validate Group data to validate.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function validate_prepare_group_member_data( $data_to_validate ) {
		$result             = array();
		$result['validate'] = true;
		$result['data']     = array();
		$group_id           = filter_var( $data_to_validate['group_id'], FILTER_SANITIZE_NUMBER_INT );
		$user_id            = filter_var( $data_to_validate['user_id'], FILTER_SANITIZE_NUMBER_INT );
		$date_added         = ! empty( $data_to_validate['date_added'] ) ? $data_to_validate['date_added'] : '';
		$status             = filter_var( $data_to_validate['status'], FILTER_SANITIZE_NUMBER_INT );

		if ( empty( $group_id ) ) {
			$result['validate']                  = false;
			AccessGroups::instance()->messages[] = array(
				'type'    => 'error',
				'message' => __( 'Please enter the group id.', 'buddyboss-app' ),
			);
		}

		if ( empty( $user_id ) ) {
			$result['validate']                  = false;
			AccessGroups::instance()->messages[] = array(
				'type'    => 'error',
				'message' => __( 'Please select the group member id.', 'buddyboss-app' ),
			);
		}

		if ( true === $result['validate'] ) {
			$result['data'] = array(
				'group_id' => $group_id,
				'user_id'  => $user_id,
				'status'   => $status,
			);
			if ( ! empty( $date_added ) ) {
				$result['data']['date_added'] = $date_added;
			}
		}

		return $result;
	}

	/**
	 * Prepare item condition.
	 *
	 * @param object|array $group         Item data.
	 * @param string       $actions_html wp list table action html.
	 * @param boolean      $is_arrow Is arrow sign.
	 *
	 * @return string
	 * @since 1.5.2.1
	 */
	public function get_condition_name( $group, $actions_html = '', $is_arrow = false ) {

		$html = '';
		ob_start();
		if ( isset( $group['condition_name'] ) ) {

			$group_name = ! empty( $is_arrow ) ? $group['name_rich'] : $group['name_sign'];
			?>
			<div class="bbapp-condition-group">
				<?php
				if ( 'disabled' !== $group['item_value_status'] ) {
					if ( ! ( 'deleted' === $group['item_value_status'] && 'any' !== $group['sub_condition_name'] ) ) {
						echo wp_kses_post( $group_name );
					} else {
						echo sprintf( '<div><sapn class="bbapp-deleted-condition">%s</sapn></div>', wp_kses_post( $group['name_rich'] ) );
					}
				} else {
					printf( '<div><sapn class="bbapp-deleted-condition">%s</sapn></div>', wp_kses_post( $group_name ) );
				}
				$allowed_html                 = wp_kses_allowed_html( 'post' );
				$allowed_html['a']['onclick'] = true;
				echo wp_kses( $actions_html, $allowed_html, array( 'javascript' ) );
				?>
			</div>
			<?php
		}
		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Get group detail for tool screen.
	 *
	 * @param array $group Group data.
	 *
	 * @return false|string
	 */
	public function get_tool_list_group_job_entry( $group ) {
		ob_start();
		include bbapp()->plugin_dir . 'views/access-controls/tools/calculate-members-single.php';
		$content = ob_get_contents();
		ob_clean();

		return $content;
	}

}
