<?php
/**
 * Used by Access Groups List class.
 *
 * @package BuddyBossApp\Admin\AccessControls\AccessGroups
 */

namespace BuddyBossApp\Admin\AccessControls\AccessGroups;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Access Groups List class.
 */
class GroupsList extends \WP_List_Table {

	/**
	 * Current user id.
	 *
	 * @var $user_id
	 */
	public $user_id;

	/**
	 * The page is edit user's profile page.
	 *
	 * @var bool
	 */
	public $is_profile_page = false;

	/**
	 * Class constructor
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {

		parent::__construct(
			array(
				// singular name of the listed records.
				'singular' => _x( 'Access Group', 'access control groups list', 'buddyboss-app' ),
				// plural name of the listed records.
				'plural'   => _x( 'Access Groups', 'access control groups list', 'buddyboss-app' ),
				// should this table support ajax?
				'ajax'     => false,
				'screen'   => 'access-group-list',
			)
		);

		$page = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Is the user attempting to edit profile page.
		if ( ! empty( $page ) && 'bbapp-user-access-groups' === $page ) {
			$this->is_profile_page = true;
		}
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 *
	 * @since 1.5.2.1
	 */
	public function prepare_items() {

		/**
		 * Init column headers.
		 */
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		/** Process bulk action */
		$this->process_bulk_action();

		$per_page     = $this->get_items_per_page( 'bb_access_controls_groups_per_page', 10 );
		$current_page = $this->get_pagenum();

		$access_groups_args = array(
			'per_page'        => $per_page,
			'page'            => $current_page,
			'is_profile_page' => $this->is_profile_page,
		);

		if ( $this->is_profile_page ) {
			$access_groups_args['include_members'] = array( $this->get_user_id() );
		}

		$access_groups = bb_access_controls_get_groups( $access_groups_args );
		$this->items   = $access_groups['result'];
		$total_items   = $access_groups['count'];

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);

	}

	/**
	 * Return the column available to this table list.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'cb'      => '<input type="checkbox" />',
			'name'    => _x( 'Group', 'access control groups list', 'buddyboss-app' ),
			'members' => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Members', 'access control groups list', 'buddyboss-app' ) ),
			'status'  => sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Status', 'access control groups list', 'buddyboss-app' ) ),
		);

		// Remove unwanted columns of access groups list on profile page.
		if ( $this->is_profile_page ) {
			unset( $columns['cb'] );
			unset( $columns['members'] );
			unset( $columns['status'] );
			$columns['name'] = _x( 'Access Group', 'access control groups list', 'buddyboss-app' );
		}

		return $columns;
	}

	/**
	 * No groups found
	 *
	 * @since 1.5.2.1
	 */
	public function no_items() {
		esc_html_e( 'No access groups found.', 'buddyboss-app' );
	}

	/**
	 * Access Groups column default values.
	 *
	 * @param object $item        Group item.
	 * @param string $column_name Column name.
	 *
	 * @since 1.5.2.1
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'members':
				$members_count = '<span class="no-group-member">-</span>';
				if ( 1 === (int) $item['status'] && 0 !== (int) $item['members_count'] ) {
					$members_count = sprintf( '<a href="javascript:void(0)" class="bbapp-access-group-members" data-group_id="%1$s"> %2$s </a>', $item['id'], $item['members_count'] );
				} elseif ( 2 === (int) $item['status'] && isset( $item['data'] ) ) {
					$access_group_added_members = bb_access_controls_get_group_members(
						array(
							'group_include' => (int) $item['id'],
							'only_count'    => true,
							'status'        => array( 'enabled' ),
						)
					);

					if ( true === (bool) $item['is_calculating'] && isset( $access_group_added_members['count'] ) ) {
						$members_count = $access_group_added_members['count'];
					}
				}
				$allowed_html                 = wp_kses_allowed_html( 'post' );
				$allowed_html['a']['onclick'] = true;
				if ( ! empty( $members_count ) ) {
					echo wp_kses( $members_count, $allowed_html, array( 'javascript' ) );
				}
				break;
			case 'status':
				if ( 1 === (int) $item['status'] ) {
					echo sprintf( "<span class='bb-status bbapp-success-msg'>%s</span>", esc_html_x( 'Enabled', 'access control groups list', 'buddyboss-app' ) );
				} elseif ( 2 === (int) $item['status'] && isset( $item['data'] ) ) {
					if ( true === (bool) $item['data']['is_creating'] ) {
						echo sprintf(
							'<span class="bb-status calculating_member">%1$s</span>',
							esc_html_x( 'Creating group...', 'access control groups list', 'buddyboss-app' )
						);
					} elseif ( true === (bool) $item['data']['is_updating'] ) {
						echo sprintf(
							'<span class="bb-status calculating_member">%1$s</span>',
							esc_html_x( 'Updating group...', 'access control groups list', 'buddyboss-app' )
						);
					} elseif ( true === (bool) $item['data']['is_enabling'] ) {
						echo sprintf(
							'<span class="bb-status calculating_member">%1$s</span>',
							esc_html_x( 'Enabling group...', 'access control groups list', 'buddyboss-app' )
						);
					} elseif ( true === (bool) $item['is_calculating'] ) {
						echo sprintf(
							'<span class="bb-status bbapp-success-msg">%1$s</span><br><span class="calculating_member">%2$s</span>',
							esc_html_x( 'Enabled', 'access control groups list', 'buddyboss-app' ),
							esc_html_x( 'Recalculating members...', 'access control groups list', 'buddyboss-app' )
						);
					} else {
						echo sprintf( "<span class='bb-status'>%s</span>", esc_html_x( 'Unknown error', 'access control groups list', 'buddyboss-app' ) );
					}
				} elseif ( 3 === (int) $item['status'] ) {
					echo sprintf( "<span class='bb-status bbapp-error-msg'>%s</span>", esc_html_x( 'Disabled', 'access control groups list', 'buddyboss-app' ) );
					if ( ! empty( $item['data'] ) && ! empty( $item['data']['error_code'] ) && in_array(
						$item['data']['error_code'],
						array(
							'creating_error',
							'enabling_error',
							'recalculating_error',
						),
						true
					) ) {
						if ( ! empty( $item['data']['error_code'] ) ) {
							if ( 'creating_error' === $item['data']['error_code'] ) {
								echo sprintf(
									'<p class="description">%1$s %2$s</p>',
									esc_html__( 'Due to error when creating group', 'buddyboss-app' ),
									wp_kses_post( bbapp_custom_tooltip( $item['data']['error_msg'] ) )
								);
							} elseif ( 'enabling_error' === $item['data']['error_code'] ) {
								echo sprintf(
									'<p class="description">%1$s %2$s</p>',
									esc_html__( 'Due to error when enabling group', 'buddyboss-app' ),
									wp_kses_post( bbapp_custom_tooltip( $item['data']['error_msg'] ) )
								);
							} elseif ( 'recalculating_error' === $item['data']['error_code'] ) {
								echo sprintf(
									'<p class="description">%1$s %2$s</p>',
									esc_html__( 'Due to error during recalculation', 'buddyboss-app' ),
									wp_kses_post( bbapp_custom_tooltip( $item['data']['error_msg'] ) )
								);
							}
						}
					} elseif ( isset( $item['item_value_status'] ) ) {
						if ( 'disabled' === $item['item_value_status'] ) {
							echo sprintf( '<p class="description">%1$s</p>', esc_html__( 'Integration is disabled', 'buddyboss-app' ) );
						} elseif ( 'deleted' === $item['item_value_status'] ) {
							echo sprintf( '<p class="description">%1$s</p>', esc_html( $item['data']['error_msg'] ) );
						}
					}
				}
				break;
			default:
				echo 'N/A';
				break;
		}

	}

	/**
	 * Function to show content condition.
	 *
	 * @param object|array $item loop item.
	 *
	 * @since 1.5.2.1
	 */
	public function column_name( $item ) {
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		$actions                      = array();

		$actions['edit']                = sprintf( '<a href="javascript:void(0)" class="bbapp-edit-access-group" data-group_id="%1$s" title="%2$s"> %3$s </a>', $item['id'], esc_attr_x( 'Edit Access Group', 'access control groups list', 'buddyboss-app' ), esc_html_x( 'Edit', 'bb-access-control', 'buddyboss-app' ) );
		$actions['recalculate-members'] = sprintf( '<a href="javascript:void(0)" class="bbapp-recalculate-members" data-group_id="%1$s" title="%2$s"> %3$s </a>', $item['id'], esc_attr_x( 'Recalculate Groups Members', 'access control groups list', 'buddyboss-app' ), esc_html_x( 'Recalculate Members', 'bb-access-control', 'buddyboss-app' ) );
		$actions['enable_group']        = sprintf( '<a href="javascript:void(0)" class="bbapp-enable-group" data-group_id="%1$s" title="%2$s"> %3$s </a>', $item['id'], esc_attr_x( 'Enable', 'access control groups list', 'buddyboss-app' ), esc_html_x( 'Enable', 'bb-access-control', 'buddyboss-app' ) );
		$actions['disable_group']       = sprintf( '<a href="javascript:void(0)" class="bbapp-disable-group" data-group_id="%1$s" title="%2$s"> %3$s </a>', $item['id'], esc_attr_x( 'Disable', 'access control groups list', 'buddyboss-app' ), esc_html_x( 'Disable', 'bb-access-control', 'buddyboss-app' ) );
		$actions['delete']              = sprintf( '<a href="javascript:void(0)" class="bbapp-delete-group submitdelete"  data-group_id="%1$s" data-nonce="%2$s">%3$s</a>', $item['id'], wp_create_nonce( 'acc-delete-group' ), esc_html_x( 'Delete', 'access control groups list', 'buddyboss-app' ) );

		if ( 2 === (int) $item['status'] && isset( $item['data'] ) ) {
			if ( isset( $item['data']['is_creating'] ) && true === (bool) $item['data']['is_creating'] ) {
				unset( $actions['enable_group'] );
				unset( $actions['disable_group'] );
				unset( $actions['edit'] );
			} elseif ( isset( $item['data']['is_enabling'] ) && true === (bool) $item['data']['is_enabling'] ) {
				unset( $actions['edit'] );
			} elseif ( isset( $item['data']['is_updating'] ) && true === (bool) $item['data']['is_updating'] ) {
				unset( $actions['edit'] );
			}
			unset( $actions['enable_group'] );
			unset( $actions['recalculate-members'] );
		} elseif ( 1 === (int) $item['status'] ) {
			unset( $actions['enable_group'] );
		} elseif ( 3 === (int) $item['status'] ) {
			unset( $actions['recalculate-members'] );
			unset( $actions['disable_group'] );
			unset( $actions['edit'] );
		}

		if ( isset( $item['item_value_status'] ) ) {
			if ( 'disabled' !== $item['item_value_status'] ) {
				if ( 'deleted' === $item['item_value_status'] && 'any' !== $item['sub_condition_name'] ) {
					unset( $actions['edit'] );
					unset( $actions['enable_group'] );
					unset( $actions['recalculate-members'] );
				}
			} else {
				unset( $actions['edit'] );
				unset( $actions['enable_group'] );
				unset( $actions['recalculate-members'] );
			}
		}

		// Remove not necessary actions on single user profile page.
		if ( $this->is_profile_page ) {
			unset( $actions['edit'] );
			unset( $actions['recalculate-members'] );
			unset( $actions['enable_group'] );
			unset( $actions['disable_group'] );
			unset( $actions['delete'] );
		}

		// Prepare item condition.
		echo wp_kses( Helper::instance()->get_condition_name( $item, $this->row_actions( $actions ), true ), $allowed_html, array( 'javascript' ) );

	}

	/**
	 * Function to change layout for the WP List table.
	 *
	 * @param object $item List table item.
	 *
	 * @since 1.5.2.1
	 */
	public function single_row( $item ) {
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		echo "<tr data-group-id='" . esc_attr( $item['id'] ) . "'>";

		ob_start();
		$this->single_row_columns( $item );
		$row_content = ob_get_clean();
		echo wp_kses( $row_content, $allowed_html, array( 'javascript' ) );
		echo "</tr>\n";
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item List table item.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		if ( current_user_can( 'manage_options' ) && ! $item['is_calculating'] ) {
			return sprintf( '<input type="checkbox" name="access_group_ids[]" value="%s" />', $item['id'] );
		} else {
			return '<input type="checkbox" name="" disabled value="" />';
		}
	}

	/**
	 * Bulk action.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		$actions = array();

		if ( $this->is_profile_page ) {
			return $actions;
		}

		$actions['bulk-recalculate-members'] = _x( 'Recalculate Members', 'access control groups list', 'buddyboss-app' );
		$actions['bulk-delete']              = _x( 'Delete', 'access control groups list', 'buddyboss-app' );

		return $actions;
	}

	/**
	 * Bulk action process.
	 *
	 * @since 1.5.2.1
	 */
	public function process_bulk_action() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'You don\'t have permission to access this page.', 'buddyboss-app' );
		}

		$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

		// Build redirection URL.
		$redirect_to = remove_query_arg(
			array(
				'action',
				'action2',
				'access_group_ids',
				'deleted',
				'recalculate-members',
				'error',
				'updated',
				'success_new',
				'error_new',
				'success_modified',
				'error_modified',
				'_wpnonce',
				'_wp_http_referer',
				'paged',
			),
			$request_url
		);

		$redirect_to = add_query_arg( array( 'setting' => 'access-groups' ), $redirect_to );

		$action = $this->current_action();

		switch ( $action ) {
			case 'recalculate-members':
			case 'bulk-recalculate-members':
				if ( ! empty( $_REQUEST['access_group_ids'] ) ) {
					$nonce_key = ( 'bulk-recalculate-members' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'acc-recalculate-members';
					check_admin_referer( $nonce_key );
					$acc_group_ids = bbapp_input_clean( wp_parse_id_list( $_REQUEST['access_group_ids'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
					$count         = 0;
					foreach ( $acc_group_ids as $acc_group_id ) {
						if ( $this->recalculate_members_access_group( $acc_group_id ) ) {
							$count ++;
						}
					}
					$redirect_to = add_query_arg(
						array(
							'recalculate-members' => $count,
							'ids'                 => $acc_group_ids,
						),
						$redirect_to
					);
					wp_safe_redirect( $redirect_to );
					exit();

				}
				break;
			case 'delete':
			case 'bulk-delete':
				if ( ! empty( $_REQUEST['access_group_ids'] ) ) {
					$nonce_key = ( 'bulk-delete' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'acc-delete-group';
					check_admin_referer( $nonce_key );
					$acc_group_ids = bbapp_input_clean( wp_parse_id_list( $_REQUEST['access_group_ids'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
					$count         = 0;
					foreach ( $acc_group_ids as $acc_group_id ) {
						if ( $this->delete_access_group( $acc_group_id ) ) {
							$count ++;
						}
					}
					$redirect_to = add_query_arg(
						array(
							'deleted' => $count,
							'ids'     => $acc_group_ids,
						),
						$redirect_to
					);
					wp_safe_redirect( $redirect_to );
					exit();

				}
				break;
		}
	}

	/**
	 * Function to delete the access group.
	 *
	 * @param int $group_id Access group id.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool|int
	 */
	public function delete_access_group( $group_id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return bb_access_controls_delete_group( $group_id );
	}

	/**
	 * Start the recalculation of members of access group.
	 *
	 * @param int $group_id Access group id.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool|int
	 */
	public function recalculate_members_access_group( $group_id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return bb_access_controls_recalculate_group_members( $group_id );
	}

	/**
	 * Get the user ID.
	 *
	 * Look for $_GET['user_id']. If anything else, force the user ID to the
	 * current user's ID so they aren't left without a user to edit.
	 *
	 * @since 1.5.2.1
	 *
	 * @return int
	 */
	private function get_user_id() {
		if ( ! empty( $this->user_id ) ) {
			return $this->user_id;
		}

		$this->user_id = (int) get_current_user_id();

		// We'll need a user ID when not on self profile.
		if ( ! empty( $_GET['user_id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->user_id = (int) $_GET['user_id']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		return $this->user_id;
	}
}
