<?php
/**
 * Used by Access Group functions.
 *
 * @package BuddyBossApp\Admin\AccessControls
 */

namespace BuddyBossApp\Admin\AccessControls;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\AccessControls\Integration_Abstract;
use BuddyBossApp\Admin\AccessControls;
use BuddyBossApp\Admin\AccessControls\AccessGroups\Helper;
use BuddyBossApp\Admin\AccessControls\AccessGroups\MembersList;

/**
 * Access Groups class.
 */
class AccessGroups {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Group list variable.
	 *
	 * @var $groups_list
	 */
	public $groups_list;

	/**
	 * Error|Success notice.
	 *
	 * @var $messages
	 */
	public $messages = array();

	/**
	 * Condition item per page.
	 *
	 * @var int $conditions_item_per_page
	 */
	public $conditions_item_per_page = 20;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return AccessGroups
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Groups constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {

		add_action( 'admin_footer', array( $this, 'admin_footer' ) );
		// Background Job.
		add_action( 'bbapp_queue_task_bb_ac_members_calc', array( $this, 'members_calculation_job' ), 10, 1 );

		// Get the conditions or membership dropdown.
		add_action( 'wp_ajax_bb_access_controls_update_access_group_data', array( $this, 'update_access_group_data' ) );

		// Get the conditions or membership dropdown.
		add_action( 'wp_ajax_bb_access_controls_get_conditions', array( $this, 'bb_get_conditions' ) );
		add_action(
			'wp_ajax_bb_access_controls_integration_items_result',
			array(
				$this,
				'ajax_get_integration_items_result',
			)
		);

		add_action( 'wp_ajax_bb_access_controls_group_template', array( $this, 'ajax_group_edit_template' ) );
		add_action( 'wp_ajax_bb_access_controls_group_members_template', array( $this, 'ajax_group_edit_template' ) );

		// Delete access group.
		add_action( 'wp_ajax_bb_access_controls_delete_group', array( $this, 'ajax_bb_access_controls_delete_group' ) );

		// Ajax get group members.
		add_action( 'wp_ajax_bb_access_controls_group_members', array( $this, 'ajax_group_members_callback' ) );

		// Tools recalculating group members.
		add_action(
			'wp_ajax_bb_access_controls_recalculate_group_members',
			array(
				$this,
				'ajax_tool_recalculate_group_members',
			)
		);
		add_action( 'wp_ajax_bb_access_controls_group_action', array( $this, 'ajax_group_action' ) );

		// Return the groups status.
		add_action( 'wp_ajax_bb_access_controls_get_groups_status', array( $this, 'ajax_get_groups_status' ) );

		Helper::instance(); // Group helper.

		if ( 'access-groups' === AccessControls::instance()->get_current_setting() ) {
			$this->groups_list = new AccessControls\AccessGroups\GroupsList();
			$this->groups_list->prepare_items();
		}
	}

	/**
	 * Add modal html on admin footer.
	 *
	 * @since 1.5.2.1
	 */
	public function admin_footer() {
		include bbapp()->plugin_dir . 'views/access-controls/access-groups/modal.php';
	}

	/**
	 * Render sub tabs.
	 *
	 * @since 1.5.2.1
	 */
	public function render_subtabs() {
		?>
		<div class="flex wrap-access wrap-access--is-list" style="position: relative">
			<div class="bbapp-back-to-groups-wrapper">
				<a class="bbapp-back-to-groups" href="
				<?php
				echo esc_url(
					add_query_arg(
						array(
							'page' => 'bbapp-access-controls',
						),
						bbapp_get_admin_url( 'admin.php' )
					)
				);
				?>
				"><i class="bb-icon-arrow-left"></i>
					<?php echo esc_html_x( 'Back to Access Controls', 'bb-access-control', 'buddyboss-app' ); ?>
				</a>
			</div>
			<h1 class="wp-heading-inline"><?php echo esc_html_x( 'Access Groups', 'bb-access-control', 'buddyboss-app' ); ?></h1>
			<div class="flex align-center bbapp-sub-nav bbapp-sub-nav--exp bbapp-sub-nav--access">
				<a class="button button-primary" id="bbapp-add-access-group" title="<?php echo esc_html_x( 'Add Access Group', 'bb-access-control', 'buddyboss-app' ); ?>" href="javascript:void(0);"><?php echo esc_html_x( 'Add Group', 'Access group', 'buddyboss-app' ); ?></a>
				<a class="button button-secondary bbapp-tutorial-btn" target="_blank" href="<?php echo esc_url( admin_url( 'admin.php?page=bbapp-help&article=125674' ) ); ?>">
					<?php esc_html_e( 'View Tutorial', 'buddyboss-app' ); ?>
				</a>
			</div>
		</div>
		<?php
	}

	/**
	 * Renders the access control screen.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool|mixed
	 */
	public function render() {

		if ( ! current_user_can( 'manage_options' ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		include bbapp()->plugin_dir . 'views/access-controls/access-groups/list.php';
	}

	/**
	 * Output the error or update messages to render.
	 *
	 * @since 1.5.2.1
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) ) {
			?>
			<div id='message' class='bbapp-error'>
				<p><?php echo esc_html_x( 'There was a problem saving your access group:', 'bb-access-control', 'buddyboss-app' ); ?></p>
			</div>
			<?php
			foreach ( $this->messages as $message ) {
				if ( ! empty( $message['message'] ) ) {
					echo sprintf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_html( $message['type'] ), esc_html( $message['message'] ) );
				}
			}
			$this->messages = array(); // Clear message variable to prevent duplicate messages.
		}
	}

	/**
	 * Function to show the group action messages.
	 *
	 * @since 1.5.2.1
	 */
	public function action_notices() {
		$bulk_counts   = array(
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'deleted'   => isset( $_REQUEST['deleted'] ) ? absint( $_REQUEST['deleted'] ) : 0,
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'trashed'   => isset( $_REQUEST['trashed'] ) ? absint( $_REQUEST['trashed'] ) : 0,
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'untrashed' => isset( $_REQUEST['untrashed'] ) ? absint( $_REQUEST['untrashed'] ) : 0,
		);
		$bulk_messages = array(
			/* translators: %s: Number of IAP Products. */
			'deleted'   => _n( '%s group permanently deleted.', '%s groups permanently deleted.', $bulk_counts['deleted'], 'buddyboss-app' ),
			/* translators: %s: Number of pages. */
			'trashed'   => _n( '%s group moved to the Trash.', '%s groups moved to the Trash.', $bulk_counts['trashed'], 'buddyboss-app' ),
			/* translators: %s: Number of pages. */
			'untrashed' => _n( '%s group restored from the Trash.', '%s groups restored from the Trash.', $bulk_counts['untrashed'], 'buddyboss-app' ),
		);
		$bulk_counts   = array_filter( $bulk_counts );
		$messages      = array();

		foreach ( $bulk_counts as $message => $count ) {
			if ( isset( $bulk_messages[ $message ] ) ) {
				$messages[] = sprintf( $bulk_messages[ $message ], number_format_i18n( $count ) );
			}
			if ( 'trashed' === $message && isset( $_REQUEST['ids'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$ids          = ! empty( $_REQUEST['ids'] ) ? bbapp_input_clean( array_map( 'absint', $_REQUEST['ids'] ) ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$request_page = ! empty( $_REQUEST['page'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$base_url     = bbapp_get_admin_url( 'admin.php?page=' . esc_attr( $request_page ) . '&amp;setting=access-groups&amp;access_group_ids=' . implode( ',', $ids ) );
				$messages[]   = '<a href="' . esc_url( wp_nonce_url( $base_url . '&amp;action=untrash', 'acc-untrash-group' ) ) . '">' . _x( 'Undo', 'bb-access-control', 'buddyboss-app' ) . '</a>';
			}

			if ( 'untrashed' === $message && isset( $_REQUEST['ids'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$ids          = ! empty( $_REQUEST['ids'] ) ? bbapp_input_clean( array_map( 'absint', $_REQUEST['ids'] ) ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$request_page = ! empty( $_REQUEST['page'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				if ( 1 === count( $ids ) ) {
					$messages[] = sprintf(
						'<a href="%1$s">%2$s</a>',
						esc_url( bbapp_get_admin_url( sprintf( 'admin.php?page=%s&setting=access-groups&action=%s&id=%s', $request_page, 'edit', $ids[0] ) ) ),
						esc_html_x( 'Edit', 'bb-access-control', 'buddyboss-app' )
					);
				}
			}
		}

		if ( empty( $messages ) ) {
			if ( isset( $_REQUEST['created'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = __( 'The group has been created successfully.', 'buddyboss-app' );
			}
			if ( isset( $_REQUEST['updated'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = __( 'The group has been updated successfully.', 'buddyboss-app' );
			}
		}
		if ( $messages ) {
			echo sprintf( '<div id="message" class="updated notice is-dismissible"><p>%s</p></div>', esc_html( implode( ' ', $messages ) ) );
		}
	}

	/**
	 * Function run when run the background job for processing members.
	 *
	 * @param object $task BG job task name.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function members_calculation_job( $task ) {

		$task_data    = maybe_unserialize( $task->data );
		$group_id     = $task_data['group_id'];
		$re_calculate = isset( $task_data['re_calculate'] ) ? $task_data['re_calculate'] : false; // Initiate the recalculations.
		$access_group = bb_access_controls_get_group( $group_id );

		if ( ! empty( $access_group ) ) {
			$group_condition_name     = $access_group['condition_name'];
			$group_sub_condition_name = $access_group['sub_condition_name'];
			$group_item_value         = $access_group['item_value'];
			$group_bg_data            = ! empty( $access_group['data']['background_job'] ) ? $access_group['data']['background_job'] : array();
			$current_page             = ! empty( $group_bg_data['current_page'] ) ? $group_bg_data['current_page'] : 1;
			$per_page                 = ! empty( $group_bg_data['per_page'] ) ? $group_bg_data['per_page'] : 10;

			/**
			 * If recalculation is requested reset the params.
			 */
			if ( $re_calculate ) {
				$group_bg_data['current_page'] = 1;
				$current_page                  = $group_bg_data['current_page']; // Change to 1 when doing calculation from start.
				$group_bg_data['status']       = 'running';
				$group_bg_data['rounds_count'] = 0;

				/**
				 * Update access group & it's all member's status to processing & hold their state temporarily.
				 */
				bb_access_controls_update_group_is_calculating( $group_id );
				bb_access_controls_update_all_group_members_status( $group_id );
			}

			$group_bg_data['rounds_count'] ++;

			/**
			 * Get all conditions.
			 *
			 * @var $integration_instance Integration_Abstract
			 */
			$registered_conditions = bb_access_controls_get_conditions();
			if ( ! isset( $registered_conditions[ $group_condition_name ] ) ) {
				return false;
			}

			$condition = $registered_conditions[ $group_condition_name ];

			$conditions_members = ( is_callable( $condition['users_callback'] ) ) ? call_user_func_array(
				$condition['users_callback'],
				array(
					array(
						'sub_condition' => $group_sub_condition_name,
						'item_value'    => $group_item_value,
						'group_id'      => $group_id,
						'rounds_count'  => (int) $group_bg_data['rounds_count'],
						// Tell counts of rounds how many times job has executed.
					),
					$current_page,
					$per_page,
				)
			) : false;

			$error_code = 'recalculating_error';

			if ( isset( $access_group['data'] ) ) {
				if ( isset( $access_group['data']['is_creating'] ) && $access_group['data']['is_creating'] ) {
					$error_code = 'creating_error';
				} elseif ( isset( $access_group['data']['is_enabling'] ) && $access_group['data']['is_enabling'] ) {
					$error_code = 'enabling_error';
				} elseif ( isset( $access_group['data']['is_updating'] ) && $access_group['data']['is_updating'] ) {
					$error_code = 'updating_error';
				}
			}

			$update_status_function = function () use ( $group_id, $group_bg_data, $error_code ) {
				$group_bg_data['status'] = 'error';
				$error_msg               = __( 'Error While Calculating Members: Conditions member is not valid type.', 'buddyboss-app' );
				bb_access_controls_update_group_error( $group_id, $error_code, $error_msg );

				/**
				 * This filter fires when the member's calculation job fails.
				 *
				 * @param int   $group_id group id.
				 *
				 * @since 1.5.2.1
				 */
				do_action( 'bb_access_controls_members_calculation_job_failed', $group_id );
			};
			/**
			 * When condition provide response.
			 */
			if ( ! empty( $conditions_members ) ) {
				if ( 'wait' === $conditions_members['type'] ) {
					Helper::instance()->bb_start_members_calculation_job( $group_id ); // start next round.

					bb_access_controls_update_group_data(
						$group_id,
						array(
							'background_job' => $group_bg_data,
						)
					);

					return true;
				} elseif ( 'users' === $conditions_members['type'] ) {

					/**
					 * When members are found.
					 */
					bp_access_controls_group_add_update_members( $group_id, $conditions_members['data'] );
					$current_page ++;
					$group_bg_data['current_page'] = $current_page;

				} elseif ( 'error' === $conditions_members['type'] ) {
					$group_bg_data['status'] = 'error';
					$error_msg               = sprintf( '%1$s %2$s', __( 'Error While Calculating Members: ', 'buddyboss-app' ), $conditions_members['data'] );
					bb_access_controls_update_group_error( $group_id, $error_code, $error_msg );

					/**
					 * This filter fires when the member's calculation job fails.
					 *
					 * @param int   $group_id group id.
					 *
					 * @since 1.5.2.1
					 */
					do_action( 'bb_access_controls_members_calculation_job_failed', $group_id );

					return false;
				} else {
					$update_status_function();
					return false;
				}
			} else {
				$update_status_function();
				return false;
			}

			if ( 'error' === $group_bg_data['status'] ) {
				return false;
			}
			$update_group_data['background_job'] = $group_bg_data;
			/**
			 * When no user has been found means the calculation is completed.
			 */
			if ( empty( $conditions_members['data'] ) ) {
				$update_group_data['background_job']['status'] = 'completed';
				$update_group_data['is_creating']              = false;
				$update_group_data['is_updating']              = false;
				$update_group_data['is_enabling']              = false;
				bb_access_controls_update_group_is_calculating( $group_id, false );
				bb_access_controls_delete_group_members_by_status( $group_id, 2 );
				bb_access_controls_update_group_status( $group_id );
				bb_access_controls_update_group_date_calculated( $group_id );

				/**
				 * This filter fires when the member's calculation job completes successfully.
				 *
				 * @param int $group_id group id.
				 *
				 * @since 1.5.2.1
				 */
				do_action( 'bb_access_controls_members_calculation_job_completed', $group_id );
			} else {
				Helper::instance()->bb_start_members_calculation_job( $group_id ); // start next round.
			}
			bb_access_controls_update_group_data( $group_id, $update_group_data );
		}
	}

	/**
	 * Function to update Access group data.
	 *
	 * @since 1.5.2.1
	 */
	public function update_access_group_data() {
		global $wpdb;

		$result            = array();
		$result['is_edit'] = false;
		$_post             = filter_input_array( INPUT_POST, FILTER_DEFAULT );
		$id                = ! empty( $_post['id'] ) ? $_post['id'] : false;
		$validate          = true;
		$messages          = array();

		/**
		 * Validations.
		 */
		if ( empty( $_post['_wpnonce'] ) || ! wp_verify_nonce( $_post['_wpnonce'], 'bbapp-access-group-edit' ) ) {
			$validate   = false;
			$messages[] = __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' );
		}

		if ( empty( $_post['group_condition'] ) ) {
			$validate   = false;
			$messages[] = __( 'Please select the group condition.', 'buddyboss-app' );
		}

		if ( ! in_array( $_post['sub_condition'], array( 'specific', 'any' ), true ) ) {
			$validate   = false;
			$messages[] = __( 'Please select the group sub condition.', 'buddyboss-app' );
		}

		if ( ! $validate ) {
			$result['messages'] = $messages;
			$result['message']  = __( 'Validation errors.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		if ( empty( $id ) ) {

			$data        = array( 'is_creating' => true );
			$insert_data = array(
				'status'             => 2,
				'condition_name'     => wp_strip_all_tags( $_post['group_condition'] ),
				'sub_condition_name' => wp_strip_all_tags( $_post['sub_condition'] ),
				'item_value'         => $_post['item_value'],
				'date_created'       => gmdate( 'Y-m-d H:i:s' ),
				'data'               => maybe_serialize( $data ),
			);
			$create      = $wpdb->insert( "{$wpdb->prefix}bb_access_groups", $insert_data ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

			if ( empty( $create ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'There is a database error while adding group record.', 'buddyboss-app' ),
					)
				);
			} else {
				$insert_id = $wpdb->insert_id;

				/**
				 * Run the recalculating member job.
				 *
				 * @since 1.5.2.1
				 */
				bb_access_controls_remove_tmp_items_by_condition( wp_strip_all_tags( $_post['group_condition'] ) );
				Helper::instance()->bb_start_members_calculation_job( $insert_id, true );
				$result['message']             = __( 'Access group created successfully.', 'buddyboss-app' );
				$result['_list_entry_html']    = Helper::instance()->prepare_group_table_row( $insert_id ); // html entry for admin access groups list.
				$result['_list_tablenav_html'] = Helper::instance()->prepare_list_tablenav(); // html entry for admin access groups list.
				$result['group']               = bb_access_controls_get_group( $insert_id );

				/**
				 * Allow user to modify the result object of ajax.
				 *
				 * @since 1.5.2.1
				 */
				$result = apply_filters( 'bb_access_controls_group_update_result', $result );

				wp_send_json_success( $result );
			}
		} else {

			$data = array(
				'is_updating' => true,
			);

			$result['is_edit'] = true;

			$update_data = array(
				'condition_name'     => wp_strip_all_tags( $_post['group_condition'] ),
				'sub_condition_name' => wp_strip_all_tags( $_post['sub_condition'] ),
				'item_value'         => $_post['item_value'],
				'data'               => maybe_serialize( $data ),
			);

			$update = $wpdb->update( "{$wpdb->prefix}bb_access_groups", $update_data, array( 'id' => $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			if ( false === $update ) {
				wp_send_json_error(
					array(
						'message' => __( 'There is a database error while updating group record.', 'buddyboss-app' ),
					)
				);
			} else {

				/**
				 * Run the recalculating member job.
				 *
				 * @since 1.5.2.1
				 */
				bb_access_controls_disable_group( $id );
				bb_access_controls_remove_tmp_items_by_condition( wp_strip_all_tags( $_post['group_condition'] ) );
				bb_access_controls_update_group_is_calculating( $id );
				bb_access_controls_update_group_status( $id, 'processing' );
				bb_access_controls_update_all_group_members_status( $id );
				bb_access_controls_start_members_recalculation( $id );
				bb_access_controls_update_group_date_calculated( $id );
				$result['message']          = __( 'Access group created successfully.', 'buddyboss-app' );
				$result['_list_entry_html'] = Helper::instance()->prepare_group_table_row( $id ); // html entry for admin access groups list.
				$result['group']            = bb_access_controls_get_group( $id );

				/**
				 * Allow user to modify the result object of ajax.
				 *
				 * @since 1.5.2.1
				 */
				$result = apply_filters( 'bb_access_controls_group_update_result', $result );

				wp_send_json_success( $result );
			}
		}
	}

	/**
	 * Function to get the relevant dropdown.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_get_conditions() {
		$result = array();
		$nonce  = ( ! empty( $_POST['_wpnonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp-access-group-edit' ) ) {
			$result['message'] = __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$group_id   = filter_input( INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT );
		$group_data = bb_access_controls_get_group( $group_id ); // require by group-condition.php.

		ob_start();
		require bbapp()->plugin_dir . 'views/access-controls/access-groups/sections/group-condition.php';
		$html = ob_get_clean();

		$result['message'] = __( 'Items found.', 'buddyboss-app' );
		$result['html']    = $html;

		wp_send_json_success( $result );
	}

	/**
	 * Function to get the results for the selected conditions
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_get_integration_items_result() {
		$_post          = filter_input_array( INPUT_POST, FILTER_DEFAULT );
		$condition_name = ! empty( $_post['condition_name'] ) ? $_post['condition_name'] : false;
		$results        = array();

		if ( empty( $condition_name ) ) {
			wp_send_json( $results );
		}

		$registered_conditions = bb_access_controls_get_conditions();
		if ( ! isset( $registered_conditions[ $condition_name ] ) ) {
			wp_send_json( $results );
		}

		$condition = $registered_conditions[ $condition_name ];

		if ( ! isset( $condition['items_callback'] ) ) {
			wp_send_json( $results );
		}

		$term = ( ! empty( $_post['term']['term'] ) ) ? sanitize_text_field( $_post['term']['term'] ) : '';
		$page = ( ! empty( $_post['term']['page'] ) ) ? sanitize_text_field( $_post['term']['page'] ) : 1;

		if ( ! is_callable( $condition['items_callback'] ) ) {
			wp_send_json( $results );
		}

		$items = call_user_func_array(
			$condition['items_callback'],
			array(
				$term,
				$page,
				$this->conditions_item_per_page,
			)
		);

		$item_values = array();
		foreach ( $items as $value => $item ) {
			$item_values[] = $value;
		}

		$configured_values = bb_access_controls_filter_condition_items_already_configured( $condition_name, $item_values );

		foreach ( $items as $value => $item ) {
			$is_disabled = false;

			if ( in_array( $value, $configured_values ) ) { //phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
				$is_disabled = true;
			}

			$results[] = array(
				'id'       => $value,
				'text'     => wp_specialchars_decode( $item['name'], ENT_QUOTES ),
				'disabled' => $is_disabled,
			);
		}

		$have_more = true; // tell if there is more items to paginate.

		if ( empty( $results ) || count( $results ) < $this->conditions_item_per_page ) {
			$have_more = false;
		}

		wp_send_json(
			array(
				'results'   => $results,
				'have_more' => $have_more,
			)
		);
	}

	/**
	 * Function to get access control group template.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_group_edit_template() {

		$result            = array();
		$result['message'] = __( 'Something went wrong please try again later.', 'buddyboss-app' );
		$nonce             = ( ! empty( $_POST['nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp_sort_nonce_' . get_current_user_id() ) ) {
			$result['message'] = __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$group_id = filter_input( INPUT_POST, 'group_id', FILTER_SANITIZE_NUMBER_INT );
		ob_start();
		if ( function_exists( 'current_action' ) ) {
			if ( 'wp_ajax_bb_access_controls_group_template' === current_action() ) {
				include bbapp()->plugin_dir . 'views/access-controls/access-groups/group-edit.php';
			}
			if ( 'wp_ajax_bb_access_controls_group_members_template' === current_action() ) {
				include bbapp()->plugin_dir . 'views/access-controls/access-groups/sections/group-members.php';
			}
		}
		$html = ob_get_clean();
		if ( ! empty( $html ) ) {
			$result['message'] = '';
			$result['content'] = $html;
			$result['type']    = 'html';
			wp_send_json_success( $result );
		}

		wp_send_json_error( $result );
		wp_die();
	}

	/**
	 * Ajax function to delete the access group.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_bb_access_controls_delete_group() {
		$result            = array();
		$result['message'] = __( 'Something went wrong please try again later.', 'buddyboss-app' );
		$nonce             = ( ! empty( $_POST['nonce'] ) ) ? wp_unslash( $_POST['nonce'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! current_user_can( 'manage_options' ) || empty( $nonce ) || ! wp_verify_nonce( $nonce, 'acc-delete-group' ) ) {
			$result['message'] = __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$group_id = filter_input( INPUT_POST, 'group_id', FILTER_SANITIZE_NUMBER_INT );

		if ( empty( $group_id ) ) {
			$result['message'] = __( 'Provided access group ID is invalid.', 'buddyboss-app' );
			wp_send_json_error( $result );
		}

		$delete = bb_access_controls_delete_group( $group_id );

		if ( $delete ) {
			$result['message'] = __( 'Group has been deleted successfully.', 'buddyboss-app' );
			wp_send_json_success( $result );
		}

		wp_send_json_error( $result );
	}

	/**
	 * Function to get access group members.
	 *
	 * @since 1.5.2.1
	 */
	public function ajax_group_members_callback() {
		$access_group_members_list = new MembersList();
		$access_group_members_list->ajax_response();
	}

	/**
	 * Processing submit form for recalculate group members.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function ajax_tool_recalculate_group_members() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error();
		}

		check_ajax_referer( 'bbapp_sort_nonce_' . get_current_user_id(), 'sort_nonce' );

		$group_id = filter_input( INPUT_POST, 'group_id', FILTER_SANITIZE_NUMBER_INT );

		if ( empty( $group_id ) ) {
			wp_send_json_error();
		}

		$calculated = bb_access_controls_recalculate_group_members( $group_id );
		if ( is_wp_error( $calculated ) || false === $calculated ) {
			$message = __( 'Unknown error', 'buddyboss-app' );
			if ( is_wp_error( $calculated ) && $calculated->get_error_message() ) {
				$message = $calculated->get_error_message();
			}

			wp_send_json_error( $message );
		}

		$group                    = bb_access_controls_get_group( $group_id );
		$group['data']            = ! empty( $group['data'] ) ? maybe_unserialize( $group['data'] ) : array();
		$group['date_calculated'] = current_time( 'mysql', 1 );
		$group['url']             = add_query_arg(
			array(
				'page'    => 'bbapp-access-controls',
				'setting' => 'access-groups',
				'action'  => 'edit',
				'id'      => (int) $group_id,
			),
			bbapp_get_admin_url( 'admin.php' )
		);

		$access_groups_jobs = bb_access_controls_get_groups(
			array(
				'status'     => array( 'processing' ),
				'only_count' => true,
			)
		);
		$jobs               = $access_groups_jobs['count'];

		$access_group_members       = bb_access_controls_get_group_members(
			array(
				'group_include' => (int) $group_id,
				'only_count'    => true,
			)
		);
		$access_group_members_count = $access_group_members['count'];

		$access_group_added_members       = bb_access_controls_get_group_members(
			array(
				'group_include' => (int) $group_id,
				'only_count'    => true,
				'status'        => array( 'enabled' ),
			)
		);
		$access_group_added_members_count = $access_group_added_members['count'];

		wp_send_json_success(
			array(
				'group'                 => $group,
				'job_status'            => ( 1 === (int) $group['status'] ) ? 'complete' : 'running',
				'total_members'         => $access_group_members_count,
				'total_added'           => $access_group_added_members_count,
				'total_jobs'            => $jobs,
				'_list_entry_html'      => Helper::instance()->prepare_group_table_row( $group_id ),
				'_list_entry_tool_html' => \BuddyBossApp\Admin\AccessControls\AccessGroups\Helper::instance()->get_tool_list_group_job_entry( $group ),
			)
		);
	}

	/**
	 * Processing submit form for recalculate group members.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function ajax_group_action() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error();
		}

		check_ajax_referer( 'bbapp_sort_nonce_' . get_current_user_id(), 'sort_nonce' );

		$group_id     = filter_input( INPUT_POST, 'group_id', FILTER_SANITIZE_NUMBER_INT );
		$group_action = ( ! empty( $_POST['group_action'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['group_action'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $group_id ) ) {
			wp_send_json_error();
		}
		$message = '';
		switch ( $group_action ) {
			case 'enabled':
				$enabling_group = bb_access_controls_enable_group( $group_id );
				if ( is_wp_error( $enabling_group ) || false === $enabling_group ) {
					$message = __( 'Unknown error', 'buddyboss-app' );
					if ( is_wp_error( $enabling_group ) && $enabling_group->get_error_message() ) {
						$message = $enabling_group->get_error_message();
					}

					bb_access_controls_disable_group( $group_id );
				}
				break;
			case 'disabled':
				bb_access_controls_disable_group( $group_id );
				break;
		}
		$group         = bb_access_controls_get_group( $group_id );
		$group['data'] = ! empty( $group['data'] ) ? maybe_unserialize( $group['data'] ) : array();

		wp_send_json_success(
			array(
				'group'            => $group,
				'message'          => $message,
				'_list_entry_html' => Helper::instance()->prepare_group_table_row( $group_id ),
			)
		);
	}


	/**
	 * Function will run after every 30 sec if found any group ids to calculate member.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function ajax_get_groups_status() {
		check_ajax_referer( 'bbapp_sort_nonce_' . get_current_user_id(), 'sort_nonce' );

		$group_ids = ! empty( $_POST['group_id'] ) ? bbapp_input_clean( array_map( 'absint', $_POST['group_id'] ) ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$check     = ! empty( $_POST['check'] ) ? bbapp_input_clean( wp_unslash( $_POST['check'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $group_ids ) ) {
			wp_send_json_error();
		}

		$group_data = array();

		$access_groups_jobs = bb_access_controls_get_groups(
			array(
				'status'     => array( 'processing' ),
				'only_count' => true,
			)
		);
		$jobs               = $access_groups_jobs['count'];

		foreach ( $group_ids as $group_id ) {
			if ( ! $check ) {
				$calculated = bb_access_controls_recalculate_group_members( $group_id );
				if ( is_wp_error( $calculated ) || false === $calculated ) {
					$message = __( 'Unknown error', 'buddyboss-app' );
					if ( is_wp_error( $calculated ) && $calculated->get_error_message() ) {
						$message = $calculated->get_error_message();
					}
					wp_send_json_error( $message );
				}
			}

			$group         = bb_access_controls_get_group( $group_id );
			$group['data'] = ! empty( $group['data'] ) ? maybe_unserialize( $group['data'] ) : array();
			$group['url']  = add_query_arg(
				array(
					'page'    => 'bbapp-access-controls',
					'setting' => 'access-groups',
					'action'  => 'edit',
					'id'      => (int) $group_id,
				),
				bbapp_get_admin_url( 'admin.php' )
			);

			$access_group_members       = bb_access_controls_get_group_members(
				array(
					'group_include' => (int) $group_id,
					'only_count'    => true,
				)
			);
			$access_group_members_count = $access_group_members['count'];

			$access_group_added_members       = bb_access_controls_get_group_members(
				array(
					'group_include' => (int) $group_id,
					'only_count'    => true,
					'status'        => array( 'enabled' ),
				)
			);
			$access_group_added_members_count = $access_group_added_members['count'];

			$group_data[ $group_id ] = array(
				'group'                 => $group,
				'total_members'         => $access_group_members_count,
				'total_added'           => $access_group_added_members_count,
				'job_status'            => ( isset( $group['status'] ) && 1 === (int) $group['status'] ) ? 'complete' : 'running',
				'total_jobs'            => $jobs,
				'_list_entry_html'      => ( ! empty( $group['id'] ) ) ? Helper::instance()->prepare_group_table_row( $group_id ) : '',
				'_list_entry_tool_html' => ( ! empty( $group['id'] ) ) ? \BuddyBossApp\Admin\AccessControls\AccessGroups\Helper::instance()->get_tool_list_group_job_entry( $group ) : '',
			);
		}

		if ( ! empty( $group_data ) ) {
			wp_send_json_success(
				array(
					'group_data' => $group_data,
				)
			);
		}
	}
}
