<?php
/**
 * Holds admin functionality for Access Controls.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\AccessControls\Settings;
use BuddyBossApp\Admin\AccessControls\AccessGroups;
use BuddyBossApp\Admin\AccessControls\AccessRules;

/**
 * AccessControls class.
 */
class AccessControls {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return AccessControls
	 * @since 1.5.2.1
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * AccessControls constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 9 );
	}

	/**
	 * Load Access groups List.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function admin_init() {
		AccessGroups::instance();
		AccessRules::instance();
		Settings::instance();
		// Load and update conditions data.
		AccessGroups\Helper::instance()->enable_disable_groups_based_on_condition_availability();

		// Remove all data from temp table.
		bb_access_controls_remove_all_tmp_items();
	}

	/**
	 * Renders the access control screen.
	 *
	 * @param string $screen Screen name.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function render_screen( $screen = '' ) {

		if ( 'access-groups' === $this->get_current_setting() ) {
			AccessGroups::instance()->render();
		}

		if ( 'access-control' === $this->get_current_setting() ) {
			$this->render();
		}

	}

	/**
	 * Return the sub settings.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function sub_settings() {

		$return = array();

		if ( bbapp_is_admin_page() ) {
			$return['access-control'] = _x( 'Access Controls', 'bb-access-control', 'buddyboss-app' );
			$return['access-groups']  = _x( 'Access Groups', 'bb-access-control-groups', 'buddyboss-app' );
			$return['view-settings']  = _x( 'View Settings', 'bb-access-control-settings', 'buddyboss-app' );
		}

		return $return;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string
	 */
	public function get_current_setting() {

		$default = '';

		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the access control screen.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool|mixed
	 */
	public function render() {

		if ( ! current_user_can( 'manage_options' ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		echo '<div class="wrap">';

		$this->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		AccessRules::instance()->render();

		echo '</div>';

	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {

		$default = 'access-controls';

		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Render the Access rules page.
	 *
	 * @param string $sub_settings Sub settings.
	 * @param string $current_tab  Current tab.
	 * @param string $param_key    Parameter key.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function render_subtabs( $sub_settings, $current_tab, $param_key = 'setting' ) {

		// only render something when there is more than 1 sub menu.
		if ( empty( $sub_settings ) || count( $sub_settings ) < 2 ) {
			return false;
		}

		$page = ! empty( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'access-control' === $current_tab ) {
			$sub_tab       = 'access-groups';
			$tab_title     = $sub_settings['access-control'];
			$sub_tab_title = $sub_settings['access-groups'];
		}
		?>
		<div class="flex" style="position: relative">
			<h1 class="wp-heading-inline"><?php echo esc_html( $tab_title ); ?></h1>
			<div class="flex align-center bbapp-sub-nav bbapp-sub-nav--exp bbapp-sub-nav--access">
				<a class="button button-primary" href="
				<?php
				echo esc_url(
					add_query_arg(
						array(
							'page'     => $page,
							$param_key => $sub_tab,
						),
						bbapp_get_admin_url( 'admin.php' )
					)
				);
				?>
				"><?php echo esc_html( $sub_tab_title ); ?></a>
				<a class="button" href="
				<?php
				echo esc_url(
					add_query_arg(
						array(
							'page'    => 'bbapp-settings',
							'setting' => 'access_controls',
						),
						bbapp_get_admin_url( 'admin.php' )
					)
				);
				?>
				"><?php esc_html_e( 'View Settings', 'buddyboss-app' ); ?></a>
			</div>
		</div>
		<?php
	}
}
