<?php
/**
 * Holds the App Access Control standalone function.
 *
 * @since   1.5.2.1
 * @package BuddyBossApp\AccessControls
 */

use BuddyBossApp\Jobs;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Function to get Access Groups Conditions.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_get_conditions() {
	/**
	 * Note - This filter is for internal usage only.
	 *        Please follow our documentation & utilize the abstract class to extend any functionality.
	 *
	 * @since 1.5.2.1
	 */
	return apply_filters( 'bb_access_controls_get_conditions', BuddyBossApp\AccessControls::instance()->conditions );
}

/**
 * Function to get Access Groups Setting screen.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_get_settings_screen() {
	/**
	 * Note - This filter is for internal usage only.
	 *        Please follow our documentation & utilize the abstract class to extend any functionality.
	 *
	 * @since 1.5.2.1
	 */
	return apply_filters( 'bb_access_controls_get_settings_screen', array() );
}

/**
 * Returns the conditions with their name.
 *
 * @since 1.5.2.1
 *
 * @return array
 */
function bb_access_controls_get_conditions_with_name() {
	$group_conditions = bb_access_controls_get_conditions();
	$conditions       = array();

	foreach ( $group_conditions as $condition_name => $group_condition ) {
		$conditions[ $condition_name ] = ! empty( $group_condition['labels']['condition_name'] ) ? $group_condition['labels']['condition_name'] : $condition_name;
	}

	return $conditions;
}

/**
 * Function to check member exists on provided group.
 *
 * @param int   $group_id Access Group ID.
 * @param array $user_ids Group member id.
 *
 * @since 1.5.2.1
 *
 * @return mixed|void
 */
function bb_access_controls_group_has_members( $group_id, $user_ids ) {
	global $wpdb;

	if ( empty( $user_ids ) ) {
		return array();
	}

	$in_placeholders = implode( ', ', array_fill( 0, count( $user_ids ), '%d' ) );
	$where_clause    = $wpdb->prepare( "user_id IN ({$in_placeholders})", $user_ids ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$exists_members  = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}bb_access_groups_members WHERE group_id = %d AND {$where_clause}", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

	if ( is_wp_error( $exists_members ) ) {
		$exists_members = array();
	}

	$access_group_members = array(
		'add'    => array_diff( $user_ids, wp_list_pluck( $exists_members, 'user_id' ) ), // User ids.
		'update' => wp_list_pluck( $exists_members, 'id' ), // Group member ids.
	);

	/**
	 * Fires after checked a member is exists on access group.
	 *
	 * @param array $access_group_members Access group member.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array (
	 *      'add' get the user ids.
	 *      'update' get the group members ids.
	 * )
	 */
	return apply_filters( 'bb_access_controls_group_has_members', $access_group_members );
}

/**
 * Function to update group member status.
 *
 * @param int $group_id Access group id.
 * @param int $status   Group member status.
 *
 * @since 1.5.2.1
 *
 * @return bool|int|resource|WP_Error
 */
function bb_access_controls_update_all_group_members_status( $group_id, $status = 2 ) {
	global $wpdb;

	$update = $wpdb->update( "{$wpdb->prefix}bb_access_groups_members", array( 'status' => $status ), array( 'group_id' => $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( ! $update ) {
		return new WP_Error( 'error_group_update', __( 'There is a database error while updating group member record.', 'buddyboss-app' ) );
	}

	return $update;
}

/**
 * Function to delete group member status.
 *
 * @param int    $group_id Access Group id.
 * @param string $status   Member status.
 *
 * @since 1.5.2.1
 *
 * @return bool|int|resource|WP_Error
 */
function bb_access_controls_delete_group_members_by_status( $group_id, $status = 'all' ) {
	global $wpdb;

	$delete_args = array(
		'group_id' => $group_id,
	);

	if ( 'all' !== $status ) {
		$delete_args['status'] = $status;
	}

	$delete = $wpdb->delete( "{$wpdb->prefix}bb_access_groups_members", $delete_args );  //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( ! $delete ) {
		return new WP_Error( 'error_group_update', __( 'There is a database error while deleting group member record.', 'buddyboss-app' ) );
	}

	return $delete;
}

/**
 * Function to get specific access group.
 *
 * @param int $group_id Access group id.
 *
 * @since 1.5.2.1
 *
 * @return array|boolean
 */
function bb_access_controls_get_group( $group_id ) {
	$group_data = array();

	if ( empty( $group_id ) ) {
		return $group_data;
	}

	$group = bb_access_controls_get_groups( array( 'include' => $group_id ) );

	return ! empty( $group['result'][0] ) ? $group['result'][0] : false;
}

/**
 * Function to get groups based on conditions.
 *
 * @param array   $args                   {.
 *
 * @type string   $status                 Get the group based on the status.
 * @type int|bool $per_page               Number of results per page.
 *                                        Default: false.
 * @type int      $page                   Which page of results to fetch. Using page=1 without per_page will result
 *                                        in no pagination.
 *                                        Default: 1.
 * @type string   $order_by               Column to order results by.
 * @type string   $order                  ASC or DESC.
 *                                        Default: 'DESC'.
 * @type array    $include                Array of group ids OR comma separated group ids to limit query by (IN).
 *                                        Default: false.
 * @type array    $exclude                Array of group ids OR comma separated group ids to limit query by (NOT IN).
 *                                        Default: false.
 * @type array    $include_conditions     Array of condition names OR comma separated condition names to limit query by
 *                                        (IN).
 *                                        Default: false.
 * @type array    $include_sub_conditions Array of sub condition names OR comma separated condition names to limit
 *                                        query by (IN).
 *                                        Default: false.
 * @type array    $exclude_conditions     Array of condition names OR comma separated condition names to limit query by
 *                                        (NOT IN).
 *                                        Default: false.
 * @type array    $exclude_sub_conditions Array of sub condition names OR comma separated condition names to limit
 *                                        query by (NOT IN).
 *                                        Default: false.
 * @type array    $include_members        Array of member ids OR comma separated member ids to limit
 *                                        query by (IN).
 *                                        Default: false.
 * @type array    $exclude_members        Array of member ids OR comma separated member ids to limit
 *                                        query by (NOT IN).
 *                                        Default: false.
 * @type array    $only_count             Whether only want count.
 *                                        Default: false.
 *                                        }
 *
 * @since 1.5.2.1
 *
 * @return null|array|object|stdClass[]
 */
function bb_access_controls_get_groups( $args = array() ) {
	global $wpdb;

	$default_args = array(
		'status'                 => array( 'enabled', 'processing', 'disabled' ),
		'per_page'               => false,
		'page'                   => 1,
		'orderby'                => 'id',
		'order'                  => 'desc',
		'include'                => false,
		'exclude'                => false,
		'include_conditions'     => false,
		'include_sub_conditions' => false,
		'exclude_conditions'     => false,
		'exclude_sub_conditions' => false,
		'include_members'        => false,
		'exclude_members'        => false,
		'only_count'             => false,
	);

	$args     = wp_parse_args( $args, $default_args );
	$dbprefix = $wpdb->prefix;
	$join     = '';

	// Join members table if it's profile page.
	if ( false !== $args['include_members'] || false !== $args['exclude_members'] ) {
		$join .= " LEFT JOIN {$dbprefix}bb_access_groups_members AS m ON ( g.id = m.group_id ) ";
	}

	$where_clause = array();

	if ( ! empty( $args['status'] ) ) {
		$status = array();

		if ( in_array( 'enabled', $args['status'], true ) ) {
			$status[] = 1;
		}

		if ( in_array( 'processing', $args['status'], true ) ) {
			$status[] = 2;
		}

		if ( in_array( 'disabled', $args['status'], true ) ) {
			$status[] = 3;
		}

		if ( ! empty( $status ) ) {
			$in_placeholders = implode( ', ', array_fill( 0, count( $status ), '%d' ) );
			$where_clause[]  = $wpdb->prepare( "g.status IN ({$in_placeholders})", $status ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	if ( false !== $args['include'] ) {
		$include_groups = $args['include'];

		if ( ! is_array( $include_groups ) ) {
			$include_groups = explode( ',', $args['include'] );
		}
	}

	if ( ! empty( $include_groups ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $include_groups ), '%s' ) );
		$where_clause[]  = $wpdb->prepare( "g.id IN ({$in_placeholders})", $include_groups ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( false !== $args['exclude'] ) {
		$exclude_groups = $args['exclude'];

		if ( ! is_array( $exclude_groups ) ) {
			$exclude_groups = explode( ',', $args['exclude'] );
		}
	}

	if ( false !== $args['exclude'] && ! empty( $exclude_groups ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $args['exclude'] ), '%s' ) );
		$where_clause[]  = $wpdb->prepare( "g.id NOT IN ({$in_placeholders})", $args['exclude'] ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( false !== $args['include_conditions'] ) {
		$include_conditions = $args['include_conditions'];

		if ( ! is_array( $include_conditions ) ) {
			$include_conditions = explode( ',', $args['include_conditions'] );
		}
	}

	if ( ! empty( $include_conditions ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $include_conditions ), '%s' ) );
		$where_clause[]  = $wpdb->prepare( "g.condition_name IN ({$in_placeholders})", $include_conditions ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( false !== $args['include_sub_conditions'] ) {
		$include_sub_conditions = $args['include_sub_conditions'];

		if ( ! is_array( $include_sub_conditions ) ) {
			$include_sub_conditions = explode( ',', $args['include_sub_conditions'] );
		}
	}

	if ( ! empty( $include_sub_conditions ) ) {
		$sub_in_placeholders = implode( ', ', array_fill( 0, count( $include_sub_conditions ), '%s' ) );
		$where_clause[]  = $wpdb->prepare( "g.sub_condition_name IN ({$sub_in_placeholders})", $include_sub_conditions ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( false !== $args['exclude_conditions'] ) {
		$exclude_conditions = $args['exclude_conditions'];

		if ( ! is_array( $exclude_conditions ) ) {
			$exclude_conditions = explode( ',', $args['exclude_conditions'] );
		}
	}

	if ( ! empty( $exclude_conditions ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_conditions ), '%s' ) );
		$where_clause[]  = $wpdb->prepare( "g.condition_name NOT IN ({$in_placeholders})", $exclude_conditions ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( false !== $args['exclude_sub_conditions'] ) {
		$exclude_sub_conditions = $args['exclude_sub_conditions'];

		if ( ! is_array( $exclude_sub_conditions ) ) {
			$exclude_sub_conditions = explode( ',', $args['exclude_sub_conditions'] );
		}
	}

	if ( ! empty( $exclude_sub_conditions ) ) {
		$sub_exclude_placeholders = implode( ', ', array_fill( 0, count( $exclude_sub_conditions ), '%s' ) );
		$where_clause[]           = $wpdb->prepare( "g.sub_condition_name NOT IN ({$sub_exclude_placeholders})", $exclude_sub_conditions ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( ! empty( $args['include_members'] ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $args['include_members'] ), '%d' ) );
		$where_clause[]  = $wpdb->prepare( "m.user_id IN ({$in_placeholders})", $args['include_members'] ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( ! empty( $args['exclude_members'] ) ) {
		$in_placeholders = implode( ', ', array_fill( 0, count( $args['exclude_members'] ), '%d' ) );
		$where_clause[]  = $wpdb->prepare( "m.user_id NOT IN ({$in_placeholders})", $args['exclude_members'] ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	$where_conditions = ( ! empty( $where_clause ) ) ? 'WHERE ' . implode( ' AND ', $where_clause ) : '';

	$limit_clause = '';

	if ( $args['per_page'] ) {
		$args['per_page'] = (int) $args['per_page'];
		$args['page']     = (int) $args['page'];

		$limit_clause  = 'LIMIT ' . $args['per_page'] . ' ';
		$limit_clause .= 'OFFSET ' . ( $args['page'] - 1 ) * $args['per_page'];
	}

	$order_clause = '';

	if ( ! empty( $args['orderby'] ) ) {
		if ( 'include' === $args['orderby'] ) {
			if ( false !== $args['include'] ) {
				$include_groups = $args['include'];

				if ( ! is_array( $include_groups ) ) {
					$include_groups = explode( ',', $args['include'] );
				}
			}

			if ( ! empty( $include_groups ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_groups ), '%s' ) );
				$order_clause   .= $wpdb->prepare( "ORDER BY FIELD (g.id, {$in_placeholders})", $include_groups ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		} else {
			$order_clause .= 'ORDER BY g.' . esc_sql( $args['orderby'] );
			$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}
	}

	$groupby = 'group BY g.id';

	if ( false === $args['only_count'] ) {
		$sql           = "SELECT g.* FROM {$dbprefix}bb_access_groups AS g {$join} {$where_conditions} {$groupby} {$order_clause} {$limit_clause}";
		$access_groups = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	$sql                 = "SELECT COUNT(*) FROM {$dbprefix}bb_access_groups AS g {$join} {$where_conditions} {$order_clause}";
	$access_groups_count = $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
	$retval              = array();

	if ( ! empty( $access_groups ) ) {
		foreach ( $access_groups as $access_group ) {
			$retval[] = bb_access_controls_prepare_group_item( $access_group );
		}
	}

	$groups_result = array(
		'result' => $retval,
		'count'  => $access_groups_count,
	);

	/**
	 * Fires after access group data fetched from the DB.
	 *
	 * @param array $groups_result Access group result.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	return apply_filters( 'bb_access_controls_get_groups', $groups_result );
}

/**
 * Function to get access group members.
 *
 * @param array $args get members arguments.
 *
 * @since 1.5.2.1
 *
 * @return null|array|object|stdClass[]
 */
function bb_access_controls_get_group_members( $args = array() ) {
	global $wpdb;

	$default_args = array(
		'status'        => array( 'enabled', 'processing', 'disabled' ),
		'per_page'      => false,
		'page'          => 1,
		'orderby'       => 'id',
		'order'         => 'desc',
		'include'       => false,
		'exclude'       => false,
		'group_include' => false,
		'group_exclude' => false,
		'search'        => '',
		'only_count'    => false,
	);

	$args          = wp_parse_args( $args, $default_args );
	$global_prefix = $wpdb->prefix;
	$where_clause  = array();

	if ( ! empty( $args['status'] ) ) {
		$status = array();

		if ( in_array( 'enabled', $args['status'], true ) ) {
			$status[] = 1;
		}

		if ( in_array( 'processing', $args['status'], true ) ) {
			$status[] = 2;
		}

		if ( in_array( 'disabled', $args['status'], true ) ) {
			$status[] = 3;
		}

		if ( in_array( 'trash', $args['status'], true ) ) {
			$status[] = 4;
		}

		if ( ! empty( $status ) ) {
			$include_status = implode( ',', $status );
			$where_clause[] = "status IN ({$include_status})";
		}
	}

	$included_members = array();

	if ( false !== $args['include'] ) {
		$included_members = $args['include'];

		if ( ! is_array( $args['include'] ) ) {
			$included_members = explode( ',', $args['include'] );
		}
	}

	if ( ! empty( $args['search'] ) ) {
		add_action( 'pre_user_query', 'bbapp_pre_user_query', 10, 1 );

		$users = new WP_User_Query(
			array(
				'search'         => '*' . esc_attr( $args['search'] ) . '*',
				'fields'         => 'ID',
				'search_columns' => array(
					'user_login',
					'user_nicename',
					'user_email',
					'display_name',
				),
			)
		);
		$users = $users->get_results();

		remove_action( 'pre_user_query', 'bbapp_pre_user_query', 10 );

		$included_members = array_unique( array_merge( $included_members, $users ) );
		if ( empty( $included_members ) ) {
			$included_members = array( 0 );
		}
	}

	if ( ! empty( $included_members ) ) {
		$include_implode_ids = implode( ',', $included_members );
		$where_clause[]      = "user_id IN ({$include_implode_ids})";
	}

	if ( false !== $args['exclude'] ) {
		$excluded_members = $args['exclude'];
		if ( ! is_array( $args['exclude'] ) ) {
			$excluded_members = explode( ',', $args['exclude'] );
		}
	}

	if ( ! empty( $excluded_members ) ) {
		$exclude_implode_ids = implode( ',', $excluded_members );
		$where_clause[]      = "user_id NOT IN ({$exclude_implode_ids})";
	}

	if ( false !== $args['group_include'] ) {
		$included_groups = $args['group_include'];
		if ( ! is_array( $args['group_include'] ) ) {
			$included_groups = explode( ',', $args['group_include'] );
		}
	}

	if ( ! empty( $included_groups ) ) {
		$group_include_implode_ids = implode( ',', $included_groups );
		$where_clause[]            = "group_id IN ({$group_include_implode_ids})";
	}

	if ( false !== $args['group_exclude'] ) {
		$excluded_groups = $args['group_exclude'];
		if ( ! is_array( $args['group_exclude'] ) ) {
			$excluded_groups = explode( ',', $args['group_exclude'] );
		}
	}

	if ( ! empty( $excluded_groups ) ) {
		$group_exclude_implode_ids = implode( ',', $excluded_groups );
		$where_clause[]            = "group_id NOT IN ({$group_exclude_implode_ids})";
	}

	$where_conditions = ( ! empty( $where_clause ) ) ? 'WHERE ' . implode( ' AND ', $where_clause ) : '';
	$limit_clause     = '';

	if ( $args['per_page'] ) {
		$args['per_page'] = (int) $args['per_page'];
		$args['page']     = (int) $args['page'];

		$limit_clause  = 'LIMIT ' . $args['per_page'] . ' ';
		$limit_clause .= 'OFFSET ' . ( $args['page'] - 1 ) * $args['per_page'];
	}

	$order_clause = '';

	if ( ! empty( $args['orderby'] ) ) {
		$order_clause .= 'ORDER BY ' . esc_sql( $args['orderby'] );
		$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
	}

	$results = array();

	if ( false === $args['only_count'] ) {
		$sql     = "SELECT * FROM {$global_prefix}bb_access_groups_members {$where_conditions} {$order_clause} {$limit_clause}";
		$results = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	$sql                 = "SELECT COUNT(*) FROM {$global_prefix}bb_access_groups_members {$where_conditions} {$order_clause}";
	$group_members_count = $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

	return array(
		'result' => $results,
		'count'  => $group_members_count,
	);
}

/**
 * Pass user search query with first name and last name.
 *
 * @param WP_User_Query $uqi Current instance of WP_User_Query (passed by reference).
 *
 * @since 1.5.5
 */
function bbapp_pre_user_query( $uqi ) {
	global $wpdb;

	$search = '';

	if ( isset( $uqi->query_vars['search'] ) ) {
		$search = trim( $uqi->query_vars['search'] );
	}

	if ( $search ) {
		$search     = trim( $search, '*' );
		$the_search = '%' . $search . '%';

		$search_meta = $wpdb->prepare( " ID IN ( SELECT user_id FROM {$wpdb->usermeta} WHERE ( ( meta_key='first_name' OR meta_key='last_name' )  AND {$wpdb->usermeta}.meta_value LIKE '%s' ) )", $the_search ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.QuotedSimplePlaceholder

		$uqi->query_where = str_replace( 'WHERE 1=1 AND (', 'WHERE 1=1 AND (' . $search_meta . ' OR ', $uqi->query_where );
	}
}

/**
 * Function to update the access group status.
 *
 * @param int    $group_id Access group id.
 * @param string $status   Status to set for the access group.
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_update_group_status( $group_id, $status = 'enabled' ) {
	global $wpdb;

	$access_group = bb_access_controls_get_groups(
		array(
			'include' => $group_id,
			'status'  => false,
		)
	);

	if ( empty( $access_group['result'] ) ) {
		return false;
	}

	if ( 'processing' === $status ) {
		$status = 2;
	} elseif ( 'disabled' === $status ) {
		$status = 3;
	} else {
		$status = 1; // enabled.
	}

	return $wpdb->update( "{$wpdb->prefix}bb_access_groups", array( 'status' => $status ), array( 'id' => $group_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

/**
 * Update group is calculation status.
 *
 * @param int  $group_id       Access group id.
 * @param bool $is_calculating Calculating status to set for the access group.
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_update_group_is_calculating( $group_id, $is_calculating = true ) {
	global $wpdb;

	if ( $is_calculating ) {
		$is_calculating = 1;
	} else {
		$is_calculating = 0;
	}

	$access_group = bb_access_controls_get_groups(
		array(
			'include' => $group_id,
			'status'  => false,
		)
	);

	if ( empty( $access_group['result'] ) ) {
		return false;
	}

	return $wpdb->update( "{$wpdb->prefix}bb_access_groups", array( 'is_calculating' => $is_calculating ), array( 'id' => $group_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

/**
 * Delete Access Group.
 *
 * @param int $group_id Access Group id to delete.
 *
 * @return bool|int
 */
function bb_access_controls_delete_group( $group_id ) {
	global $wpdb;

	$access_group = bb_access_controls_get_group( $group_id );

	if ( empty( $access_group ) ) {
		return false;
	}

	$group_delete = $wpdb->delete( "{$wpdb->prefix}bb_access_groups", array( 'id' => $group_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( $group_delete ) {
		bb_access_controls_delete_group_members_by_status( $group_id );
	}

	return $group_delete;
}

/**
 * Prepare an additional information on access group object.
 *
 * @param object $group Group data object.
 *
 * @since 1.5.2.1
 *
 * @return mixed|void $data
 */
function bb_access_controls_prepare_group_item( $group ) {
	$data = array(
		'id'                 => ( ! empty( $group->id ) ) ? $group->id : 0,
		'status'             => ( ! empty( $group->status ) ) ? $group->status : '',
		'data'               => ( ! empty( $group->data ) ) ? maybe_unserialize( $group->data ) : array(),
		'date_created'       => ( ! empty( $group->date_created ) ) ? $group->date_created : '',
		'date_calculated'    => ( ! empty( $group->date_calculated ) ) ? $group->date_calculated : '',
		'condition_name'     => ( ! empty( $group->condition_name ) ) ? $group->condition_name : '',
		'sub_condition_name' => ( ! empty( $group->sub_condition_name ) ) ? $group->sub_condition_name : '',
		'item_value'         => ( ! empty( $group->item_value ) ) ? $group->item_value : '',
		'is_calculating'     => ( ! empty( $group->is_calculating ) ) ? $group->is_calculating : 0,
	);

	/**
	 * Fires after prepared access group data.
	 *
	 * @param array $data  Prepare Access group.
	 * @param array $group Access group.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	return apply_filters( 'bb_access_controls_prepare_group_item', $data, $group );
}

/**
 * Handle the condition item callback safely.
 *
 * @param array  $condition  registered condition object.
 * @param string $item_value item value to pass into item callback.
 *
 * Returns false when item is not found. and return the preformed link &
 * text when not proper information found.
 *
 * @since 1.5.2.1
 *
 * @return array|bool
 */
function bb_access_controls_condition_item_callback_handler( $condition, $item_value ) {
	$item = isset( $condition['item_callback'] ) && is_callable( $condition['item_callback'] ) ? call_user_func_array( $condition['item_callback'], array( $item_value ) ) : false;

	if ( false === $item || ! is_array( $item ) ) {
		return false;
	} else {
		$link = '#'; // default.
		$name = "#{$item_value}"; // default.

		if ( isset( $item['name'] ) ) {
			$name = $item['name'];
		}

		if ( isset( $item['link'] ) ) {
			$link = $item['link'];
		}

		return array(
			'link' => $link,
			'name' => $name,
		);
	}
}

/**
 * Handle the condition has any item callback safely.
 *
 * @param array  $condition registered condition object.
 * @param string $user_id   User id.
 *
 * @since 1.5.2.1
 *
 * @return array|bool
 */
function bb_access_controls_condition_has_any_item_callback_handler( $condition, $user_id ) {
	$has_any_items = isset( $condition['has_any_items_callback'] ) && is_callable( $condition['has_any_items_callback'] ) ? call_user_func_array( $condition['has_any_items_callback'], array( $user_id ) ) : false;

	if ( ! is_bool( $has_any_items ) ) {
		return false;
	} else {
		return (bool) $has_any_items;
	}
}

/**
 * Update the group data with merge method.
 *
 * Data will not be deleted it will just overrides.
 *
 * @param int   $group_id Access group id.
 * @param array $data     Data to set for the access group.
 *
 * @since 1.5.2.1
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_update_group_data( $group_id, $data = array() ) {
	global $wpdb;

	$access_group = bb_access_controls_get_group( $group_id );
	if ( empty( $access_group ) ) {
		return false;
	}

	$group_data = ! empty( $access_group['data'] ) ? $access_group['data'] : array();
	$group_data = wp_parse_args( $data, $group_data );

	return $wpdb->update( "{$wpdb->prefix}bb_access_groups", array( 'data' => maybe_serialize( $group_data ) ), array( 'id' => $group_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

/**
 * Returns the group data.
 *
 * @param int $group_id Access group id.
 *
 * @since 1.5.2.1
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_get_group_data( $group_id ) {
	$access_group = bb_access_controls_get_group( $group_id );

	if ( empty( $access_group ) ) {
		return false;
	}

	$group_data = ! empty( $access_group['data'] ) ? $access_group['data'] : array();

	return $group_data;
}


/**
 * Function to update the access group date calculated.
 *
 * @param int  $group_id Access group id.
 * @param bool $date     Date str to update, default: Current GMT Date.
 *
 * @since 1.5.2.1
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_update_group_date_calculated( $group_id, $date = false ) {
	global $wpdb;

	if ( empty( $date ) ) {
		$date = current_time( 'mysql', 1 );
	} else {
		$date = date( 'Y-m-d H:i:s', strtotime( $date ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
	}

	$access_group = bb_access_controls_get_group( $group_id );

	if ( ! isset( $access_group ) ) {
		return false;
	}

	return $wpdb->update( "{$wpdb->prefix}bb_access_groups", array( 'date_calculated' => $date ), array( 'id' => $group_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

/**
 * Function to check if certain group has certain user or not.
 *
 * @param int $group_id Group id to check.
 * @param int $user_id  User id to check.
 *
 * @since 1.5.2.1
 *
 * @return null|string
 */
function bb_access_controls_group_has_user( $group_id = 0, $user_id = 0 ) {
	global $wpdb;

	$get_var = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->prefix}bb_access_groups_members WHERE group_id=%d AND user_id=%d AND status=1", (int) $group_id, (int) $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	return ( ! empty( $get_var ) );
}

/**
 * Function to check if certain groups has certain user or not.
 *
 * @param array $group_ids Group ids to check.
 * @param int   $user_id   User id to check.
 *
 * @since 1.5.2.1
 *
 * @return null|string
 */
function bb_access_controls_groups_has_user( $group_ids, $user_id = 0 ) {
	global $wpdb;

	if ( ! empty( $group_ids ) ) {
		$table_name      = $wpdb->prefix . 'bb_access_groups_members';
		$in_placeholders = array_fill( 0, count( $group_ids ), '%s' );
		$in_placeholders = implode( ', ', $in_placeholders );
		$in_statement    = $wpdb->prepare( "`group_id` IN ({$in_placeholders})", $group_ids ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$result          = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$table_name} WHERE {$in_statement} AND user_id=%d AND status=1", (int) $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return ! empty( $result ) && count( $group_ids ) === (int) $result;
	}

	return false;
}

/**
 * Function to check if used has particular group or not.
 *
 * @param int $user_id  User id to check.
 * @param int $group_id Group id to check.
 *
 * @since 1.5.2.1
 *
 * @return bool
 */
function bb_access_controls_user_has_group( $user_id = 0, $group_id = 0 ) {
	global $wpdb;

	$get_var = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) FROM {$wpdb->prefix}bb_access_groups_members WHERE user_id=%d AND group_id=%d AND status=1", (int) $user_id, (int) $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	return ! empty( $get_var );
}

/**
 * Function to add user into particular condition.
 * 2 groups can be eligible for same condition with matched value. specific and any.
 *
 * @param array  $user_ids       User id to add.
 * @param string $condition_name Condition name where user has to be added.
 * @param string $item_value     Condition item value where user need to be added.
 *
 * @since 1.5.2.1
 *
 * @return false|void
 */
function bb_access_controls_condition_add_users( $user_ids, $condition_name, $item_value ) {
	global $wpdb;

	if ( empty( $user_ids ) || empty( $condition_name ) || empty( $item_value ) ) {
		return false;
	}

	$registered_conditions = bb_access_controls_get_conditions();
	$condition_keys        = array_keys( $registered_conditions );

	if ( ! in_array( $condition_name, $condition_keys, true ) ) {
		return false;
	}

	$group_ids = $wpdb->get_col( $wpdb->prepare( "SELECT ID from {$wpdb->prefix}bb_access_groups WHERE condition_name = %s AND ((sub_condition_name = 'specific' AND item_value = %s) OR (sub_condition_name = 'any'))", $condition_name, $item_value ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( empty( $group_ids ) ) {
		return false;
	}

	foreach ( $group_ids as $group_id ) {
		bp_access_controls_group_add_update_members( $group_id, $user_ids );

		/**
		 * Filter fires when a users is added into a group from condition.
		 *
		 * @param int   $group_id group id.
		 * @param array $user_ids User ids.
		 *
		 * @since 1.5.2.1
		 */
		do_action( 'bb_access_controls_condition_group_add_users', $group_id, $user_ids );
	}
	
	/**
	 * Action fires after all users have been added to all groups for a condition.
	 * This can be used for batch processing to improve performance.
	 *
	 * @param array  $group_ids      All group IDs that users were added to.
	 * @param array  $user_ids       All user IDs that were added.
	 * @param string $condition_name The condition name.
	 * @param string $item_value     The item value.
	 *
	 * @since 2.3.70
	 */
	do_action( 'bb_access_controls_condition_batch_group_add_users_complete', $group_ids, $user_ids, $condition_name, $item_value );
}

/**
 * Function to delete users from condition.
 *
 * @param array      $user_ids       User ids to delete from condition.
 * @param string     $condition_name Condition to users delete from.
 * @param int|string $item_value     Condition item value.
 *
 * @since 1.5.2.1
 *
 * @return false|void
 */
function bb_access_controls_condition_remove_users( $user_ids, $condition_name, $item_value ) {
	global $wpdb;

	$registered_conditions = bb_access_controls_get_conditions();

	if ( empty( $user_ids ) || ! is_array( $user_ids ) || empty( $condition_name ) || empty( $item_value ) ) {
		return false;
	}

	$groups             = $wpdb->get_results( $wpdb->prepare( "SELECT id, sub_condition_name from {$wpdb->prefix}bb_access_groups WHERE condition_name = %s AND ((sub_condition_name = 'specific' AND item_value = %s) OR (sub_condition_name = 'any'))", $condition_name, $item_value ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$condition          = isset( $registered_conditions[ $condition_name ] ) ? $registered_conditions[ $condition_name ] : false;
	$any_group_ids      = array();
	$specific_group_ids = array();

	foreach ( $groups as $group ) {
		$group_id           = $group->id;
		$sub_condition_name = $group->sub_condition_name;

		if ( 'any' === $sub_condition_name ) {
			$any_group_ids[] = $group_id;
		} else {
			$specific_group_ids[] = $group_id;
			$in_placeholders      = array_fill( 0, count( $user_ids ), '%s' );
			$in_placeholders      = implode( ', ', $in_placeholders );
			$in_statement         = $wpdb->prepare( "`user_id` IN ({$in_placeholders})", $user_ids ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}bb_access_groups_members WHERE group_id=%d AND {$in_statement}", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			/**
			 * Fires when users deletes from a group via condition.
			 *
			 * @param int $group_id group id.
			 * @param array $user_ids user ids.
			 *
			 * @since 1.5.2.1
			 */
			do_action( 'bb_access_controls_condition_group_remove_users', $group_id, $user_ids );
		}
	}

	$any_users_filter = array();

	foreach ( $user_ids as $user_id ) {
		/**
		 * Only delete the user from any condition if the user has_any_items callbacks returns false.
		 * We have did this in order to only delete users from any conditions of the user has no access to any items.
		 */
		$has_any_items_access = bb_access_controls_condition_has_any_item_callback_handler( $condition, $user_id );

		if ( ! $has_any_items_access ) {
			$any_users_filter[] = $user_id;
		}
	}

	if ( ! empty( $any_users_filter ) ) {
		foreach ( $any_group_ids as $group_id ) {
			if ( $condition ) {
				$in_placeholders = array_fill( 0, count( $any_users_filter ), '%s' );
				$in_placeholders = implode( ', ', $in_placeholders );
				$in_statement    = $wpdb->prepare( "`user_id` IN ({$in_placeholders})", $any_users_filter ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}bb_access_groups_members WHERE group_id=%d AND {$in_statement}", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				/**
				 * Fires when users deletes from a group via condition.
				 *
				 * @param int $group_id group id.
				 * @param array $user_ids user ids.
				 *
				 * @since 1.5.2.1
				 */
				do_action( 'bb_access_controls_condition_group_remove_users', $group_id, $user_ids );
			}
		}
	}

	/**
	 * Action fires after all users have been removed from groups for a condition.
	 * This can be used for batch processing to improve performance.
	 *
	 * @param array $specific_group_ids Groups with specific condition that had users removed.
	 * @param array $any_group_ids Groups with "any" condition that had users removed.
	 * @param array $user_ids All user IDs that were removed.
	 * @param array $any_users_filter Users removed from "any" condition groups.
	 * @param string $condition_name The condition name.
	 * @param int|string $item_value The item value.
	 *
	 * @since 2.3.70
	 */
	do_action( 'bb_access_controls_condition_batch_group_remove_users_complete', $specific_group_ids, $any_group_ids, $user_ids, $any_users_filter, $condition_name, $item_value );
}

/**
 * Function to delete users from condition.
 *
 * @param string     $condition_name Condition to users delete from.
 * @param int|string $item_value     Condition item value.
 *
 * @since 1.5.2.1
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_condition_remove_all_users( $condition_name, $item_value ) {
	global $wpdb;

	if ( empty( $condition_name ) || empty( $item_value ) ) {
		return false;
	}

	$group_ids = $wpdb->get_col( $wpdb->prepare( "SELECT ID from {$wpdb->prefix}bb_access_groups WHERE condition_name = %s AND ((sub_condition_name = 'specific' AND item_value = %s) OR (sub_condition_name = 'any'))", $condition_name, $item_value ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	foreach ( $group_ids as $group_id ) {
		bb_access_controls_delete_group_members_by_status( $group_id );
	}

	return true;
}

/**
 * Function to get group id from specific condition.
 *
 * @param string         $condition_name     Condition to users delete from.
 * @param string|boolean $sub_condition_name sub condition name enum [any|specific], having false will return all.
 * @param bool           $item_value         Condition item value.
 *
 * @since 1.5.2.1
 * @return array|bool
 */
function bb_access_controls_get_group_ids_by_condition( $condition_name, $sub_condition_name = false, $item_value = false ) {
	global $wpdb;

	if ( empty( $condition_name ) ) {
		return false;
	}

	$sub_query = '';

	if ( in_array( $sub_condition_name, array( 'any', 'specific' ), true ) ) {
		if ( 'specific' === $sub_condition_name ) {
			$sub_query = $wpdb->prepare( 'AND (sub_condition_name = %s AND item_value = %s)', $sub_condition_name, $item_value );
		} else {
			$sub_query = $wpdb->prepare( 'AND (sub_condition_name = %s)', $sub_condition_name );
		}
	}

	$group_ids = $wpdb->get_col( $wpdb->prepare( "SELECT ID from {$wpdb->prefix}bb_access_groups WHERE condition_name = %s {$sub_query}", $condition_name ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

	return $group_ids;
}

/**
 * Function to add log for deleted condition item.
 *
 * @param string     $condition_name Condition to users delete from.
 * @param int|string $item_value     Condition item value.
 *
 * @since 1.5.2.1
 *
 * @return bool
 */
function bb_access_controls_condition_item_deleted( $condition_name, $item_value ) {
	global $wpdb;

	if ( empty( $condition_name ) || empty( $item_value ) ) {
		return false;
	}

	$group_ids = $wpdb->get_col( $wpdb->prepare( "SELECT ID from {$wpdb->prefix}bb_access_groups WHERE condition_name = %s AND (sub_condition_name = 'specific' AND item_value = %s)", $condition_name, $item_value ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	foreach ( $group_ids as $group_id ) {
		bb_access_controls_update_group_error( $group_id, 'item_not_found', sprintf( __( 'Item not found.', 'buddyboss-app' ) ), '' );
		bb_access_controls_disable_group( $group_id );
	}

	return true;
}

/**
 * Function add single user to particular condition.
 *
 * @param int    $user_id        User id to add to condition.
 * @param string $condition_name Condition to add user to.
 * @param string $match_value    Condition value to add user to.
 *
 * @since 1.5.2.1
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_condition_add_user( $user_id, $condition_name, $match_value ) {
	return bb_access_controls_condition_add_users( array( $user_id ), $condition_name, $match_value );
}

/**
 * Function to delete single user from condition.
 *
 * @param int    $user_id        User id to delete from condition.
 * @param string $condition_name Condition name to delete user from.
 * @param string $item_value     Item value.
 *
 * @since 1.5.2.1
 *
 * @return null|bool|int|mysqli_result|resource
 */
function bb_access_controls_condition_remove_user( $user_id, $condition_name, $item_value ) {
	return bb_access_controls_condition_remove_users( array( $user_id ), $condition_name, $item_value );
}


/**
 * Function to use add or update user based on group and condition.
 *
 * @param int   $group_id access group id.
 * @param array $user_ids access group member id.
 *
 * @since 1.5.2.1
 *
 * @return null|void
 */
function bp_access_controls_group_add_update_members( $group_id, $user_ids ) {
	$group_members = bb_access_controls_group_has_members( $group_id, $user_ids );

	if ( ! empty( $group_members['add'] ) ) {
		global $wpdb;

		/**
		 * Fires before adding members to the group.
		 *
		 * @since 2.1.20
		 *
		 * @param int   $group_id Group id.
		 * @param array $user_ids User ids.
		 */
		do_action( 'bb_access_controls_group_add_members_before', $group_id, $group_members['add'] );

		$dbprefix = $wpdb->prefix;
		$values   = array();

		foreach ( $group_members['add'] as $user_id ) {
			$values[] = $wpdb->prepare( '(%d,%d,%s,%d)', $group_id, $user_id, gmdate( 'Y-m-d H:i:s' ), 1 );
		}

		if ( ! empty( $values ) ) {
			$query  = "INSERT INTO {$dbprefix}bb_access_groups_members (group_id, user_id, date_added, status) VALUES ";
			$query .= implode( ",\n", $values );
			$wpdb->query( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		}

		/**
		 * Fires after adding members to the group.
		 *
		 * @since 2.1.20
		 *
		 * @param int   $group_id Group id.
		 * @param array $user_ids User ids.
		 */
		do_action( 'bb_access_controls_group_add_members_after', $group_id, $group_members['add'] );
	}

	if ( ! empty( $group_members['update'] ) ) {
		global $wpdb;

		/**
		 * Fires before updating members to the group.
		 *
		 * @since 2.1.20
		 *
		 * @param int   $group_id Group id.
		 * @param array $user_ids User ids.
		 */
		do_action( 'bb_access_controls_group_update_members_before', $group_id, $group_members['update'] );

		$in_placeholders = implode( ', ', array_fill( 0, count( $group_members['update'] ), '%s' ) );
		$where_clause    = $wpdb->prepare( "id IN ({$in_placeholders})", $group_members['update'] ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}bb_access_groups_members SET status=1 WHERE group_id = %d AND {$where_clause}", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		/**
		 * Fires after updating members to the group.
		 *
		 * @since 2.1.20
		 *
		 * @param int   $group_id Group id.
		 * @param array $user_ids User ids.
		 */
		do_action( 'bb_access_controls_group_update_members_after', $group_id, $group_members['update'] );
	}
}

/**
 * Prepare a dynamic group name.
 *
 * @param array  $group          Group data.
 * @param string $sign           Group name sign.
 * @param false  $is_item_status Item status.
 *
 * @since 1.5.2.1
 * @since 1.5.5 Added $is_item_status parameter.
 * @return string
 */
function bb_access_controls_prepare_group_name( $group, $sign = ' → ', $is_item_status = false ) {
	if ( empty( $group ) ) {
		return '';
	}

	$registered_conditions = bb_access_controls_get_conditions();
	$condition_name        = $group['condition_name'];
	$sub_condition_name    = $group['sub_condition_name'];
	$access_group_name     = '';

	if ( 'disabled' === $group['item_value_status'] ) {
		$access_group_name = sprintf( '%1$s "%2$s"', esc_html_x( 'Unknown', 'access controls', 'buddyboss-app' ), esc_html( $condition_name ) );
	} else {
		$condition = $registered_conditions[ $condition_name ];

		if ( 'any' === $sub_condition_name ) {
			$selected_conditions = $condition['labels']['member_of_any_items'];
			$access_group_name   = "{$condition['labels']['condition_name']}{$sign}{$selected_conditions}";
		} elseif ( 'specific' === $sub_condition_name ) {
			if ( 'deleted' === $group['item_value_status'] ) {
				if ( true === $is_item_status ) {
					$access_group_name = sprintf( '%1$s%2$s%3$s', esc_html( $condition['labels']['condition_name'] ), esc_html( $sign ), esc_html_x( 'Deleted', 'access controls', 'buddyboss-app' ) );
				} else {
					$access_group_name = sprintf( '%1$s "%2$s"', esc_html_x( 'Unknown', 'access controls', 'buddyboss-app' ), esc_html( $condition_name ) );
				}
			} else {
				$item_link         = sprintf( '<a class="access-control-group-link" href="%s">%s</a>', $group['item_value_data']['link'], $group['item_value_data']['name'] );
				$access_group_name = "{$condition['labels']['condition_name']}{$sign}{$item_link}";
			}
		}
	}

	return $access_group_name;
}

/**
 * Start Recalculate Group Members.
 *
 * @param int $group_id Access control group id.
 *
 * @since 1.5.2.1
 *
 * @return bool|WP_Error
 */
function bb_access_controls_recalculate_group_members( $group_id ) {
	$group = bb_access_controls_get_group( $group_id );

	if ( empty( $group ) ) {
		bb_access_controls_update_group_error( $group_id, 'recalculating_error', __( 'Provided access group id is invalid.', 'buddyboss-app' ) );

		return new WP_Error( 'error_group_id', __( 'Provided access group id is invalid.', 'buddyboss-app' ) );
	}

	if ( 3 === (int) $group['status'] ) {
		bb_access_controls_update_group_error( $group_id, 'recalculating_error', __( 'Provided access group is disabled.', 'buddyboss-app' ) );

		return new WP_Error( 'error_group_status', __( 'Provided access group is disabled.', 'buddyboss-app' ) );
	}

	$main_group_condition = $group['condition_name'];

	if ( false === bb_access_controls_condition_exists( $main_group_condition ) ) {
		bb_access_controls_update_group_error( $group_id, 'recalculating_error', __( 'Integration not found', 'buddyboss-app' ) );

		return new WP_Error( 'integration_not_found', __( 'Integration not found', 'buddyboss-app' ) );
	}

	bb_access_controls_update_group_is_calculating( $group_id );
	bb_access_controls_update_group_status( $group_id, 'processing' );
	bb_access_controls_update_all_group_members_status( $group_id );
	bb_access_controls_start_members_recalculation( $group_id );
	bb_access_controls_update_group_date_calculated( $group_id );

	return true;
}

/**
 * This function to check condition is exists or not.
 *
 * @param string|int $condition_name Name of the registered condition.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bb_access_controls_condition_exists( $condition_name ) {
	$bb_registered_conditions = bb_access_controls_get_conditions();

	return ( ! empty( $condition_name ) && array_key_exists( $condition_name, $bb_registered_conditions ) ) ? true : false;
}

/**
 * Function to get access groups list.
 *
 * @since 1.5.2.1
 *
 * @return mixed|void
 */
function bb_access_controls_groups_list() {
	$access_groups      = bb_access_controls_get_groups();
	$access_groups_data = array(
		'disabled_items' => array(),
		'list_items'     => array(),
	);

	if ( ! empty( $access_groups['result'] ) ) {
		foreach ( $access_groups['result'] as $access_group ) {
			if ( ! bb_access_controls_condition_exists( $access_group['condition_name'] ) ) {
				continue;
			}

			$disabled = '';

			if ( 3 === (int) $access_group['status'] ) {
				$disabled                               = __( ' (disabled)', 'buddyboss-app' );
				$access_groups_data['disabled_items'][] = (int) $access_group['id'];
			}

			if ( 'deleted' === $access_group['item_value_status'] ) {
				$disabled = '';
			}

			$access_groups_data['list_items'][ $access_group['id'] ] = wp_strip_all_tags( $access_group['name_by_status'] ) . $disabled;
		}
	}

	/**
	 * Access group list filter.
	 *
	 * @param array $access_groups_data Access group list.
	 *
	 * @since 1.5.2.1
	 */
	return apply_filters( 'bb_access_controls_groups', $access_groups_data );
}

/**
 * Prepare allow rules preview html.
 *
 * @param array $rule_data Access rule data.
 *
 * @since 1.5.2.1
 * @return false|string
 */
function bb_access_controls_rule_preview( $rule_data ) {
	ob_start();
	if ( ! empty( $rule_data ) ) {
		if ( ! empty( $rule_data['rule_condition'] ) && 'all-logged-in-members' === $rule_data['rule_condition'] ) {
			echo esc_html_x( 'All logged-in members', 'access controls', 'buddyboss-app' );
		} else {
			$group_ids = ! empty( $rule_data['group_ids'] ) ? $rule_data['group_ids'] : array();
			$group_ids = ! empty( $rule_data['order_group_ids'] ) ? $rule_data['order_group_ids'] : $group_ids;

			if ( ! empty( $group_ids ) ) {
				$groups = bb_access_controls_get_groups(
					array(
						'include' => $group_ids,
						'orderby' => 'include',
					)
				);

				if ( ! empty( $groups['result'] ) ) {
					$html = array();

					foreach ( $groups['result'] as $group_data ) {
						$disabled = '';
						if ( 3 === (int) $group_data['status'] ) {
							$disabled = __( ' (disabled)', 'buddyboss-app' );
						}
						$html[] = $group_data['name_sign'] . $disabled;
					}

					$all_required = ( ! empty( $rule_data['group_condition'] ) && 'all' === $rule_data['group_condition'] ) ? ' *' : '';

					echo wp_kses_post( implode( ', ', $html ) ) . esc_html( $all_required );
				} else {
					echo esc_html_x( '(Group is deleted)', 'access controls', 'buddyboss-app' );
				}
			} else {
				if ( bbapp_is_private_app_enabled() ) {
					echo esc_html_x( 'All logged-in members', 'access controls', 'buddyboss-app' );
				}
			}
		}
	} else {
		if ( bbapp_is_private_app_enabled() ) {
			echo esc_html_x( 'All logged-in members', 'access controls', 'buddyboss-app' );
		} else {
			echo esc_html_x( 'Everyone', 'access controls', 'buddyboss-app' );
		}
	}

	return ob_get_clean();
}

/**
 * Function to get rule content.
 *
 * @param string $field_key   name of the form index where you want to save the rule data.
 * @param string $render_type render style of the rule form fields. enum [table|modal].
 * @param array  $settings    Settings array.
 * @param array  $label       Singular and Plural labels.
 * @param array  $extra       Extra arguments.
 *
 * @since 1.5.2.1
 * @return false|string
 */
function bb_access_controls_get_rule_fields( $field_key, $render_type = 'table', $settings = array(), $label = array(), $extra = array() ) {
	$singular_label             = isset( $label['singular'] ) ? $label['singular'] : '';
	$plural_label               = isset( $label['plural'] ) ? $label['plural'] : '';
	$access_group_condition     = isset( $settings['group_condition'] ) ? $settings['group_condition'] : 'all';
	$access_group_ids           = isset( $settings['group_ids'] ) ? $settings['group_ids'] : array();
	$access_group_ids           = isset( $settings['order_group_ids'] ) ? $settings['order_group_ids'] : $access_group_ids;
	$restrict_to_visible        = isset( $extra['restrict_to_visible'] ) ? $extra['restrict_to_visible'] : false;
	$restricted_message_visible = isset( $extra['restricted_message_visible'] ) ? $extra['restricted_message_visible'] : false;
	$_item_ids                  = isset( $extra['item_ids'] ) ? $extra['item_ids'] : '';

	if ( ! empty( $_item_ids ) && is_string( $_item_ids ) ) {
		$_item_ids = json_decode( htmlspecialchars_decode( $_item_ids ) );
	}

	if ( $restrict_to_visible ) {
		$restrict_to = isset( $settings['rule_condition'] ) ? $settings['rule_condition'] : 'all-logged-in-members';
	}

	if ( $restricted_message_visible ) {
		$restricted_message_data          = ! empty( $settings['restricted_message'] ) ? $settings['restricted_message'] : array();
		$restricted_message               = isset( $restricted_message_data['source'] ) ? $restricted_message_data['source'] : 'default';
		$restricted_message_title         = isset( $restricted_message_data['content']['title'] ) ? $restricted_message_data['content']['title'] : '';
		$restricted_message_content       = isset( $restricted_message_data['content']['content'] ) ? $restricted_message_data['content']['content'] : '';
		$restricted_message_button_action = isset( $restricted_message_data['content']['button_action'] ) ? $restricted_message_data['content']['button_action'] : '';
		$app_page_id                      = isset( $restricted_message_data['content']['app_page_id'] ) ? $restricted_message_data['content']['app_page_id'] : 0;
		$iap_product_id                   = isset( $restricted_message_data['content']['iap_product_id'] ) ? $restricted_message_data['content']['iap_product_id'] : 0;
		$restricted_message_show_teaser   = isset( $restricted_message_data['content']['show_teaser'] ) ? $restricted_message_data['content']['show_teaser'] : '';
		$restricted_message_button_text   = isset( $restricted_message_data['content']['button_text'] ) ? $restricted_message_data['content']['button_text'] : '';
	}

	$type = $render_type;

	ob_start();
	if ( 'modal' === $render_type ) {
		require bbapp()->plugin_dir . 'views/access-controls/access-rules/rule-content-modal.php';
	} else {
		require bbapp()->plugin_dir . 'views/access-controls/access-rules/rule-content-table.php';
	}

	return ob_get_clean();
}

/**
 * Function to prepare restrict message array data.
 *
 * @param array $restricted_message restrict message data.
 *
 * @since 1.5.2.1
 * @return array
 */
function bb_access_controls_get_restrict_message( $restricted_message ) {
	$return['source']  = ! empty( $restricted_message['source'] ) ? $restricted_message['source'] : 'default';
	$return['content'] = null;

	if ( 'custom' === $return['source'] ) {
		$title          = ! empty( $restricted_message['content']['title'] ) ? $restricted_message['content']['title'] : '';
		$content        = ! empty( $restricted_message['content']['content'] ) ? $restricted_message['content']['content'] : '';
		$button_action  = ! empty( $restricted_message['content']['button_action'] ) ? $restricted_message['content']['button_action'] : '';
		$show_teaser    = ! empty( $restricted_message['content']['show_teaser'] ) ? (bool) $restricted_message['content']['show_teaser'] : false;
		$button_text    = ! empty( $restricted_message['content']['button_text'] ) ? $restricted_message['content']['button_text'] : '';
		$iap_product_id = ! empty( $restricted_message['content']['iap_product_id'] ) ? (int) $restricted_message['content']['iap_product_id'] : 0;
		$app_page_id    = ! empty( $restricted_message['content']['app_page_id'] ) ? (int) $restricted_message['content']['app_page_id'] : 0;

		$return['content'] = array(
			'title'          => $title,
			'content'        => $content,
			'button_action'  => $button_action,
			'iap_product_id' => $iap_product_id,
			'app_page_id'    => $app_page_id,
			'button_text'    => $button_text,
			'show_teaser'    => $show_teaser,
		);
	}

	return $return;
}

/**
 * Function check access for user against provided groups rule data.
 *
 * @param array $rule_data Access group data from access rule.
 *
 * @param bool  $user_id   default current user ID.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bb_access_controls_user_can_access_rule( $rule_data, $user_id = false ) {
	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	if ( empty( $rule_data ) || empty( $user_id ) ) {
		return false;
	}

	if ( ! empty( $rule_data['rule_condition'] ) && 'all-logged-in-members' === $rule_data['rule_condition'] && is_user_logged_in() ) {
		return true;
	}

	if ( ! isset( $rule_data['group_condition'] ) || ! isset( $rule_data['group_ids'] ) ) {
		return false;
	}

	$groups    = bb_access_controls_get_groups(
		array(
			'include' => $rule_data['group_ids'],
			'status'  => array( 'enabled' ),
		)
	);
	$group_ids = wp_list_pluck( $groups['result'], 'id' );

	if ( 'all' === $rule_data['group_condition'] ) {
		$can_access = true;

		if ( ! bb_access_controls_groups_has_user( $group_ids, $user_id ) ) {
			$can_access = false;
		}

		return $can_access;
	} elseif ( 'any' === $rule_data['group_condition'] ) {
		$members = bb_access_controls_get_group_members(
			array(
				'status'        => array( 'enabled' ),
				'group_include' => $group_ids,
				'include'       => array( $user_id ),
				'only_count'    => true,
			)
		);
		if ( ! empty( $members['count'] ) ) {
			return true;
		}
	}

	return false;
}

/**
 * Filter the condition item values which are found configured with other Access Groups.
 *
 * @param string $condition_name condition name.
 * @param array  $item_values    list of conditions item values.
 *
 * @return array
 */
function bb_access_controls_filter_condition_items_already_configured( $condition_name, $item_values ) {
	global $wpdb;

	$in_placeholders = array_fill( 0, count( $item_values ), '%s' );
	$in_placeholders = implode( ', ', $in_placeholders );
	$in_statement    = $wpdb->prepare( "`item_value` IN ({$in_placeholders})", $item_values ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$db_values       = $wpdb->get_col( $wpdb->prepare( "SELECT item_value FROM {$wpdb->prefix}bb_access_groups WHERE `condition_name`=%s AND {$in_statement}", $condition_name ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

	return $db_values;
}

/**
 * Trigger background job.
 *
 * @param int $access_group_id Access group id.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_start_members_recalculation( $access_group_id ) {
	// Start the job process.
	\BuddyBossApp\Admin\AccessControls\AccessGroups\Helper::instance()->bb_start_members_calculation_job( $access_group_id, true );
}

/**
 * Remove all tmp items from temp table.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_remove_all_tmp_items() {
	if ( ! bb_access_controls_get_is_calculating_group_ids() && bb_access_controls_get_group_tmp_items() ) {
		\BuddyBossApp\AccessControls\TmpItems::instance()->truncate_table();
	}
}

/**
 * Function to get calculating all group ids .
 *
 * @since 1.5.2.1
 * @return array|false
 */
function bb_access_controls_get_is_calculating_group_ids() {
	global $wpdb;

	$group_ids = $wpdb->get_col( $wpdb->prepare( "SELECT ID from {$wpdb->prefix}bb_access_groups WHERE is_calculating = %s", 1 ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( empty( $group_ids ) ) {
		return false;
	}

	return $group_ids;
}

/**
 * Remove items from temp table by condition.
 * Function to add error code on group.
 *
 * @param int    $group_id   Group id.
 * @param string $error_code Error code.
 * @param string $error_msg  Error message.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_update_group_error( $group_id, $error_code = '', $error_msg = '' ) {
	if ( ! empty( $error_code ) && ! empty( $error_msg ) ) {
		bb_access_controls_disable_group( $group_id );
	}

	bb_access_controls_update_group_data(
		$group_id,
		array(
			'error_code' => $error_code,
			'error_msg'  => $error_msg,
		)
	);
}

/**
 * This function use to enabling group.
 *
 * @param int $group_id group id.
 *
 * @since 1.5.2.1
 * @return bool|WP_Error
 */
function bb_access_controls_enable_group( $group_id ) {
	$group = bb_access_controls_get_group( $group_id );

	if ( empty( $group ) ) {
		bb_access_controls_update_group_error( $group_id, 'enabling_error', __( 'Provided access group id is invalid.', 'buddyboss-app' ) );

		return new WP_Error( 'enabling_error', __( 'Provided access group id is invalid.', 'buddyboss-app' ) );
	}

	// If group status not found to be disabled return error.
	if ( 3 !== (int) $group['status'] ) {
		return new WP_Error( 'enabling_error', __( 'Provided group is not in disabled state.', 'buddyboss-app' ) );
	}

	// Check if group disable reason was item_not_found then return error.
	if ( ( isset( $group['data']['error_code'] ) && 'item_not_found' === $group['data']['error_code'] ) ) {
		return new WP_Error( 'enabling_error', __( 'Provided item is deleted.', 'buddyboss-app' ) );
	}

	$main_group_condition = $group['condition_name'];

	if ( false === bb_access_controls_condition_exists( $main_group_condition ) ) {
		bb_access_controls_update_group_error( $group_id, 'enabling_error', __( 'Integration not found.', 'buddyboss-app' ) );

		return new WP_Error( 'integration_not_found', __( 'Integration not found', 'buddyboss-app' ) );
	}

	bb_access_controls_update_group_data( $group_id, array( 'is_enabling' => true ) );
	bb_access_controls_update_group_status( $group_id, 'processing' );
	bb_access_controls_start_members_recalculation( $group_id );
	bb_access_controls_update_group_error( $group_id, '', '' );

	return true;
}


/**
 * This function use to disabling group.
 *
 * @param int $group_id group id.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_disable_group( $group_id ) {
	bb_access_controls_update_group_is_calculating( $group_id, false );
	bb_access_controls_update_group_status( $group_id, 'disabled' );
}

/**
 * Returns the tmp items by condition name.
 *
 * @param string $condition_name Condition name.
 *
 * @since 1.5.2.1
 */
function bb_access_controls_remove_tmp_items_by_condition( $condition_name ) {
	\BuddyBossApp\AccessControls\TmpItems::instance()->remove_items_by_condition( $condition_name );
}

/**
 * Function returns rule items which provide Access Group restricts.
 *
 * @param array $args     {.
 *
 * @type array  $include  Group ids to be included.
 * @type int    $per_page Limit the items.
 * @type int    $page     Number of page values to be retrieved.
 *                        }
 *
 * @since 1.5.2.1
 *
 * @return array|bool
 */
function bb_access_controls_get_items_by_groups( $args ) {
	$default_args = array(
		'include'  => false,
		'per_page' => false,
		'page'     => 1,
	);

	$args = wp_parse_args( $args, $default_args );

	if ( empty( $args['include'] ) ) {
		return false;
	}

	$access_rules = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rules(
		array(
			'include_group_id' => $args['include'],
			'per_page'         => $args['per_page'],
			'page'             => $args['page'],
		)
	);

	$item_arr = array();
	$count    = 0;

	if ( ! empty( $access_rules['results'] ) ) {
		foreach ( $access_rules['results'] as $access_rule ) {
			$item_arr[ $count ]['item_id']   = $access_rule['item_id'];
			$item_arr[ $count ]['item_type'] = $access_rule['item_type'];
			$item_arr[ $count ]['groups']    = $access_rule['rule_data']['group_ids'];
			$item_arr[ $count ]['condition'] = $access_rule['rule_data']['group_condition'];
			$count ++;
		}
	}

	return $item_arr;
}

/**
 * Function returns all groups linked to provided rule items.
 * You need to specify the item_type and include in order to get groups linked to them.
 *
 * @param array $args      {.
 *
 * @type string $item_type Item type.
 * @type array  $include   Limit results to specified item ids.
 * @type int    $per_page  Limit the items.
 * @type int    $page      Number of page values to be retrieved.
 *                         }
 *
 * @return array
 * @since 1.5.2.1
 */
function bb_access_controls_get_groups_by_items( $args ) {
	$default_args = array(
		'item_type' => '',
		'include'   => false,
		'per_page'  => false,
		'page'      => 1,
	);

	$args = wp_parse_args( $args, $default_args );

	$items = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rules(
		array(
			'include_item_id'    => $args['include'],
			'include_item_types' => $args['item_type'],
			'per_page'           => $args['per_page'],
			'page'               => $args['page'],
		)
	);

	$group_restricted_items = array();

	if ( ! empty( $items['results'] ) ) {
		foreach ( $items['results'] as $item ) {
			$group_ids = ( ! empty( $item['rule_data']['group_ids'] ) ) ? $item['rule_data']['group_ids'] : array();
			$groups    = bb_access_controls_get_groups( array( 'include' => $group_ids ) );

			if ( ! empty( $groups['result'] ) ) {
				foreach ( $groups['result'] as $group_item ) {
					// Get items linked to the Access Group.
					$rule_items = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rules(
						array(
							'include_group_id'   => $group_item['id'],
							'include_item_types' => $args['item_type'],
							'per_page'           => false,
							'page'               => false,
						)
					);

					$rule_items                        = wp_list_pluck( $rule_items['results'], 'item_id' );
					$group_restricted_item             = $group_item;
					$group_restricted_item['item_ids'] = $rule_items;
					$group_restricted_items[]          = $group_restricted_item;
				}
			}
		}
	}

	return $group_restricted_items;
}


/**
 * Function to return if user has access to item base don item id and item type.
 *
 * @param int    $item_id   Item id.
 * @param string $item_type Item type.
 * @param bool   $user_id   User id to check the access.
 *
 * @return bool
 * @since 1.5.2.1
 */
function bb_access_controls_user_can_access_item( $item_id, $item_type, $user_id = false ) {
	$rule_data = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rule( $item_id, $item_type );
	$rule_data = ! empty( $rule_data['rule_data'] ) ? $rule_data['rule_data'] : array();

	return bb_access_controls_user_can_access_rule( $rule_data, $user_id );
}

/**
 * Function to check if item has access rule.
 *
 * @param int    $item_id   Item ID.
 * @param string $item_type Item type.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bb_access_controls_item_has_rule( $item_id, $item_type ) {
	$access_rule = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rule( $item_id, $item_type );

	return ! empty( $access_rule );
}

/**
 * Function to add category rule order.
 *
 * @param int    $term      Taxonomy term.
 * @param string $item_type item type name.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bb_access_controls_add_category_rule_order( $term, $item_type ) {
	global $wpdb;

	if ( empty( $term ) || empty( $item_type ) ) {
		return false;
	}

	$max_var    = $wpdb->get_var( $wpdb->prepare( "SELECT MAX(meta_value) FROM {$wpdb->prefix}termmeta WHERE meta_key=%s", "bb_access_control_{$item_type}_menu_order" ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$menu_order = ( empty( $max_var ) || null === $max_var ) ? 1 : ( (int) $max_var ) + 1;

	$add = update_term_meta( $term, "bb_access_control_{$item_type}_menu_order", $menu_order );

	if ( $add ) {
		return true;
	}

	return false;
}

/**
 * Returns the registered rules item types.
 *
 * @since 1.6.3
 * @return array
 */
function bb_access_controls_get_rules_item_types() {
	return \BuddyBossApp\AccessControls::instance()->rules_item_types;
}

/**
 * Check if provided rules item type is registered or not.
 *
 * @param string $item_type Item type.
 *
 * @since 1.6.3
 * @return bool
 */
function bb_access_controls_is_rules_item_type_registered( $item_type ) {
	$types = bb_access_controls_get_rules_item_types();

	if ( isset( $types[ $item_type ] ) ) {
		return true;
	} else {
		return false;
	}
}

/**
 * Returns the SQL for getting the items which user doesn't has access to.
 *
 * @param array $args sql argument.
 *
 * @since 1.6.3
 * @return string
 */
function bb_access_controls_has_no_access_items_sql_query( $args ) {
	global $wpdb;

	$default_args = array(
		'columns'   => 'item_id',
		'item_type' => false,
	);

	$args = wp_parse_args( $args, $default_args );

	$rules_table = "{$wpdb->prefix}bb_access_rules";

	$has_access_items = bb_access_controls_has_access_items_sql_query( $args );

	if ( ! $has_access_items ) {
		return $has_access_items;
	}

	$query = $wpdb->prepare( "SELECT  {$args['columns']} FROM {$rules_table} as access_rule_main WHERE  access_rule_main.item_type = %s AND access_rule_main.item_id NOT IN ({$has_access_items})", $args['item_type'] ); //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

	return $query;
}

/**
 * Returns the SQL for getting the items which user has access to.
 *
 * @param array $args sql argument.
 *
 * @since 1.6.3
 * @return bool
 */
function bb_access_controls_has_access_items_sql_query( $args ) {
	global $wpdb;

	$default_args = array(
		'columns'   => 'item_id',
		'item_type' => false,
		'user_id'   => get_current_user_id(),
	);

	$args = wp_parse_args( $args, $default_args );

	if ( empty( $args['item_type'] ) || empty( $args['columns'] ) ) {
		return false;
	}

	$rules_table             = "{$wpdb->prefix}bb_access_rules";
	$rules_groups_table      = "{$wpdb->prefix}bb_access_rules_groups";
	$access_groups_members   = "{$wpdb->prefix}bb_access_groups_members";
	$logged_in_members_match = '';

	// allow rules with logged in rule.
	if ( is_user_logged_in() ) {
		$logged_in_members_match = " OR
	      (
	        access_rule.rule_condition = 'all-logged-in-members'
	      )";
	}
	// phpcs:disable
	$query = $wpdb->prepare(
		"
	SELECT  
	  {$args['columns']}
	FROM 
	  {$rules_table} as access_rule 
	WHERE 
	  access_rule.item_type = %s
	  AND 
	  (access_rule.id IN (
	    SELECT 
	      main_sub.rule_id 
	    FROM 
	      {$rules_groups_table} as main_sub 
	    WHERE 
	      main_sub.rule_id = access_rule.id 
	      AND main_sub.group_id IN (SELECT group_id
				FROM {$access_groups_members}
				WHERE `user_id` = %s) 
	    HAVING 
	      (
	        access_rule.group_condition = %s 
	        AND count(rule_id) = (
	          SELECT 
	            count(group_id) 
	          FROM 
	            {$rules_groups_table} 
	          WHERE 
	            rule_id = main_sub.rule_id
	        )
	      ) 
	      OR (
	        access_rule.group_condition = %s
	      )
	  )
	  {$logged_in_members_match} 
	  )
	",
		$args['item_type'],
		$args['user_id'],
		'all',
		'any'
	);
	// phpcs:enable
	return $query;
}

/**
 * Function to run store meta id background job.
 *
 * @param string $post_type Post type.
 * @param array  $bg_data   Background job data.
 *
 * @since 1.6.3
 * @return bool
 */
function bb_access_controls_run_terms_calculation_job( $post_type, $bg_data = array() ) {

	$default_bg_data = array(
		'status'    => 'start',
		'page'      => 1,
		'post_type' => $post_type,
	);

	$bg_data = wp_parse_args( $bg_data, $default_bg_data );

	// Add the queue job to run on background.
	$bbapp_queue = Jobs::instance();
	// job type allows only 20 char.
	$bbapp_queue->add(
		'bb_ac_term_calc',
		$bg_data,
		1
	);

	return $bbapp_queue->start();
}


/**
 * Function to run store meta id background job.
 *
 * @param array $ids   Post type post ids.
 * @param array $extra Extra data.
 *
 * @since 1.6.3
 * @return bool
 */
function bb_access_controls_run_store_meta_job_by_ids( $ids, $extra = array() ) {

	$bg_data = array(
		'ids'   => $ids,
		'extra' => $extra,
	);

	// Add the queue job to run on background.
	$bbapp_queue = Jobs::instance();
	// job type allows only 20 char.
	$bbapp_queue->add(
		'bb_store_meta_by_ids',
		$bg_data,
		1
	);

	return $bbapp_queue->start();
}

/**
 * Get access items from item ids.
 *
 * @param array      $item_ids  Item ids.
 * @param string     $item_type Item type.
 * @param int|string $user_id   User id.
 *
 * @since 1.7.3
 *
 * @return array
 */
function bb_access_controls_get_access_items_by_item_ids( $item_ids, $item_type, $user_id = 0 ) {
	global $wpdb;
	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$where        = '';
	$default_rule = array();

	/**
	 * Access controls default rule.
	 *
	 * @param array  $default_rule Default rule in array.
	 * @param string $_post_type   Post type.
	 *
	 * @since 1.7.3
	 */
	$default_rule_data    = apply_filters( 'bb_access_controls_default_rule', $default_rule, $item_type );
	$can_access_all_posts = true;

	// Get the default posts rule.
	if ( isset( $default_rule_data['default_setting'] ) && isset( $default_rule_data['default_setting']['restrict_all'] ) && '1' === (string) $default_rule_data['default_setting']['restrict_all']['enabled'] ) {
		$can_access_all_posts = bb_access_controls_user_can_access_rule( $default_rule_data, $user_id );
	}

	/**
	 * Access controls term item type.
	 *
	 * @param string $item_type  Default rule in array.
	 * @param string $_post_type Post type.
	 *
	 * @since 1.7.3
	 */
	$item_types       = apply_filters( 'bb_access_controls_get_term_item_type', '', $item_type );
	$term_rules       = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rules( array( 'include_item_types' => $item_types ) );
	$restricted_terms = array();
	$allowed_terms    = array();
	if ( ! empty( $term_rules['results'] ) ) {
		foreach ( $term_rules['results'] as $term_rule ) {
			$can_access = bb_access_controls_user_can_access_rule( $term_rule['rule_data'], $user_id );
			if ( ! $can_access ) {
				$restricted_terms[] = $term_rule['item_id'];
			} else {
				$allowed_terms[] = $term_rule['item_id'];
			}
		}
	}

	$in_statements     = array();
	$not_in_statements = array();
	if ( $can_access_all_posts ) {  // If user has access to all posts by default.

		// Exclude posts of restricted terms.
		if ( ! empty( $restricted_terms ) ) {
			$allowed_posts       = bb_access_controls_has_access_items_sql_query( array( 'item_type' => $item_type ) );
			$in_placeholders     = implode( ', ', array_fill( 0, count( $restricted_terms ), '%d' ) );
			$not_in_statements[] = $wpdb->prepare( " {$wpdb->prefix}posts.ID NOT IN (SELECT object_id FROM `{$wpdb->prefix}term_relationships` WHERE `term_taxonomy_id` IN ({$in_placeholders}) AND object_id NOT IN ({$allowed_posts}) ) ", $restricted_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	} else { // when by default user has access to no posts.

		// We will allow the posts user has specially access to as priority.
		$allowed_posts   = bb_access_controls_has_access_items_sql_query( array( 'item_type' => $item_type ) );
		$in_statements[] = " {$wpdb->prefix}posts.ID IN ({$allowed_posts}) ";

		// Allowed the posts from user allowed terms.
		if ( ! empty( $allowed_terms ) ) {
			$in_placeholders = implode( ', ', array_fill( 0, count( $allowed_terms ), '%d' ) );
			$in_statements[] = $wpdb->prepare( " {$wpdb->prefix}posts.ID IN (SELECT object_id FROM `{$wpdb->prefix}term_relationships` WHERE `term_taxonomy_id` IN ({$in_placeholders}))", $allowed_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	// Remove all items which user has no access to.
	$not_allowed_posts   = bb_access_controls_has_no_access_items_sql_query( array( 'item_type' => $item_type ) );
	$not_in_statements[] = " {$wpdb->prefix}posts.ID NOT IN ({$not_allowed_posts}) ";

	$and_post_type = $wpdb->prepare( " AND {$wpdb->prefix}posts.post_type = %s", $item_type ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$in_statements = implode( ' OR ', $in_statements );

	if ( ! empty( $in_statements ) ) {
		$where .= " AND  ({$in_statements})";
		if ( ! empty( $restricted_terms ) ) {
			$or_in_placeholders = implode( ', ', array_fill( 0, count( $restricted_terms ), '%d' ) );
			$where             .= $wpdb->prepare( " AND {$wpdb->prefix}posts.ID NOT IN ( SELECT post_id FROM {$wpdb->prefix}postmeta AS acpm WHERE acpm.meta_key = 'bb_access_term_id' AND acpm.meta_value IN ({$or_in_placeholders}) ) {$and_post_type}", $restricted_terms ); // //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	$not_in_statements = implode( ' AND ', $not_in_statements );
	if ( ! empty( $not_in_statements ) ) {
		$where .= " AND  ({$not_in_statements})";
		if ( ! empty( $allowed_terms ) ) {
			$or_in_placeholders = implode( ', ', array_fill( 0, count( $allowed_terms ), '%d' ) );
			$where             .= $wpdb->prepare( " OR {$wpdb->prefix}posts.ID  IN ( SELECT post_id FROM {$wpdb->prefix}postmeta AS acpm WHERE acpm.meta_key = 'bb_access_term_id' AND acpm.meta_value IN ({$or_in_placeholders}) ) {$and_post_type}", $allowed_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	$sticky_posts = implode( ',', $item_ids );
	$query        = $wpdb->prepare( "SELECT {$wpdb->prefix}posts.ID FROM {$wpdb->prefix}posts WHERE {$wpdb->prefix}posts.post_type = %s AND {$wpdb->prefix}posts.post_status = 'publish' AND {$wpdb->prefix}posts.ID IN ({$sticky_posts}) {$where} ", $item_type ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$results      = $wpdb->get_col( $query ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

	return ! empty( $results ) ? array_map( 'absint', $results ) : array();
}

/**
 * Function to get all group temp items.
 * 
 * @since 1.7.90
 *
 * @return array|false
 */
function bb_access_controls_get_group_tmp_items() {
	global $wpdb;

	$tmp_items = $wpdb->get_col( "SELECT ID FROM {$wpdb->prefix}bb_access_group_tmp_items" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	return ! empty( $tmp_items ) ? $tmp_items : false;
}


/**
 * Function to recalculate the members for has any membership access group.
 *
 * @param string $condition_name condition name.
 *
 * @since 2.0.90
 *
 * @return bool|void|WP_Error
 */
function bb_access_recalculate_member_for_has_any_membership_group( $condition_name ) {
	global $wpdb;

	if ( empty( $condition_name ) ) {
		return false;
	}

	$group_id = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}bb_access_groups WHERE `condition_name`=%s AND `sub_condition_name`='any'", $condition_name ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

	if ( $group_id ) {
		return bb_access_controls_recalculate_group_members( $group_id );
	}
}
