<?php
/**
 * Used by Temp items functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

/**
 * Used by access control condition to process temporary data.
 *
 * Class TmpItems
 */
class TmpItems {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class construct
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return TmpItems
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Insert Bulk Items into Database.
	 *
	 * @todo  : Optimize the insertion process by splitting queries into multiples.
	 *
	 * @param array $items array of items each containing item_type, item_id, item_secondary_id, date_added (utc).
	 *
	 * @since 1.5.2.1
	 * @return bool|false|int
	 */
	public function insert_items( $items ) {
		global $wpdb;

		$dbprefix = $wpdb->prefix;
		$values   = array();

		foreach ( $items as $item ) {
			$item_type         = isset( $item['item_type'] ) ? $item['item_type'] : false;
			$item_id           = isset( $item['item_id'] ) ? $item['item_id'] : 0;
			$item_secondary_id = isset( $item['item_secondary_id'] ) ? $item['item_secondary_id'] : 0;
			$date_added        = isset( $item['date_added'] ) ? gmdate( 'Y-m-d H:i:s', strtotime( $item['date_added'] ) ) : gmdate( 'Y-m-d H:i:s' );

			if ( ! empty( $item_type ) ) {
				$values[] = $wpdb->prepare( '(%s,%d,%d,%s)', $item_type, $item_id, $item_secondary_id, $date_added );
			}
		}

		if ( ! empty( $values ) ) {
			$query  = "INSERT INTO {$dbprefix}bb_access_group_tmp_items (item_type, item_id, item_secondary_id, date_added) VALUES ";
			$query .= implode( ",\n", $values );

			return $wpdb->query( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		}

		return false;
	}

	/**
	 * Select the items from database based on args.
	 *
	 * @param array $args                       {.
	 *
	 * @type string   $exclude_duplicate          Providing a column will exclude all duplicates from results. Default: item_id Enum (item_id|item_secondary_id|false)
	 * @type int|bool $per_page                   Number of results per page. Default: false.
	 * @type int      $page                       Which page of results to fetch. Using page=1 without per_page will result
	 *                                            in no pagination. Default: 1.
	 * @type string   $order_by                   Column to order results by.
	 * @type string   $order                      ASC or DESC. Default: 'DESC'.
	 * @type array    $include_item_types         Array of items types OR comma separated items types to limit query by (IN). Default: false.
	 * @type array    $include_items              Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $include_secondary_items    Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $exclude_items              Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type array    $exclude_secondary_items    Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type boolean  $only_count                 True or false if want to return only counts of selection. Default: false.
	 * @since 1.5.2.1
	 * @return array
	 */
	public function get_items( $args ) {
		global $wpdb;

		$default_args = array(
			'per_page'                => false,
			'exclude_duplicate'       => 'item_id',
			'page'                    => 1,
			'orderby'                 => 'id',
			'order'                   => 'desc',
			'include_item_types'      => false,
			'include_items'           => false,
			'include_secondary_items' => false,
			'exclude_items'           => false,
			'exclude_secondary_items' => false,
			'only_count'              => false,
		);

		$args         = wp_parse_args( $args, $default_args );
		$dbprefix     = $wpdb->prefix;
		$where_clause = array();

		// Include Types.
		if ( false !== $args['include_item_types'] ) {
			$include_types = $args['include_item_types'];

			if ( ! is_array( $include_types ) ) {
				$include_types = explode( ',', $include_types );
			}

			if ( ! empty( $include_types ) ) {
				$include_implode = implode( ',', $include_types );
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_types ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "items.item_type IN ({$in_placeholders})", $include_implode ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include Items.
		if ( false !== $args['include_items'] ) {
			$include_items = $args['include_items'];

			if ( ! is_array( $include_items ) ) {
				$include_items = explode( ',', $include_items );
			}

			if ( ! empty( $include_items ) ) {
				$include_implode_ids = implode( ',', $include_items );
				$in_placeholders     = implode( ', ', array_fill( 0, count( $include_items ), '%s' ) );
				$where_clause[]      = $wpdb->prepare( "items.item_id IN ({$in_placeholders})", $include_implode_ids ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude Item.
		if ( false !== $args['exclude_items'] ) {
			$exclude_items = $args['exclude_items'];

			if ( ! is_array( $exclude_items ) ) {
				$exclude_items = explode( ',', $exclude_items );
			}

			if ( ! empty( $exclude_items ) ) {
				$include_implode_ids = implode( ',', $exclude_items );
				$in_placeholders     = implode( ', ', array_fill( 0, count( $exclude_items ), '%s' ) );
				$where_clause[]      = $wpdb->prepare( "items.item_id NOT IN ({$in_placeholders})", $include_implode_ids ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include Secondary Items.
		if ( false !== $args['include_secondary_items'] ) {
			$include_secondary_items = $args['include_secondary_items'];

			if ( ! is_array( $include_secondary_items ) ) {
				$include_secondary_items = explode( ',', $args['include_secondary_items'] );
			}

			if ( ! empty( $include_secondary_items ) ) {
				$include_implode_ids = implode( ',', $include_secondary_items );
				$in_placeholders     = implode( ', ', array_fill( 0, count( $include_secondary_items ), '%s' ) );
				$where_clause[]      = $wpdb->prepare( "items.item_secondary_id IN ({$in_placeholders})", $include_implode_ids ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude Secondary Item.
		if ( false !== $args['exclude_secondary_items'] ) {
			$exclude_secondary_items = $args['exclude_secondary_items'];

			if ( ! is_array( $exclude_secondary_items ) ) {
				$exclude_secondary_items = explode( ',', $exclude_secondary_items );
			}

			if ( ! empty( $exclude_secondary_items ) ) {
				$include_implode_ids = implode( ',', $exclude_secondary_items );
				$in_placeholders     = implode( ', ', array_fill( 0, count( $exclude_secondary_items ), '%s' ) );
				$where_clause[]      = $wpdb->prepare( "items.item_secondary_id NOT IN ({$in_placeholders})", $include_implode_ids ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		$where_conditions = ( ! empty( $where_clause ) ) ? 'WHERE ' . implode( ' AND ', $where_clause ) : '';

		$limit_clause = '';

		if ( $args['per_page'] ) {
			$args['per_page'] = (int) $args['per_page'];
			$args['page']     = (int) $args['page'];

			$limit_clause  = 'LIMIT ' . $args['per_page'] . ' ';
			$limit_clause .= 'OFFSET ' . ( $args['page'] - 1 ) * $args['per_page'];
		}

		$order_clause = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order_clause .= 'ORDER BY items.' . esc_sql( $args['orderby'] );
			$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$groupby      = '';
		$count_select = 'COUNT(id)';

		if ( $args['exclude_duplicate'] ) {
			$groupby      = "group BY items.{$args["exclude_duplicate"]}";
			$count_select = "COUNT(DISTINCT items.{$args["exclude_duplicate"]})";
		}

		$items = array();

		if ( false === $args['only_count'] ) {
			$sql   = "SELECT items.* FROM {$dbprefix}bb_access_group_tmp_items AS items {$where_conditions} {$groupby} {$order_clause} {$limit_clause}";
			$items = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		}

		$sql   = "SELECT {$count_select} as count FROM {$dbprefix}bb_access_group_tmp_items AS items {$where_conditions}  {$order_clause}";
		$count = $wpdb->get_row( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return array(
			'results' => $items,
			'count'   => isset( $count->count ) ? $count->count : null,
		);
	}

	/**
	 * Truncate table.
	 *
	 * @since 1.5.2.1
	 */
	public function truncate_table() {
		global $wpdb;

		$wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}bb_access_group_tmp_items" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Remove items by condition type.
	 *
	 * @param string $condition_type condition type.
	 *
	 * @since 1.5.2.1
	 */
	public function remove_items_by_condition( $condition_type ) {
		global $wpdb;
		$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}bb_access_group_tmp_items WHERE item_type=%s", $condition_type ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}
}
