<?php
/**
 * Used by Performance functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

use BuddyBoss\Performance\Cache;

/**
 * Class Performance
 */
class Performance {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class construct
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Performance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action(
			'bb_access_controls_condition_batch_group_add_users_complete',
			array(
				$this,
				'purge_items_cache_by_added_users',
			),
			10,
			2
		);
		add_action(
			'bb_access_controls_condition_batch_group_remove_users_complete',
			array(
				$this,
				'purge_items_cache_by_specific_users_and_keys',
			),
			10,
			3
		);
		add_action(
			'bb_access_controls_members_calculation_job_failed',
			array(
				$this,
				'purge_items_cache_by_groups',
			),
			10,
			1
		);
		add_action(
			'bb_access_controls_members_calculation_job_completed',
			array(
				$this,
				'purge_items_cache_by_groups',
			),
			10,
			1
		);

		add_action( 'bb_access_controls_save_access_rules', array( $this, 'purge_items_cache' ), 10, 2 );
		add_action( 'bb_access_controls_delete_access_rules', array( $this, 'purge_items_cache' ), 10, 2 );
		add_action( 'bbapp_menu_created', array( $this, 'purge_app_core_cache' ), 10, 1 );
		add_action( 'bbapp_menu_updated', array( $this, 'purge_app_core_cache' ), 10, 1 );
		add_action( 'bbapp_menu_setting_saved', array( $this, 'purge_app_core_cache' ), 10, 1 );
		add_action( 'bbapp_menu_setting_deleted', array( $this, 'purge_app_core_cache' ), 10, 1 );
		add_action( 'bbapp_menu_priority', array( $this, 'purge_app_core_cache' ), 10, 1 );
		add_action( 'bb_access_controls_taxonomy_items_order_update_after', array( $this, 'taxonomy_order_items_after' ), 10, 2 );
	}

	/**
	 * Purge the user cache for the rule items linked with a Access Group where the users are added.
	 *
	 * @param array $specific_group_ids Groups with specific condition that had users added.
	 * @param array $user_ids All user IDs that were added.
	 *
	 * @since 2.3.70
	 */
	public function purge_items_cache_by_added_users( $specific_group_ids, $user_ids ) {
		$this->purge_items_cache_by_groups_and_users( $specific_group_ids, $user_ids );
	}

	/**
	 * Purge the user cache for the rule items linked with a Access Group where the users are removed.
	 *
	 * @param array $specific_group_ids Groups with specific condition that had users removed.
	 * @param array $user_ids All user IDs that were removed.
	 *
	 * @since 1.5.2.1
	 */
	public function purge_items_cache_by_specific_users( $specific_group_ids, $user_ids ) {
		$this->purge_items_cache_by_groups_and_users( $specific_group_ids, $user_ids );
	}

	/**
	 * Purge the user cache for the rule items linked with a Access Group where the users are added.
	 *
	 * @param array $specific_group_ids Groups with specific condition that had users added.
	 * @param array $user_ids All user IDs that were added.
	 *
	 * @since 2.3.70
	 */
	public function purge_items_cache_by_specific_users_and_keys( $specific_group_ids, $any_group_ids, $user_ids ) {
		$this->purge_items_cache_by_groups_and_users( array_merge( $specific_group_ids, $any_group_ids ), $user_ids );
	}

	/**
	 * Common functionality to purge cache by groups and users.
	 *
	 * @param array $group_ids Group IDs to purge cache for.
	 * @param array $user_ids User IDs to purge cache for.
	 *
	 * @since 2.3.70
	 */
	private function purge_items_cache_by_groups_and_users( $group_ids, $user_ids ) {
		$items      = bb_access_controls_get_items_by_groups( array( 'include' => $group_ids ) );
		$group_keys = $this->get_group_keys_for_items( $items );

		Cache::instance()->purge_by_user_ids( wp_parse_id_list( $user_ids ), $group_keys );
		Cache::instance()->purge_by_component( 'app_core' );
	}

	/**
	 * Get cache group keys for given items.
	 *
	 * @param array $items Array of items with item_type and item_id.
	 *
	 * @return array Array of group keys.
	 *
	 * @since 2.3.70
	 */
	private function get_group_keys_for_items( $items ) {
		$group_keys = array();

		foreach ( $items as $item ) {
			$item_type = $item['item_type'];
			$item_id   = $item['item_id'];

			switch ( $item_type ) {
				case 'app_page':
					$group_keys[] = 'app_page_' . $item_id;
					$group_keys[] = 'bbapp-deeplinking_' . get_permalink( $item_id );
					break;

				case 'post':
					$group_keys[] = 'blog-post_' . $item_id;
					$group_keys[] = 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) );
					break;

				case 'sfwd-courses_term':
				case 'post_term':
					$group_keys[] = 'categories_' . $item_id;
					break;

				case 'sfwd-courses':
					$group_keys[] = 'sfwd-courses_' . $item_id;
					$group_keys[] = 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) );
					$group_keys[] = 'sfwd-lessons';
					$group_keys[] = 'sfwd-topic';
					$group_keys[] = 'sfwd-quiz';
					break;
			}
		}

		return $group_keys;
	}

	/**
	 * Purge Rest API cache by Users
	 *
	 * @param int    $item_id   Item id from access rule.
	 * @param string $item_type Item type from access rule. default (app_page,page,post,post_term,sfwd-courses,sfwd-courses_term).
	 * @param array  $user_ids  User ids.
	 *
	 * @since 1.5.2.1
	 */
	public function purge_by_users( $item_id, $item_type, $user_ids ) {
		$group_keys = array();

		if ( 'app_page' === $item_type ) {
			$group_keys[] = 'app_page_' . $item_id;
			$group_keys[] = 'bbapp-deeplinking_' . get_permalink( $item_id );
		}

		if ( 'post' === $item_type ) {
			$group_keys[] = 'blog-post_' . $item_id;
			$group_keys[] = 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) );
		}

		if ( 'post_term' === $item_type ) {
			$group_keys[] = 'categories_' . $item_id;
		}

		if ( 'sfwd-courses' === $item_type ) {
			$group_keys[] = 'sfwd-courses_' . $item_id;
			$group_keys[] = 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) );
			$group_keys[] = 'sfwd-lessons';
			$group_keys[] = 'sfwd-topic';
			$group_keys[] = 'sfwd-quiz';
		}

		if ( 'sfwd-courses_term' === $item_type ) {
			$group_keys[] = 'categories_' . $item_id;
		}

		Cache::instance()->purge_by_user_ids( wp_parse_id_list( $user_ids ), $group_keys );
		Cache::instance()->purge_by_component( 'app_core' );

		/**
		 * Fires after cache are purged by users ids.
		 *
		 * @type string $item_id   Item id.
		 * @type string $item_type Item type.
		 * @type array  $user_ids  Array of user ids..
		 */
		do_action( 'bb_access_controls_purge_cache_by_users', $item_id, $item_type, $user_ids );
	}

	/**
	 * Purge the all for the rule items linked with a Access Group where the users are added.
	 *
	 * @param int $group_id argument of access groups id.
	 *
	 * @since 1.5.2.1
	 */
	public function purge_items_cache_by_groups( $group_id ) {
		$items = bb_access_controls_get_items_by_groups( array( 'include' => $group_id ) );

		if ( ! empty( $items ) ) {
			foreach ( $items as $item ) {
				$this->purge_by_group( $item['item_id'], $item['item_type'] );
			}
		}
	}

	/**
	 * Purge cache by group.
	 *
	 * @param int    $item_id   Item id from access rule.
	 * @param string $item_type Item type from access rule. default (app_page,page,post,post_term,sfwd-courses,sfwd-courses_term).
	 *
	 * @since 1.5.2.1
	 */
	public function purge_by_group( $item_id, $item_type ) {
		if ( 'app_page' === $item_type ) {
			Cache::instance()->purge_by_group( 'app_page_' . $item_id );
			Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . get_permalink( $item_id ) );
		}

		if ( 'post' === $item_type ) {
			Cache::instance()->purge_by_group( 'blog-post_' . $item_id );
			Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) ) );
		}

		if ( 'post_term' === $item_type ) {
			Cache::instance()->purge_by_group( 'categories_' . $item_id );
			Cache::instance()->purge_by_component( 'blog-post' );
		}

		if ( 'sfwd-courses' === $item_type ) {
			Cache::instance()->purge_by_group( 'sfwd-courses_' . $item_id );
			Cache::instance()->purge_by_group( 'bbapp-deeplinking_' . untrailingslashit( get_permalink( $item_id ) ) );
		}

		if ( 'sfwd-courses_term' === $item_type ) {
			Cache::instance()->purge_by_group( 'categories_' . $item_id );
			Cache::instance()->purge_by_component( 'sfwd-courses' );
			Cache::instance()->purge_by_component( 'sfwd-lessons' );
			Cache::instance()->purge_by_component( 'sfwd-topic' );
			Cache::instance()->purge_by_component( 'sfwd-quiz' );
		}

		if ( 'settings' === $item_id && 'default_post' === $item_type ) {
			Cache::instance()->purge_by_component( 'blog-post' );
		}

		if ( 'settings' === $item_id && 'default_ld_course' === $item_type ) {
			Cache::instance()->purge_by_component( 'sfwd-courses' );
		}

		Cache::instance()->purge_by_component( 'app_core' );

		do_action( 'bb_access_controls_purge_cache_by_group', $item_id, $item_type );
	}

	/**
	 * Purge items cache.
	 *
	 * @param array  $item_ids  Item id from access rule.
	 * @param string $item_type Item type from access rule.
	 *
	 * @since 1.5.2.1
	 */
	public function purge_items_cache( $item_ids, $item_type ) {
		foreach ( $item_ids as $item_id ) {
			$this->purge_by_group( $item_id, $item_type );
		}

		Cache::instance()->purge_by_component( 'app_core' );
	}

	/**
	 * Purge App core cache.
	 *
	 * @param string $menu_type Menu type.
	 *
	 * @since 1.5.2.1
	 */
	public function purge_app_core_cache( $menu_type ) {
		Cache::instance()->purge_by_component( 'app_core' );
	}

	/**
	 * Purge Component cache.
	 *
	 * @param string $post_type Post type name.
	 * @param string $item_type Category item type.
	 *
	 * @since 1.5.2.1
	 * @since 1.6.3 Added new argument $post type and $item type
	 */
	public function taxonomy_order_items_after( $post_type, $item_type ) {
		if ( 'post' === $post_type ) {
			Cache::instance()->purge_by_component( 'blog-post' );
		}
		if ( 'sfwd-courses' === $post_type ) {
			Cache::instance()->purge_by_component( 'sfwd-courses' );
		}
	}
}
