<?php
/**
 * Used by Access Controls integration functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

use BuddyBossApp\AccessControls;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class BB_Access_Condition
 */
abstract class Integration_Abstract {

	/**
	 * Class instance.
	 *
	 * @var bool
	 */
	private static $instances = array();

	/**
	 * Conditions
	 *
	 * @var array
	 */
	public $conditions = array();

	/**
	 * Setting screen.
	 *
	 * @var array
	 */
	public $settings_screens = array();

	/**
	 * CoreAbstract constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Class instance called.
	 *
	 * @since 1.5.2.1
	 *
	 * @return Integration_Abstract
	 */
	public static function instance() {
		$class = get_called_class();

		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->init();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Setup method.
	 *
	 * @since 1.5.2.1
	 */
	public function init() {
		if ( method_exists( $this, 'setup' ) ) {
			$this->setup();
		}

		$this->hooks();
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.5.2.1
	 */
	public function hooks() {
		add_filter( 'bb_access_controls_get_conditions', array( $this, 'get_conditions' ) );
		add_filter( 'bb_access_controls_get_settings_screen', array( $this, 'get_settings_screen' ) );
	}

	/**
	 * Register Condition for Access Group.
	 *
	 * @param array $args                         {.
	 *
	 * @type    string  $condition                    Required. name of the condition, shouldn't contain space & special
	 *                                                character excluding "-","_"
	 * @type    string  $items_callback               Required. callback for items of the conditions.
	 * @type    string  $users_callback               Required. callback for users results for the condition.
	 * @type    array   $labels                       Required. labels for conditions used on UI.
	 * @type    boolean $support_any_items            Optional. If condition support all items then true & if only
	 *                                                specific then false.
	 *                                                Default: false
	 *
	 * }
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function register_condition( array $args ) {
		$defaults = array(
			'condition'         => '',
			'items_callback'    => '',
			'users_callback'    => '',
			'labels'            => array(
				'condition_name'          => '',
				'member_of_specific_item' => '',
				'member_of_any_items'     => '',
				'item_singular'           => '',
			),
			'support_any_items' => false,
		);

		$args = wp_parse_args( $args, $defaults );

		if ( empty( $args['condition'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please add condition name.', 'buddyboss-app' ), '' );
		}

		if ( 20 < strlen( $args['condition'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Condition name should be less than 20 charcters', 'buddyboss-app' ), '' );
		}

		if ( empty( $args['items_callback'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please define condition item callback function.', 'buddyboss-app' ), '' );
		}

		if ( empty( $args['users_callback'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please define condition user callback function.', 'buddyboss-app' ), '' );
		}

		// Label Fallback.
		if ( empty( $args['labels']['condition_name'] ) ) {
			$args['labels']['condition_name'] = $args['condition'];
		}

		if ( empty( $args['labels']['member_of_any_items'] ) ) {
			$args['labels']['member_of_any_items'] = sprintf( '%1$s %2$s', _x( 'Member of any', 'access controls', 'buddyboss-app' ), $args['labels']['condition_name'] );
		}

		if ( empty( $args['labels']['member_of_specific_item'] ) ) {
			$args['labels']['member_of_specific_item'] = sprintf( '%1$s %2$s', _x( 'Member of a specific', 'access controls list table', 'buddyboss-app' ), $args['labels']['condition_name'] );
		}

		if ( empty( $args['labels']['item_singular'] ) ) {
			$args['labels']['item_singular'] = $args['labels']['condition_name'];
		}

		AccessControls::instance()->conditions[ $args['condition'] ] = $args;
	}

	/**
	 * Register Rules Item Type.
	 * Only registered item types rules can be added inside Access Control.
	 *
	 * @param string $item_type item type.
	 */
	public function register_rules_item_type( $item_type ) {
		AccessControls::instance()->rules_item_types[ $item_type ] = array(
			'item_type' => $item_type,
		);
	}

	/**
	 * Function to register a setting screen for Access Controls.
	 *
	 * @param array $args            {.
	 *
	 * @type    string $id              Screen ID
	 * @type    string $menu_title      The text to be used for the screen menu.
	 * @type    string $page_title      The text to be displayed in the title tags of the page when the screen menu is selected.
	 * @type    array  $render_callback The function to be called to output the content for this screen page
	 * @type    array  $options         {
	 * @type    array  $tutorial_link   Link which will be displayed on content of this screen page.
	 *
	 * }
	 *
	 * }
	 * @return bool
	 */
	public function register_settings_screen( $args ) {
		$defaults = array(
			'id'              => '',
			'menu_title'      => '',
			'page_title'      => '',
			'render_callback' => '',
			'options'         => array(
				'tutorial_link' => '',
			),
		);

		$args = wp_parse_args( $args, $defaults );

		if ( empty( $args['id'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Screen id is required on register_settings_screen.', 'buddyboss-app' ), '' );

			return false;
		}

		if ( empty( $args['menu_title'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Screen menu_title is required on register_settings_screen.', 'buddyboss-app' ), '' );

			return false;
		}

		if ( empty( $args['page_title'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Screen page_title is required on register_settings_screen.', 'buddyboss-app' ), '' );

			return false;
		}

		if ( empty( $args['render_callback'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Screen render_callback is required on register_settings_screen.', 'buddyboss-app' ), '' );

			return false;
		}

		$this->settings_screens[ $args['id'] ] = $args;

		return true;
	}

	/**
	 * This function to return users_callback valid format.
	 *
	 * @param array $users user ids.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function return_users( $users ) {
		return array(
			'type' => 'users',
			'data' => $users,
		);
	}

	/**
	 * This function to return users_callback valid format.
	 *
	 * @param string $string wait message.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function return_wait( $string ) {
		return array(
			'type' => 'wait',
			'data' => $string,
		);
	}

	/**
	 * This function to return users_callback valid format.
	 *
	 * @param \WP_Error|string $error Error message or WP error object.
	 *
	 * @since 1.5.2.1
	 * @return string|\WP_Error|array
	 */
	public function return_error( $error ) {
		return array(
			'type' => 'error',
			'data' => $error,
		);
	}

	/**
	 * Get conditions.
	 *
	 * @param array $conditions conditions.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function get_conditions( $conditions ) {
		foreach ( $this->conditions as $k => $v ) {
			if ( ! isset( $conditions[ $k ] ) ) {
				$conditions[ $k ] = $v;
			}
		}

		return $conditions;
	}

	/**
	 * Get setting screen.
	 *
	 * @param array $settings_screen setting screen.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function get_settings_screen( $settings_screen ) {
		return array_merge( $settings_screen, $this->settings_screens );
	}

	/**
	 * Starts the members recalculation for groups matching with condition & item_value.
	 *
	 * @param string     $condition_name Condition name to delete user from.
	 * @param int|string $item_value     Condition item value.
	 *
	 * @since 1.5.2.1
	 */
	public function condition_recalculate( $condition_name, $item_value ) {
		$group_ids = bb_access_controls_get_group_ids_by_condition( $condition_name, 'specific', $item_value );

		foreach ( $group_ids as $group_id ) {
			bb_access_controls_recalculate_group_members( $group_id );
		}
	}

	/**
	 * Starts members calculation for group (any items group type) matching with condition.
	 *
	 * @param string $condition_name Condition name to delete user from.
	 *
	 * @since 1.5.2.1
	 */
	public function condition_recalculate_any( $condition_name ) {
		$group_ids = bb_access_controls_get_group_ids_by_condition( $condition_name, 'any' );

		foreach ( $group_ids as $group_id ) {
			bb_access_controls_recalculate_group_members( $group_id );
		}
	}

	/**
	 * Function to add user into particular condition.
	 *
	 * @param array  $user_ids       User id to add.
	 * @param string $condition_name Registered condition to add user.
	 * @param string $match_value    Condition value to add user.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 * @since 1.5.2.1
	 */
	public function condition_add_users( $user_ids, $condition_name, $match_value ) {
		return bb_access_controls_condition_add_users( $user_ids, $condition_name, $match_value );
	}

	/**
	 * Function add single user to particular condition.
	 *
	 * @param int    $user_id        User id to add to condition.
	 * @param string $condition_name Condition to add user to.
	 * @param string $match_value    Condition value to add user to.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 * @since 1.5.2.1
	 */
	public function condition_add_user( $user_id, $condition_name, $match_value ) {
		return bb_access_controls_condition_add_user( $user_id, $condition_name, $match_value );
	}

	/**
	 * Function to remove users from condition.
	 *
	 * @param array  $user_ids       User ids to delete from codition.
	 * @param string $condition_name Condition to users delete from.
	 * @param string $item_value     Item value.
	 *
	 * @since 1.5.2.1
	 * @return null|bool|int|\mysqli_result|resource
	 */
	public function condition_remove_users( $user_ids, $condition_name, $item_value ) {
		return bb_access_controls_condition_remove_users( $user_ids, $condition_name, $item_value );
	}

	/**
	 * Function to delete single user from condition.
	 *
	 * @param int        $user_id        User id to delete from condition.
	 * @param string     $condition_name Condition name to delete user from.
	 * @param int|string $item_value     Condition item value.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 * @since 1.5.2.1
	 */
	public function condition_remove_user( $user_id, $condition_name, $item_value ) {
		return bb_access_controls_condition_remove_user( $user_id, $condition_name, $item_value );
	}

	/**
	 * Function to delete all user from condition.
	 *
	 * @param string     $condition_name Condition name to delete user from.
	 * @param int|string $item_value     Condition item value.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 * @since 1.5.2.1
	 */
	public function condition_remove_all_users( $condition_name, $item_value ) {
		return bb_access_controls_condition_remove_all_users( $condition_name, $item_value );
	}

	/**
	 * Function to update error message for deleted condition item.
	 *
	 * @param string     $condition  Condition name to delete user from.
	 * @param int|string $item_value Condition item value.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 * @since 1.5.2.1
	 */
	public function condition_item_deleted( $condition, $item_value ) {
		return bb_access_controls_condition_item_deleted( $condition, $item_value );
	}

	/**
	 * Helpers function for paginating from a list of array.
	 * It's Recommended to Paginate the items directly from Database for performance & scalability.
	 * Mainly used for a paginating the big array before passing to list when option to paginate direct from
	 * database is unavailable.
	 *
	 * @param array  $items  Items array.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit items.
	 * @param string $search Search string.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|mixed
	 */
	public function paginate_items_list( $items, $page, $limit, $search = '' ) {
		return Helpers::instance()->bb_process_items_pagination( $items, $page, $limit, $search );
	}

	/**
	 * Helpers function for paginating from a list of array.
	 * It's Recommended to Paginate the users directly from Database for performance & scalability.
	 * Mainly used for a paginating the big array before passing to list when option to paginate direct from
	 * database is unavailable.
	 *
	 * @param array $items Items array.
	 * @param int   $page  Page number.
	 * @param int   $limit Limit items.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|mixed
	 */
	public function paginate_users_list( $items, $page, $limit ) {
		return Helpers::instance()->bb_process_users_pagination( $items, $page, $limit );
	}

	/*********** Abstract methods **********/
	/**
	 * Do all register and initiate from this function when writing extend class.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	abstract public function setup();
}
