<?php
/**
 * Used by Helper functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

use BuddyBossApp\Helpers\Custom_Multiselect_Dropdown;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class AccessConditionHelper
 */
class Helpers {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class construct
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Helpers
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Init function
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		/**
		 * Filters dynamic select2 callbacks.
		 *
		 * @param array $dynamic_select2_callbacks Array of dynamic select2 dropdown slugs.
		 *
		 * @since 1.8.00
		 */
		$dynamic_select2_callbacks = apply_filters(
			'bb_access_controls_dynamic_select_callbacks',
			array(
				'access_control_settings[rule][group_ids]',
				'access_control_settings[group_ids]',
			)
		);

		if ( ! empty( $dynamic_select2_callbacks ) && is_array( $dynamic_select2_callbacks ) ) {
			foreach ( $dynamic_select2_callbacks as $dynamic_select2_callback ) {
				Custom_Multiselect_Dropdown::instance()->register_results_callback(
					$dynamic_select2_callback,
					array(
						'results_callback' => array(
							$this,
							'get_access_group_list',
						),
					)
				);
			}
		}
	}

	/**
	 * Function to process pagination data.
	 *
	 * @param array  $items  Items array.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit items.
	 * @param string $search Search string.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|mixed
	 */
	public function bb_process_items_pagination( $items, $page, $limit, $search ) {
		if ( empty( $items ) ) {
			return $items;
		}

		$results = array();
		$count   = 0;

		foreach ( $items as $key => $item ) {
			if ( ! empty( $search ) && strpos( strtolower( $item['name'] ), strtolower( $search ) ) !== false ) {
				$results[ $count ]['id']   = $key;
				$results[ $count ]['name'] = $item;
			} elseif ( empty( $search ) ) {
				$results[ $count ]['id']   = $key;
				$results[ $count ]['name'] = $item;
			}
			$count ++;
		}

		if ( - 1 === $limit ) {
			$result_arr = $results;
		} else {
			$offset     = ( $page - 1 ) * $limit;
			$offset     = ( $offset < 0 ) ? 0 : $offset;
			$result_arr = array_slice( $results, $offset, $limit );
		}

		$send_arr = array();

		if ( ! empty( $result_arr ) ) {
			foreach ( $result_arr as $result_arr_item ) {
				$send_arr[ $result_arr_item['id'] ] = $result_arr_item['name'];
			}
		}

		return $send_arr;
	}

	/**
	 * Function to process user pagination data.
	 *
	 * @param array $items Items array.
	 * @param int   $page  Page number.
	 * @param int   $limit Limit items.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|mixed
	 */
	public function bb_process_users_pagination( $items, $page, $limit ) {
		if ( empty( $items ) ) {
			return $items;
		}

		if ( - 1 === $limit ) {
			$result_arr = $items;
		} else {
			$offset     = ( $page - 1 ) * $limit;
			$offset     = ( $offset < 0 ) ? 0 : $offset;
			$result_arr = array_slice( $items, $offset, $limit );
		}

		return $result_arr;
	}

	/**
	 * Prepare Access Groups response for select2 ajax request.
	 *
	 * @param string $search Search term.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit.
	 * @param array  $extras Extra data.
	 *
	 * @since 1.8.00
	 * @return void
	 */
	public function get_access_group_list( $search, $page, $limit, $extras ) {
		$access_controls_groups = bb_access_controls_groups_list();
		$results                = array();

		if ( ! empty( $access_controls_groups['list_items'] ) ) {
			foreach ( $access_controls_groups['list_items'] as $key => $item ) {
				if ( ! empty( $search ) && strpos( strtolower( $item ), strtolower( $search ) ) !== false ) {
					$results[] = array(
						'id'       => $key,
						'text'     => $item,
						'disabled' => ( ! empty( $access_controls_groups['disabled_items'] ) && in_array( $key, $access_controls_groups['disabled_items'], true ) ),
					);
				} elseif ( empty( $search ) ) {
					$results[] = array(
						'id'       => $key,
						'text'     => $item,
						'disabled' => ( ! empty( $access_controls_groups['disabled_items'] ) && in_array( $key, $access_controls_groups['disabled_items'], true ) ),
					);
				}
			}
		}

		if ( - 1 === $limit ) {
			$result_arr = $results;
		} else {
			$offset     = ( $page - 1 ) * $limit;
			$offset     = ( $offset < 0 ) ? 0 : $offset;
			$result_arr = array_slice( $results, $offset, $limit );
		}

		$send_arr = array();

		if ( ! empty( $result_arr ) ) {
			foreach ( $result_arr as $result_arr_item ) {
				$send_arr[] = array(
					'id'       => $result_arr_item['id'],
					'text'     => html_entity_decode( $result_arr_item['text'], ENT_QUOTES ) ,
					'disabled' => $result_arr_item['disabled'],
				);
			}
		}

		wp_send_json(
			array(
				'items' => $send_arr,
			)
		);
	}
}
