<?php
/**
 * Holds post listing for Access Controls.
 *
 * @package BuddyBossApp\AccessControls\Core\Settings
 */

namespace BuddyBossApp\AccessControls\Core\Settings;

use BuddyBossApp\AccessControls\Core\List_Table;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Posts custom list.
 */
class PostsList extends List_Table {

	/**
	 * Class constructor
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {

		parent::__construct(
			Posts::instance()->post_type,
			array(
				'singular'     => _x( 'Post', 'access control posts integration settings', 'buddyboss-app' ), // singular name of the listed records.
				'plural'       => _x( 'Posts', 'access control posts integration settings', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'         => false, // should this table support ajax?
				'screen'       => Posts::instance()->screen_name,
				'column_slug'  => 'post',
				'column_title' => _x( 'Post', 'access control posts integration settings', 'buddyboss-app' ),
			)
		);
	}

	/**
	 * No post found
	 *
	 * @since 1.5.2.1
	 */
	public function no_items() {
		esc_html_e( 'No access post found.', 'buddyboss-app' );
	}

	/**
	 * Posts column custom values.
	 *
	 * @param object $item Post item.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_post( $item ) {
		$output = '';

		$title = $item->post_title;
		if ( empty( $title ) ) {
			$title = _x( '(no title)', 'access control posts integration settings', 'buddyboss-app' );
		}

		if ( current_user_can( 'edit_posts', $item->ID ) ) {
			$output .= '<strong><a href="' . get_edit_post_link( $item->ID ) . '">' . apply_filters( 'the_title', $title, $item->ID ) . '</a></strong>';
		} else {
			$output .= '<strong>' . apply_filters( 'the_title', $title, $item->ID ) . '</strong>';
		}

		echo wp_kses_post( $output );
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Post item.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_inherited_from( $item ) {
		$inherited_from = Posts::instance()->get_access_inherited_from( $item->ID, $item->post_type );

		if ( 'item' === $inherited_from['type'] ) {
			echo '-';
		} elseif ( 'taxonomy' === $inherited_from['type'] ) {
			$stored_term = $inherited_from['term'];

			echo sprintf( '%s: <a href="%s">%s</a>', esc_html( get_taxonomy( $stored_term->taxonomy )->labels->singular_name ), esc_url( get_term_link( $stored_term, $stored_term->taxonomy ) ), esc_html( $stored_term->name ) );
		} else {
			esc_html_e( 'Default Settings', 'buddyboss-app' );
		}
	}

}
