<?php
/**
 * Holds posts access rules render funcationality.
 *
 * @package BuddyBossApp\AccessControls\Core\Settings
 */

namespace BuddyBossApp\AccessControls\Core\Settings;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\AccessControls\Actions;
use BuddyBossApp\AccessControls\Core\Core;
use BuddyBossApp\Admin\AccessControls\AccessRules;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class Posts
 */
class Posts {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Determines if need to load on the current page.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Posts list.
	 *
	 * @var $posts_list
	 */
	public $posts_list;

	/**
	 * Post type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $post_type = 'post';

	/**
	 * Screen type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $screen_name = 'posts';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return Posts
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Posts constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		CategoryRule::instance();

		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );

		// Screen options.
		add_action( 'load-buddyboss-app_page_bbapp-access-controls', array( $this, 'get_screen_options' ), 99 );
		add_filter( 'set-screen-option', array( $this, 'bb_access_rules_screen_option' ), 10, 3 );

		// Save default settings.
		add_action( 'wp_ajax_bb_access_controls_save_default_setting', array( $this, 'save_default_setting' ) );
		// Rest response.
		add_filter( 'bbapp_settings_rest_response_access_controls', array( $this, 'get_default_settings' ), 10, 2 );

		add_filter( 'bb_access_controls_data', array( $this, 'get_bb_access' ), 10, 4 );
		add_filter( 'bb_access_controls_allow_post_types_filter', array( $this, 'allow_post_types_filter' ), 10, 1 );
		add_filter( 'bb_access_controls_default_rule', array( $this, 'get_default_rule' ), 10, 2 );
		add_filter( 'bb_access_controls_get_term_item_type', array( $this, 'get_term_item_type' ), 10, 2 );
		add_filter( 'bbapp_render_app_page_data', array( $this, 'add_block_visibility_fitler' ), 10, 2 );
		add_filter( "rest_{$this->post_type}_query", array( $this, 'bb_access_controls_visibility_filter' ), 10, 2 );

		add_action( 'bb_access_controls_taxonomy_items_order_update_after', array( $this, 'run_job_after_taxonomy_order_update' ), 10, 2 );
		add_action( 'bb_access_controls_save_access_rules', array( $this, 'run_job_after_save_access_rules' ), 10, 2 );
		add_action( 'bb_access_controls_deleted_item_rule', array( $this, 'run_job_after_delete_item' ), 10, 2 );
		add_action( 'bb_access_controls_delete_category_rule', array( $this, 'run_job_after_delete_category_item' ), 10, 1 );
		add_action( 'set_object_terms', array( $this, 'run_job_after_category_add_or_remove' ), 10, 6 );

		add_filter( 'bb_access_controls_get_access_inherited_from', array( $this, 'prepare_job_get_access_inherited_from' ), 10, 3 );

	}

	/**
	 * Load posts custom List.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function admin_init() {
		if ( $this->will_render() ) {
			// Post list table.
			$this->posts_list = new PostsList();
			$this->posts_list->prepare_items();
		}
	}

	/**
	 * Register the screen options.
	 *
	 * @since 1.5.5
	 */
	public function get_screen_options() {
		$screen  = get_current_screen();
		$tab_get = ( ! empty( $_GET['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-access-controls' !== $screen->id || $this->screen_name !== $tab_get ) {
			return;
		}

		// Loop through all the options and add a screen option for each.
		add_screen_option(
			'per_page',
			array(
				'label'   => _x( 'Number of Posts per page:', 'access controls list table', 'buddyboss-app' ),
				'default' => 10,
				'option'  => "bb_access_controls_{$this->post_type}_per_page",
			)
		);
	}

	/**
	 * Handle save/update of screen options for Posts.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @since 1.5.5
	 *
	 * @return string|int Option value. False to abandon update.
	 */
	public function bb_access_rules_screen_option( $value, $option, $new_value ) {
		if ( "bb_access_controls_{$this->post_type}_per_page" !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;
		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}

	/**
	 * Functions tells & sets if current page is one where it's will render.
	 *
	 * @param bool $set current page flag set.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Screen callback.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function render_callback() {
		$current_sub_tab = ( ! empty( $_GET['sub_tab'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['sub_tab'] ) ) : 'posts'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( isset( $current_sub_tab ) && 'cat_tag' === $current_sub_tab ) {
			CategoryRule::instance()->render_callback();
		} elseif ( isset( $current_sub_tab ) && 'default_settings' === $current_sub_tab ) {
			$this->bb_default_settings_form_output();
		} else {
			$this->bb_posts_form_output();
		}
	}

	/**
	 * Load the post screen list.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function bb_posts_form_output() {
		global $plugin_page;

		// Display message once single rule and multiple rule remove.
		$message    = '';
		$style_attr = 'none';
		$page_get   = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$tab_get    = ( ! empty( $_REQUEST['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$deleted    = ( ! empty( $_REQUEST['deleted'] ) ) ? absint( $_REQUEST['deleted'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( ! empty( $page_get ) && 'bbapp-access-controls' === $page_get && ! empty( $tab_get ) && 'posts' === $tab_get && ! empty( $deleted ) ) {
			$style_attr = 'block';
			$message    = ( $deleted > 1 ) ? esc_html__( 'All access rules has been successfully removed.', 'buddyboss-app' ) : esc_html__( 'The access rule has been successfully removed.', 'buddyboss-app' );
		}
		?>
		<div class="bb-access-ctrls-rules-saved updated" style="display:<?php echo esc_attr( $style_attr ); ?>"><p><?php echo esc_html( $message ); ?></p>
		</div>
		<form method="get" class="bbapp-access-rules-form bbapp-access-rules-list bbapp-posts">
			<p><?php echo sprintf( '%1$s <a href="%2$s">%3$s</a>.', esc_html__( 'Create rules to restrict access which members can access your', 'buddyboss-app' ), esc_url( admin_url( 'edit.php' ) ), esc_html__( 'posts', 'buddyboss-app' ) ); ?></p>
			<input type="hidden" name="page" value="<?php echo esc_attr( $plugin_page ); ?>"/>
			<input type="hidden" name="tab" value="<?php echo esc_attr( Core::instance()->get_current_tab() ); ?>"/>
			<input type="hidden" name="item_type" value="<?php echo esc_attr( $this->posts_list->post_type ); ?>"/>
			<?php
			$this->posts_list->check_table_filters();
			$this->posts_list->prepare_items();
			$this->posts_list->search_box( 'Search', 'search' );
			$this->posts_list->display();
			echo sprintf( '<p class="bottom-note">%1$s <strong>%2$s</strong> %3$s</p>', esc_html__( '* Member requires access to', 'buddyboss-app' ), esc_html__( 'all', 'buddyboss-app' ), esc_html__( 'these access groups', 'buddyboss-app' ) );
			?>
		</form>
		<?php
	}

	/**
	 * Get Posts default settings.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_default_settings_form_output() {
		$settings                          = $this->get_settings();
		$label['singular']                 = _x( 'Post', 'access controls integration posts list', 'buddyboss-app' );
		$label['plural']                   = _x( 'Posts', 'access controls integration posts list', 'buddyboss-app' );
		$label['dir_visibility_message']   = __( 'restricted posts in the blog directory', 'buddyboss-app' );
		$label['block_visibility_message'] = __( 'restricted posts in blog post block', 'buddyboss-app' );
		require bbapp()->plugin_dir . 'views/access-controls/default-settings.php';
	}

	/**
	 * Function to get settings.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function get_settings() {
		$settings = AccessRule::instance()->get_access_rule( 'settings', 'default_post' );

		return ! empty( $settings['rule_data'] ) ? $settings['rule_data'] : array();
	}

	/**
	 * Function to update Access default setting.
	 *
	 * @since 1.5.2.1
	 */
	public function save_default_setting() {
		$result      = array();
		$_post       = filter_input_array( INPUT_POST, FILTER_DEFAULT );
		$screen_type = ! empty( $_post['screen-type'] ) ? $_post['screen-type'] : '';
		if ( $this->screen_name === $screen_type ) {
			$data = array(
				'screen_type'             => $screen_type,
				'_wpnonce'                => $_post['_wpnonce'],
				'access_control_settings' => $_post['access_control_settings'],
			);

			$validate = true;
			$messages = array();

			/**
			 * Validations.
			 */
			if ( empty( $data['_wpnonce'] ) || ! wp_verify_nonce( $data['_wpnonce'], 'bbapp-access-default-settings' ) ) {
				$validate   = false;
				$messages[] = __( 'Security check failed.', 'buddyboss-app' );
			}

			if ( isset( $access_control_settings['restrict_all']['enabled'] ) ) {
				if ( ! empty( $data['access_control_settings']['rule_condition'] ) && 'all-logged-in-members' !== $data['access_control_settings']['rule_condition'] ) {
					if ( ! empty( $data['access_control_settings']['rule']['groups_ids'] ) ) {
						if ( ! array_filter( $data['access_control_settings']['rule']['groups_ids'] ) ) {
							$validate   = false;
							$messages[] = __( 'Select at least one access group.', 'buddyboss-app' );
						}
					}
				}
				if ( 'custom' === $data['access_control_settings']['restricted_message']['source'] ) {
					if ( empty( $data['access_control_settings']['restricted_message']['content']['content'] ) ) {
						$validate   = false;
						$messages[] = __( 'Enter the content for the restricted message.', 'buddyboss-app' );
					}
					if ( 'no_button' !== $data['access_control_settings']['restricted_message']['content']['button_action'] ) {
						if ( empty( $data['access_control_settings']['restricted_message']['content']['button_text'] ) ) {
							$validate   = false;
							$messages[] = __( 'Enter the text for button.', 'buddyboss-app' );
						}
						if ( 'app_page_id' === $data['access_control_settings']['restricted_message']['content']['button_action'] && empty( $_post[ $data['access_control_settings']['restricted_message']['content']['button_action'] ] ) ) {
							$validate   = false;
							$messages[] = __( 'Select an app page.', 'buddyboss-app' );
						}
						if ( 'iap_product_id' === $data['access_control_settings']['restricted_message']['content']['button_action'] && empty( $_post[ $data['access_control_settings']['restricted_message']['content']['button_action'] ] ) ) {
							$validate   = false;
							$messages[] = __( 'Select an app product.', 'buddyboss-app' );
						}
					}
				}
			}

			if ( ! $validate ) {
				$result['messages'] = $messages;
				$result['message']  = __( 'Validation errors.', 'buddyboss-app' );
				wp_send_json_error( $result );
			} else {

				// Default setting data.
				$settings['restrict_all']['enabled'] = ! empty( $data['access_control_settings']['restrict_all']['enabled'] ) ? $data['access_control_settings']['restrict_all']['enabled'] : 0;
				$settings['directory_visibility']    = ! empty( $data['access_control_settings']['directory_visibility'] ) ? $data['access_control_settings']['directory_visibility'] : 'show';
				$settings['block_visibility']        = ! empty( $data['access_control_settings']['block_visibility'] ) ? $data['access_control_settings']['block_visibility'] : 'show';

				// Default setting access rule data.
				$_prepare_settings = AccessRules::instance()->bb_prepare_access_control_settings( $data['access_control_settings']['rule'], $_post );
				if ( 0 === $settings['restrict_all']['enabled'] ) {
					$_prepare_settings['rule_condition']  = '';
					$_prepare_settings['group_condition'] = '';
					$_prepare_settings['group_ids']       = array();
					$_prepare_settings['data']            = array();
				}

				$_prepare_settings['data']['default_setting'] = $settings;
				$saved                                        = AccessRule::instance()->save_access_rules( array( 'settings' ), 'default_post', $_prepare_settings );

				if ( $saved ) {
					$result['message'] = __( 'Your settings have been saved.', 'buddyboss-app' );
					wp_send_json_success( $result );

				} else {
					$result['message'] = __( 'There was a problem saving the settings.', 'buddyboss-app' );
					wp_send_json_error( $result );
				}
			}
		}
	}

	/**
	 * Function to get the access inherited from.
	 *
	 * @param int    $item_id   item id.
	 * @param string $item_type item type.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|array
	 */
	public function get_access_inherited_from( $item_id, $item_type ) {
		$inherited = array(
			'type' => null,
			'data' => array(),
		);
		$rule_data = AccessRule::instance()->get_access_rule( $item_id, $item_type );
		if ( ! empty( $rule_data ) ) {
			$inherited['type'] = 'item';
			$inherited['data'] = $rule_data;

			return $inherited;
		}

		$stored_terms = CategoryRule::instance()->get_stored_category_access_rules();
		if ( ! empty( $stored_terms ) ) {
			foreach ( $stored_terms as $stored_term ) {
				$args = array(
					'post_type' => $this->post_type,
					'include'   => array( $item_id ),
					'tax_query' => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
						array(
							'taxonomy' => $stored_term->taxonomy,
							'field'    => 'term_id',
							'terms'    => $stored_term->term_id,
						),
					),
				);

				$found_posts = get_posts( $args );
				if ( ! empty( $found_posts ) ) {
					$tax_rule_data = AccessRule::instance()->get_access_rule( $stored_term->term_id, CategoryRule::instance()->item_type );
					if ( ! empty( $tax_rule_data ) ) {
						$inherited['type'] = 'taxonomy';
						$inherited['data'] = $tax_rule_data;
						$inherited['term'] = $stored_term;

						return $inherited;
					}
				}
			}
		}

		$inherited['type'] = 'default';
		$rule_data         = $this->get_settings();
		if ( isset( $rule_data['default_setting']['restrict_all']['enabled'] ) && 0 === $rule_data['default_setting']['restrict_all']['enabled'] ) {
			$rule_data = array();
		}
		$inherited['data']['rule_data'] = $rule_data;

		return $inherited;
	}

	/**
	 * Function to get general settings on rest.
	 *
	 * @param array        $access_controls Access control array data.
	 * @param object|array $request         Rest request.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function get_default_settings( $access_controls, $request ) {
		$settings               = $this->get_settings();
		$restricted_message     = ! empty( $settings['restricted_message'] ) ? $settings['restricted_message'] : array();
		$rest_response_settings = array(
			'restrict_message'     => bb_access_controls_get_restrict_message( $restricted_message ),
			'directory_visibility' => ! empty( $settings['default_setting']['directory_visibility'] ) && 'show' === $settings['default_setting']['directory_visibility'],
			'block_visibility'     => ! empty( $settings['default_setting']['block_visibility'] ) && 'show' === $settings['default_setting']['block_visibility'],
		);

		$access_controls['posts'] = $rest_response_settings;

		return $access_controls;
	}

	/**
	 * This function return show or hide based on setting if set.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function get_directory_visibility() {
		$settings = $this->get_settings();

		return ! empty( $settings['default_setting']['directory_visibility'] ) ? $settings['default_setting']['directory_visibility'] : 'show';
	}

	/**
	 * This function return show or hide based on setting if set.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function get_block_visibility() {
		$settings = $this->get_settings();

		return ! empty( $settings['default_setting']['block_visibility'] ) ? $settings['default_setting']['block_visibility'] : 'show';
	}

	/**
	 * Function to get bb access.
	 *
	 * @param array             $bb_access Access rule format for rest.
	 * @param \WP_REST_Response $response  The response object.
	 * @param \WP_Post          $post      Post object.
	 * @param \WP_REST_Request  $request   Request object.
	 *
	 * @since 1.5.2.1
	 * @return array|mixed
	 */
	public function get_bb_access( $bb_access, $response, $post, $request ) {
		if ( $this->post_type === $post->post_type ) {
			$rule_data = $this->get_access_inherited_from( $post->ID, $post->post_type );

			if ( ! empty( $rule_data['data'] ) ) {
				$rule_data = $rule_data['data'];
			}

			$can_access = true;
			// Check if stored access group conditions are still exist in code.
			if ( ! empty( $rule_data['rule_data'] ) ) {
				$can_access = bb_access_controls_user_can_access_rule( $rule_data['rule_data'] );
			}
			$bb_access['can_access']       = $can_access;
			$bb_access['restrict_message'] = ( true !== $can_access && isset( $rule_data['rule_data']['restricted_message'] ) ) ? bb_access_controls_get_restrict_message( $rule_data['rule_data']['restricted_message'] ) : null;
		}

		return $bb_access;
	}

	/**
	 * Function to allow post type to provide visibility post_where.
	 *
	 * @param array $post_types Post types.
	 *
	 * @since 1.6.3
	 * @return array|mixed
	 */
	public function allow_post_types_filter( $post_types ) {

		return array_merge( $post_types, array( $this->post_type ) );
	}

	/**
	 * Function to get default rule based on post type.
	 *
	 * @param array  $default_rule Default rule.
	 * @param string $post_type    Post type.
	 *
	 * @since 1.6.3
	 * @return array|mixed
	 */
	public function get_default_rule( $default_rule, $post_type ) {
		if ( $this->post_type !== $post_type ) {
			return $default_rule;
		}

		return $this->get_settings();
	}

	/**
	 * Function to get default rule based on post type.
	 *
	 * @param string $term_item_type Term item type name.
	 * @param string $post_type      Post type.
	 *
	 * @since 1.6.3
	 * @return array|mixed
	 */
	public function get_term_item_type( $term_item_type, $post_type ) {
		if ( $this->post_type !== $post_type ) {
			return $term_item_type;
		}

		return CategoryRule::instance()->item_type;
	}

	/**
	 * Function to pass access control filter.
	 *
	 * @param array $app_page_data Array of app page data.
	 * @param array $block_data    Block data.
	 *
	 * @since 1.6.3
	 * @return array|mixed
	 */
	public function add_block_visibility_fitler( $app_page_data, $block_data ) {
		if ( 'bbapp/posts' === $block_data['blockName'] ) {
			if ( isset( $app_page_data['data'] ) && isset( $app_page_data['data']['data_source'] ) && isset( $app_page_data['data']['data_source']['request_params'] ) ) {
				$visibility = $this->get_block_visibility();

				$app_page_data['data']['data_source']['request_params']['access_control_filter'] = ! ( 'show' === $visibility );
			}
		}

		return $app_page_data;
	}

	/**
	 * Function to pass access control filter.
	 *
	 * @param array            $args    Array of arguments for WP_Query.
	 * @param \WP_REST_Request $request The REST API request.
	 *
	 * @since 1.6.3
	 * @return array|mixed
	 */
	public function bb_access_controls_visibility_filter( $args, $request ) {
		if ( $request->get_param( 'access_control_filter' ) ) {
			$args['access_control_filter'] = ! ( 'false' === $request->get_param( 'access_control_filter' ) );
		} else {
			$visibility                    = $this->get_directory_visibility();
			$args['access_control_filter'] = ! ( 'show' === $visibility );
		}

		return $args;
	}

	/**
	 * Run the job to store category to tag in item meta.
	 *
	 * @param string $post_type Post type name.
	 * @param string $item_type Category item type.
	 *
	 * @since 1.6.3
	 */
	public function run_job_after_taxonomy_order_update( $post_type, $item_type ) {
		if ( $this->post_type === $post_type ) {
			\bb_access_controls_run_terms_calculation_job( $this->post_type );
		}
	}

	/**
	 * Run the job when new term added/update.
	 *
	 * @param array  $item_ids  Item id from access rule.
	 * @param string $item_type Item type from access rule.
	 *
	 * @since 1.6.3
	 */
	public function run_job_after_save_access_rules( $item_ids, $item_type ) {

		if ( CategoryRule::instance()->item_type === $item_type ) {
			$taxonomies = CategoryRule::instance()->bb_get_taxonomy_list();
			$args       = array();
			foreach ( $item_ids as $item_id ) {
				$args = $this->prepare_args_for_taxonomy_post( $item_id, $taxonomies );
			}
			if ( ! empty( $args ) ) {
				$found_posts = get_posts( $args );
				bb_access_controls_run_store_meta_job_by_ids( $found_posts, array( 'post_type' => $this->post_type ) );
			}
		}
		if ( $this->post_type === $item_type ) {
			Actions::instance()->update_store_meta_ids( $item_ids, $item_type );
		}
	}

	/**
	 * Run the job when item delete.
	 *
	 * @param string|int $item_id   Item id from access rule.
	 * @param string     $item_type Item type from access rule.
	 *
	 * @since 1.6.3
	 */
	public function run_job_after_delete_item( $item_id, $item_type ) {
		if ( $this->post_type === $item_type && ! empty( $item_id ) ) {
			bb_access_controls_run_store_meta_job_by_ids( array( $item_id ), array( 'post_type' => $this->post_type ) );
		}
	}

	/**
	 * Run the job when term delete.
	 *
	 * @param string|int $item_id Item id from access rule.
	 *
	 * @since 1.6.3
	 */
	public function run_job_after_delete_category_item( $item_id ) {
		if ( ! empty( $item_id ) ) {
			$args        = $this->prepare_args_for_taxonomy_post( $item_id );
			$found_posts = get_posts( $args );
			bb_access_controls_run_store_meta_job_by_ids(
				$found_posts,
				array(
					'deleted'   => true,
					'item_id'   => $item_id,
					'post_type' => $this->post_type,
				)
			);
		}
	}

	/**
	 * Prepare args function to get taxonomy by post ids.
	 *
	 * @param string $item_id    Item id.
	 * @param array  $taxonomies Taxonomy list array.
	 *
	 * @since  1.6.3
	 * @return array
	 */
	public function prepare_args_for_taxonomy_post( $item_id, $taxonomies = array() ) {
		$tax_query = array();
		if ( ! empty( $taxonomies ) ) {
			$taxonomies = CategoryRule::instance()->bb_get_taxonomy_list();
		}
		if ( ! empty( $taxonomies ) ) {
			$tax_query['relation'] = 'OR';
			foreach ( $taxonomies as $taxonomy ) {
				$tax_query[] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'term_id',
					'terms'    => $item_id,
				);
			}
		}

		return array(
			'post_type'      => $this->post_type,
			'fields'         => 'ids',
			'tax_query'      => $tax_query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			'posts_per_page' => - 1,
		);
	}

	/**
	 * Added and remove category from post.
	 *
	 * @param int    $object_id  Object ID.
	 * @param array  $terms      An array of object term IDs or slugs.
	 * @param array  $tt_ids     An array of term taxonomy IDs.
	 * @param string $taxonomy   Taxonomy slug.
	 * @param bool   $append     Whether to append new terms to the old terms.
	 * @param array  $old_tt_ids Old array of term taxonomy IDs.
	 *
	 * @since 1.6.3
	 */
	public function run_job_after_category_add_or_remove( $object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids ) {
		$taxonomies = CategoryRule::instance()->bb_get_taxonomy_list();
		if ( in_array( $taxonomy, $taxonomies, true ) ) {
			bb_access_controls_run_store_meta_job_by_ids( array( $object_id ), array( 'post_type' => $this->post_type ) );
		}
	}

	/**
	 * Prepare inherited from data.
	 *
	 * @param array      $inherited Inherited from data.
	 * @param int|string $post_id   Post id or item id.
	 * @param string     $post_type Post type.
	 *
	 * @return array|string
	 */
	public function prepare_job_get_access_inherited_from( $inherited, $post_id, $post_type ) {
		if ( $this->post_type === $post_type ) {
			return $this->get_access_inherited_from( $post_id, $post_type );
		}

		return $inherited;
	}

}
