<?php
/**
 * Holds page access rule list render.
 *
 * @package BuddyBossApp\AccessControls\Core\Settings
 */

namespace BuddyBossApp\AccessControls\Core\Settings;

use BuddyBossApp\AccessControls\Core\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class Pages
 */
class Pages {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Determine if current page is page list.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * PAge list.
	 *
	 * @var $pages_list
	 */
	public $pages_list;

	/**
	 * Post type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $post_type = 'page';

	/**
	 * Screen type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $screen_name = 'pages';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return Pages
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Pages constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );

		// Screen options.
		add_action( 'load-buddyboss-app_page_bbapp-access-controls', array( $this, 'get_screen_options' ), 99 );
		add_filter( 'set-screen-option', array( $this, 'bb_access_rules_screen_option' ), 10, 3 );
	}

	/**
	 * Register the screen options.
	 *
	 * @since 1.5.5
	 */
	public function get_screen_options() {
		$screen  = get_current_screen();
		$tab_get = ( ! empty( $_GET['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-access-controls' !== $screen->id || $this->screen_name !== $tab_get ) {
			return;
		}

		// Loop through all the options and add a screen option for each.
		add_screen_option(
			'per_page',
			array(
				'label'   => _x( 'Number of Pages per page:', 'access controls list table', 'buddyboss-app' ),
				'default' => 10,
				'option'  => "bb_access_controls_{$this->post_type}_per_page",
			)
		);
	}

	/**
	 * Handle save/update of screen options for Pages.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @since 1.5.5
	 *
	 * @return string|int Option value. False to abandon update.
	 */
	public function bb_access_rules_screen_option( $value, $option, $new_value ) {
		if ( "bb_access_controls_{$this->post_type}_per_page" !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;
		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}

	/**
	 * Functions tells & sets if current page is one where it's will render.
	 *
	 * @param bool $set current page flag set.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Load Pages custom List.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function admin_init() {
		if ( $this->will_render() ) {
			$this->pages_list = new PagesList();
			$this->pages_list->prepare_items();
		}
	}

	/**
	 * Screen callback.
	 *
	 * @since 1.5.2.1
	 */
	public function render_callback() {
		global $plugin_page;

		// Display message once single rule and multiple rule remove.
		$message    = '';
		$style_attr = 'none';
		$page_get   = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$tab_get    = ( ! empty( $_REQUEST['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$deleted    = ( ! empty( $_REQUEST['deleted'] ) ) ? absint( $_REQUEST['deleted'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( ! empty( $page_get ) && 'bbapp-access-controls' === $page_get && ! empty( $tab_get ) && 'pages' === $tab_get && ! empty( $deleted ) ) {
			$style_attr = 'block';
			$message    = ( $deleted > 1 ) ? esc_html__( 'All access rules has been successfully removed.', 'buddyboss-app' ) : esc_html__( 'The access rule has been successfully removed.', 'buddyboss-app' );
		}
		?>
		<div class="bb-access-ctrls-rules-saved updated" style="display:<?php echo esc_attr( $style_attr ); ?>"><p><?php echo esc_html( $message ); ?></p></div>
		<form method="get" class="bbapp-access-rules-form bbapp-access-rules-list bbapp-pages">
			<p><?php printf( '%1$s <a href="%2$s">%3$s</a>.', esc_html__( 'Create rules to restrict access which members can access your', 'buddyboss-app' ), esc_url( admin_url( 'edit.php?post_type=page' ) ), esc_html__( 'pages', 'buddyboss-app' ) ); ?></p>
			<input type="hidden" name="page" value="<?php echo esc_attr( $plugin_page ); ?>"/>
			<input type="hidden" name="tab" value="<?php echo esc_attr( Core::instance()->get_current_tab() ); ?>"/>
			<input type="hidden" name="item_type" value="<?php echo esc_attr( $this->pages_list->post_type ); ?>"/>
			<?php
			$this->pages_list->check_table_filters();
			$this->pages_list->prepare_items();
			$this->pages_list->search_box( 'Search', 'search' );
			$this->pages_list->display();
			printf( '<p class="bottom-note">%1$s <strong>%2$s</strong> %3$s</p>', esc_html__( '* Member requires access to', 'buddyboss-app' ), esc_html__( 'all', 'buddyboss-app' ), esc_html__( 'these access groups', 'buddyboss-app' ) );
			?>
		</form>
		<?php
	}
}
