<?php
/**
 * Holds WP core ralted access controls functionality.
 *
 * @package BuddyBossApp\AccessControls\Core\Settings
 */

namespace BuddyBossApp\AccessControls\Core\Settings;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\AccessControls\Core\Core;
use BuddyBossApp\AccessControls\Helpers;
use BuddyBossApp\Helpers\Custom_Multiselect_Dropdown;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class General
 */
class General {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Screen type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $screen_name = 'general';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return General
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * General constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action( 'admin_footer', array( $this, 'load_script' ), 999 );
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );

		// Save data.
		add_action( 'wp_ajax_bb_access_controls_update_general_data', array( $this, 'update_general_data' ) );
		add_filter( 'bbapp_settings_rest_response_access_controls', array( $this, 'bbapp_restrict_response' ), 10, 2 );
	}

	/**
	 * Load Access Controls.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function admin_init() {
	}

	/**
	 * Function to get settings.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function get_settings() {
		$settings = AccessRule::instance()->get_access_rule( 'settings', 'general' );
		$settings = ! empty( $settings['rule_data'] ) ? $settings['rule_data'] : array();

		return $settings;
	}

	/**
	 * Screen callback.
	 *
	 * @since 1.5.2.1
	 */
	public function render_callback() {
		$settings        = $this->get_settings();
		$default_setting = array();
		if ( ! empty( $settings['default_setting'] ) ) {
			$default_setting = $settings['default_setting'];
			unset( $settings['default_setting'] );
		}
		$restrict_access_enabled = ! empty( $default_setting['restrict_access']['enabled'] ) ? $default_setting['restrict_access']['enabled'] : 0;
		$restrict_action         = ! empty( $default_setting['restrict_action'] ) ? $default_setting['restrict_action'] : 'login_screen';
		$restrict_page_id        = ! empty( $default_setting['restrict_page_id'] ) ? $default_setting['restrict_page_id'] : 0;
		?>
		<div class="bbapp-general-err-msgs"></div>
		<form method="post" name="bbapp-access-control-general-form" id="bbapp-access-control-general-form">
			<input type="hidden" name="action" value="general_setting">
			<input type="hidden" name="_wpnonce" value="<?php echo esc_attr( wp_create_nonce( 'bbapp-save-general-setting' ) ); ?>">
			<input type="hidden" name="bbapp-error-msg" value="<?php echo esc_html_x( 'Please resolve the errors below to save your settings.', 'access controls general settings', 'buddyboss-app' ); ?>">

			<table class="form-table">
				<tbody>
				<tr class="bbapp_field_wrapper">
					<th scope="row" style="width:250px">
						<label><?php echo esc_html_x( 'Restrict Access', 'access controls general settings', 'buddyboss-app' ); ?></label>
					</th>
					<td>
						<input id="bbapp_field_app_restrict_access_enabled" class="regular-text bbapp_field_app_restrict_access_enabled" name="access_control_settings[restrict_access][enabled]" type="checkbox" <?php checked( $restrict_access_enabled, true ); ?> value="1">
						<label for="bbapp_field_app_restrict_access_enabled">
							<?php echo esc_html_x( 'Restrict who can login to the BuddyBoss App', 'access controls general settings', 'buddyboss-app' ); ?>
						</label>
						<p class="description bbapp-admin-description">
							<?php echo esc_html_x( 'When enabled, only members who are in the specified access groups will be able to view app\'s content after logging in.', 'General screen access rule', 'buddyboss-app' ); ?>
						</p>
					</td>
				</tr>
				<?php
				$allowed_html = wp_kses_allowed_html( 'post' );

				$allowed_html['a']['onclick'] = true;
				$allowed_html['textarea']     = array(
					'cols'         => true,
					'rows'         => true,
					'disabled'     => true,
					'name'         => true,
					'readonly'     => true,
					'id'           => true,
					'class'        => true,
					'data-err-msg' => true,
				);
				$allowed_html['script']       = array();
				$allowed_html['select']       = array(
					'id'                    => true,
					'class'                 => true,
					'name'                  => true,
					'data-nonce'            => true,
					'disabled'              => true,
					'data-slug'             => true,
					'data-err-msg'          => true,
					'data-option-disable'   => true,
					'data-disabled-options' => true,
					'data-select-key'       => true,
					'data-extras'           => true,
				);
				$allowed_html['option']       = array(
					'value'    => true,
					'selected' => true,
					'disabled' => true,
				);
				$allowed_html['input']        = array(
					'id'           => true,
					'class'        => true,
					'name'         => true,
					'type'         => true,
					'checked'      => true,
					'value'        => true,
					'data-err-msg' => true,
				);

				echo wp_kses(
					bb_access_controls_get_rule_fields(
						'access_control_settings[rule]',
						'table',
						$settings,
						array(),
						array(
							'restrict_to_visible'        => false,
							'restricted_message_visible' => false,
						)
					),
					$allowed_html,
					array( 'javascript' )
				);
				?>
				<tr class="bbapp_field_wrapper bbapp_restricted hidden">
					<th scope="row" style="width:250px">
						<label><?php echo esc_html_x( 'When Restricted', 'access controls general settings', 'buddyboss-app' ); ?></label>
					</th>
					<td>
						<?php
						if ( 'iap_products' === $restrict_action && ! bbapp_is_active( 'iap' ) ) {
							$restrict_action = 'login_screen';
						}
						if ( 'app_page' === $restrict_action && 'publish' !== get_post_status( $restrict_page_id ) ) {
							$restrict_action = 'login_screen';
						}
						$fields = array(
							'login_screen' => array(
								'label' => _x( 'Return to login', 'access controls general settings', 'buddyboss-app' ),
							),
							'iap_products' => array(
								'label' => _x( 'Show Products Screen', 'access controls general settings', 'buddyboss-app' ),
							),
							'app_page'     => array(
								'label' => _x( 'Show App Page', 'access controls general settings', 'buddyboss-app' ),
							),
						);
						?>
						<fieldset>
							<?php foreach ( $fields as $key => $field ) { ?>
								<?php
								if ( ! bbapp_is_active( 'iap' ) && 'iap_products' === $key ) {
									continue;
								}
								?>
								<label>
									<input class="access_control_settings_restrict_action" name="access_control_settings[restrict_action]" type="radio" <?php checked( $restrict_action, $key ); ?>value="<?php echo esc_attr( $key ); ?>">
									<?php echo esc_html( $field['label'] ); ?>
								</label><br>
							<?php } ?>
							<label class="access_control_settings_restrict_action_app_page" style="display: none;">
								<?php
								$app_pages  = get_posts(
									array(
										'post_type'   => 'app_page',
										'numberposts' => - 1,
									)
								);
								$field_name = 'access_control_settings[restrict_page_id]';
								$id         = 'access_control_settings_restrict_action_app_page';
								?>
								<div class="bbapp-apppage-field">
									<select name="<?php echo esc_attr( "{$field_name}" ); ?>"
											id="<?php echo esc_attr( "{$id}" ); ?>"
											data-err-msg="<?php echo esc_attr_x( 'Select an app page.', 'access controls general settings', 'buddyboss-app' ); ?>"
									>
										<option value="">
											<?php echo esc_html_x( '-- Select App Page --', 'select app page access control general settings', 'buddyboss-app' ); ?>
										</option>
										<?php
										if ( ! empty( $app_pages ) ) {
											foreach ( $app_pages as $app_page ) {
												?>
												<option <?php selected( $restrict_page_id, $app_page->ID ); ?>
														value="<?php echo esc_attr( $app_page->ID ); ?>">
													<?php echo esc_html( $app_page->post_title ); ?>
												</option>
												<?php
											}
										}
										?>
									</select>
									<div class="bbapp_loading dark" style="display: none;"></div>
									<?php $is_hidden = ( ! empty( AccessRule::instance()->get_access_rule( $restrict_page_id, 'app_page' ) ) && 'publish' === get_post_status( $restrict_page_id ) ) ? '' : 'hidden'; ?>
									<p class="bbapp-notice-box bbapp-notice-box--warning <?php echo esc_attr( $is_hidden ); ?>">
										<?php echo esc_html_x( 'This app page is restricted to certain members. We recommend removing any access rules from a page used in this setting.', 'access controls general settings', 'buddyboss-app' ); ?>
									</p>
								</div>
							</label>
						</fieldset>

					</td>
				</tr>
				</tbody>
			</table>
			<button type="submit" name="submit"
					value="submit-general"
					class="button button-primary submitbtn access_control_settings_general">
				<?php echo esc_html_x( 'Save Changes', 'access controls general settings', 'buddyboss-app' ); ?>
			</button>
			<div class="bbapp_loading dark" style="display: none;"></div>
			<div class="bbapp-ie__footer">
				<?php
				$class = 'off';
				$text  = _x( 'OFF', 'access controls general settings', 'buddyboss-app' );
				$desc  = sprintf(
				/* translators: %s: Link to BuddyBoss App settings page. */
					__( 'To restrict all your app\'s content to logged-in members, enable the <a href="%s">Private App</a> setting. When enabled, logged-out members will only be able to access the login and registration screens.', 'buddyboss-app' ),
					admin_url( 'admin.php?page=bbapp-settings' )
				);
				if ( bbapp_is_private_app_enabled() ) {
					$class = 'on';
					$text  = _x( 'ON', 'access controls general settings', 'buddyboss-app' );
					$desc  = sprintf(
					/* translators: %s: Link to BuddyBoss App settings page. */
						__( 'Your app\'s content is restricted to logged-in members. Logged-out members will only be able to access the login and registration screens. To make your app\'s content public by default, disable the <a href="%s">Private App</a> setting.', 'buddyboss-app' ),
						admin_url( 'admin.php?page=bbapp-settings' )
					);
				}
				?>
				<h4>
					<?php echo esc_html_x( 'Private App', 'access controls general settings', 'buddyboss-app' ); ?>
					<span class="bbapp-status bbapp-status--access <?php echo esc_attr( $class ); ?>">
						<?php echo esc_html( $text ); ?>
					</span>
				</h4>
				<p><?php echo wp_kses_post( $desc ); ?></p>
			</div>
		</form>
		<?php
	}

	/**
	 * Function to load script.
	 *
	 * @since 1.5.2.1
	 */
	public function load_script() {
		$current_tab = Core::instance()->get_current_tab();
		if ( 'general' !== $current_tab ) {
			return;
		}
		?>
		<script>
			jQuery( document ).ready( function() {
				// Run the menu access control select.
				jQuery( document ).bind( 'bbapp_dropdown_after_clone', function() {
					jQuery( '.bbapp-access-groups' ).last().val( [ '0' ] ).trigger( 'change' );
				} );

				jQuery( '.access_control_settings_restrict_action' ).each( function( index, element ) {
					if ( jQuery( element ).is( ':checked' ) ) {
						whenRestrictedBox( jQuery( element ).is( ':checked' ), jQuery( element ).val() );
					}
				} );

				jQuery( document ).on( 'change', '.access_control_settings_restrict_action', function() {
					whenRestrictedBox( jQuery( this ).is( ':checked' ), jQuery( this ).val() );
				} );

				function whenRestrictedBox( checked, val ) {
					if ( checked ) {
						if ( 'app_page' === val ) {
							jQuery( '.access_control_settings_restrict_action_app_page' ).css( 'display', 'inherit' ).addClass( 'bb-validate-field-cmn' );
							jQuery( '#access_control_settings_restrict_action_app_page' ).addClass( 'bb-validate-field-cmn' );
						} else {
							jQuery( '.access_control_settings_restrict_action_app_page' ).css( 'display', 'none' );
							jQuery( '#access_control_settings_restrict_action_app_page' ).removeClass( 'bb-validate-field-cmn' );
						}
					}
				}

				restrictedBox( jQuery( '.bbapp_field_app_restrict_access_enabled' ).prop( 'checked' ) );

				jQuery( document ).on( 'click', '.bbapp_field_app_restrict_access_enabled', function() {
					restrictedBox( jQuery( this ).prop( 'checked' ) );
				} );

				/**
				 * Restricted Box
				 *
				 * @param {boolean} checked - Checked.
				 */
				function restrictedBox( checked ) {
					if ( checked ) {
						jQuery( '.bbapp_restricted' ).removeClass( 'hidden' );
						jQuery( '.bb-select' ).addClass( 'bb-validate-field-cmn' );
					} else {
						jQuery( '.bbapp_restricted' ).addClass( 'hidden' );
						jQuery( '.bb-select' ).removeClass( 'bb-validate-field-cmn' );
					}
				}

				// Access rule General tab
				jQuery( document ).on( 'submit', '#bbapp-access-control-general-form', function( e ) {
					e.preventDefault();
					jQuery( '.bbapp-field-err-cmn' ).remove();

					const form = jQuery( this );
					let validate = true;
					const selectCmn = form.find( '.bb-validate-field-cmn' );
					const errMsgEl = jQuery( '.bbapp-general-err-msgs' );

					errMsgEl.empty();

					const selectCmnIds = jQuery( selectCmn ).map( function() {
						return jQuery( this ).attr( 'id' );
					} );

					jQuery( selectCmnIds ).each( function( index, value ) {
						const selectEl = jQuery( '#' + value );
						const selectVal = selectEl.val();

						if ( 'undefined' === typeof selectVal || '' === selectVal || 0 ===
							parseInt( selectVal ) ) {
							validate = false;
							selectEl.addClass( 'bbapp-has-err' );
							selectEl.after( '<div class="bbapp-field-err-cmn bbapp-field-err">' + selectEl.attr( 'data-err-msg' ) + '</div>' );
						} else if ( selectEl.hasClass( 'bbapp-has-err' ) ) {
							selectEl.removeClass( 'bbapp-has-err' );
							selectEl.parent().find( '.bbapp-field-err-cmn' ).remove();
						}
					} );

					if ( false === validate ) {
						errMsgEl.html( '<div id="message" class="bbapp-error"><p>' + jQuery( 'input[name="bbapp-error-msg"]' ).val() + '</p></div>' ).fadeIn();

						return false;
					}

					const formData = jQuery( form ).serialize();

					updateAccessControlGeneral( formData );
				} );

				/**
				 * Update access general dialog.
				 *
				 * @param {Object} data - Form data.
				 */
				function updateAccessControlGeneral( data = null ) {
					const form = jQuery( '#bbapp-access-control-general-form' );
					const loader = form.find( '#bbapp-loading' );
					const submit = form.find( 'button[type="submit"]' );
					const errMsgEl = jQuery( '.bbapp-general-err-msgs' );
					loader.show();
					submit.prop( 'disabled', true );
					jQuery.ajax( {
						url: ajaxurl,
						type: 'POST',
						data: {
							action: 'bb_access_controls_update_general_data',
							general_data: data,
						},
						beforeSend() {
							form.find( '.bbapp_loading' ).show();
						},
						error( e ) {
							// eslint-disable-next-line no-console
							console.log( 'ERROR: ' + JSON.stringify( e, null, 4 ) );
							form.find( '.bbapp_loading' ).hide();
						},
						success( response ) {
							let messages = '';

							if ( true === response.success ) {
								// show the update message.
								errMsgEl.append( '<div class="bbapp-success"><p>' + response.data.message + '</p></div>' );
							} else {
								// show the update message.
								errMsgEl.append( '<div class="bbapp-success"><p>' + response.data.message + '</p></div>' );
								// show the update message.
								errMsgEl.html( '<div id="message" class="bbapp-error"><p>' + response.data.message + '</p></div>' );

								if ( 'undefined' !== typeof response.data.messages ) {
									messages = '<ul style="list-style-type:disc;padding-left: 25px;">';
									response.data.messages.forEach( function( item ) {
										messages += '<li>' + item + '</li>';
									} );
									messages += '</ul>';
									errMsgEl.find( '.bbapp-error p' ).append( messages );
								}
							}

							errMsgEl.fadeIn();

							// hide the message after certain time.
							if ( true === response.success ) {
								setTimeout( function() {
									errMsgEl.fadeOut(); // hide the save confirmation message.
								}, 5000 );
							}

							loader.hide();
							submit.prop( 'disabled', false );
							form.find( '.bbapp_loading' ).hide();
						},
					} );
				}
			} );
		</script>
		<?php
	}

	/**
	 * Function to update access rule general data.
	 *
	 * @since 1.5.2.1
	 */
	public function update_general_data() {
		$result             = array();
		$result['type']     = 'object';
		$result['messages'] = array();
		$general_data       = ( ! empty( $_POST['general_data'] ) ) ? wp_unslash( $_POST['general_data'] ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		parse_str( $general_data, $general_data_arr );

		if ( ! empty( $general_data_arr['action'] ) && 'general_setting' === $general_data_arr['action'] ) {
			// Do the validation.
			$validate = $this->bbapp_validate_settings( $general_data_arr );

			if ( ! empty( $validate['validate'] ) && true === (bool) $validate['validate'] ) {
				if ( true === (bool) $validate['data']['status'] ) {
					$result['message'] = __( 'Your settings have been saved.', 'buddyboss-app' );
					$result['saved']   = AccessRule::instance()->save_access_rules( array( 'settings' ), 'general', $validate['data']['general_data'] );
					wp_send_json_success( $result );
				} else {
					$result            = $validate;
					$result['message'] = __( 'Please resolve the errors below to save your settings.', 'buddyboss-app' );
					wp_send_json_error( $result );
				}
			} else {
				$result            = $validate;
				$result['message'] = __( 'Validation errors.', 'buddyboss-app' );
				wp_send_json_error( $result );
			}
		}
		wp_send_json_error( $result );
	}

	/**
	 * Function to validate general post data.
	 *
	 * @param array $data_to_validate general data to validate.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function bbapp_validate_settings( $data_to_validate ) {
		$result['validate'] = true;
		$result['data']     = array();
		$result['messages'] = array();
		$action             = ( ! empty( $data_to_validate['action'] ) ) ? bbapp_input_clean( wp_unslash( $data_to_validate['action'] ) ) : '';

		if ( 'general_setting' === $action ) {
			$nonce = ( ! empty( $data_to_validate['_wpnonce'] ) ) ? wp_unslash( $data_to_validate['_wpnonce'] ) : '';

			if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp-save-general-setting' ) ) {
				$result['validate']   = false;
				$result['messages'][] = __( 'Security check failed.', 'buddyboss-app' );
			}

			if ( isset( $data_to_validate['access_control_settings']['restrict_access']['enabled'] ) ) {
				if ( ! empty( $data_to_validate['access_control_settings']['rule']['group_ids'] ) && ! array_filter( $data_to_validate['access_control_settings']['rule']['group_ids'] ) ) {
					$result['validate']   = false;
					$result['messages'][] = __( 'Select at least one access group.', 'buddyboss-app' );
				}

				if ( ! empty( $data_to_validate['access_control_settings']['restrict_action'] ) && 'app_page' === $data_to_validate['access_control_settings']['restrict_action'] && empty( $data_to_validate['access_control_settings']['restrict_page_id'] ) ) {
					$result['validate']   = false;
					$result['messages'][] = __( 'Select an app page.', 'buddyboss-app' );
				}
			}

			if ( true === $result['validate'] ) {
				if ( ! empty( $data_to_validate['access_control_settings'] ) ) {
					$settings_data                                 = array(
						'group_condition' => '',
						'group_ids'       => array(),
					);
					$default_setting                               = array();
					$access_control_settings                       = $data_to_validate['access_control_settings'];
					$default_setting['restrict_access']['enabled'] = ! empty( $access_control_settings['restrict_access']['enabled'] ) ? $access_control_settings['restrict_access']['enabled'] : 0;

					if ( ! empty( $default_setting['restrict_access']['enabled'] ) ) {
						$settings_data['group_condition']   = ! empty( $access_control_settings['rule']['group_condition'] ) ? $access_control_settings['rule']['group_condition'] : 'all';
						$settings_data['group_ids']         = ! empty( $access_control_settings['rule']['group_ids'] ) ? array_filter( array_unique( $access_control_settings['rule']['group_ids'] ) ) : array();
						$default_setting['restrict_action'] = ! empty( $access_control_settings['restrict_action'] ) ? $access_control_settings['restrict_action'] : 'login_screen';
						if ( 'app_page' === $default_setting['restrict_action'] ) {
							$default_setting['restrict_page_id'] = ! empty( $access_control_settings['restrict_page_id'] ) ? $access_control_settings['restrict_page_id'] : 0;
						}
					}

					$settings_data['data']['default_setting'] = $default_setting;
					$result['data']                           = array(
						'status'       => 1,
						'general_data' => $settings_data,
					);
					$result['validate']                       = true;
				}
			}
		}

		return $result;
	}

	/**
	 * Function to get general settings on rest.
	 *
	 * @param array        $access_controls Access control array data.
	 * @param object|array $request         Rest request.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function bbapp_restrict_response( $access_controls, $request ) {
		$settings         = $this->get_settings();
		$can_access       = true;
		$restrict_action  = 'login_screen';
		$restrict_page_id = 0;

		$default_setting = array();
		if ( ! empty( $settings['default_setting'] ) ) {
			$default_setting = $settings['default_setting'];
			unset( $settings['default_setting'] );
		}

		if ( isset( $default_setting['restrict_access']['enabled'] ) && true === (bool) $default_setting['restrict_access']['enabled'] ) {
			$can_access       = bb_access_controls_user_can_access_rule( $settings );
			$restrict_action  = $default_setting['restrict_action'];
			$restrict_page_id = ! empty( $default_setting['restrict_page_id'] ) ? (int) $default_setting['restrict_page_id'] : 0;
		}

		if ( 'iap_products' === $restrict_action && ! bbapp_is_active( 'iap' ) ) {
			$restrict_action = 'login_screen';
		}
		if ( 'app_page' === $restrict_action && 'publish' !== get_post_status( $restrict_page_id ) ) {
			$restrict_action  = 'login_screen';
			$restrict_page_id = 0;
		}

		$access_controls['app'] = array(
			'can_access'       => $can_access,
			'restrict_action'  => $restrict_action,
			'restrict_page_id' => $restrict_page_id,
		);

		return $access_controls;
	}
}
