<?php
/**
 * Holds category rule listing render.
 *
 * @package BuddyBossApp\AccessControls\Core\Settings
 */

namespace BuddyBossApp\AccessControls\Core\Settings;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\Jobs;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class Category rules
 */
class CategoryRule {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Taxonomy List.
	 *
	 * @var $cat_tag_list
	 */
	public $cat_tag_list;

	/**
	 * Item type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $item_type = 'post_term';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return CategoryRule
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Posts constructor.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );
		// prepare rule data.
		add_filter( 'bb_access_controls_update_rule_data_result', array( $this, 'bb_prepare_result_data' ), 10, 4 );
		// Get taxonomy.
		add_action( 'wp_ajax_bb_get_taxonomies', array( $this, 'bb_get_taxonomies' ) );
		// Update category rule order.
		add_action( 'wp_ajax_bb_taxonomy_rules_order', array( $this, 'update_taxonomy_rules_order' ) );

		// Validate and save data.
		add_action(
			"bb_access_controls_after_saved_rule_data_{$this->item_type}",
			array(
				$this,
				'add_category_rule_order',
			),
			10,
			2
		);
	}

	/**
	 * Load posts custom List.
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function admin_init() {
		$tab     = ( ! empty( $_REQUEST['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_tab = ( ! empty( $_REQUEST['sub_tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['sub_tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( Posts::instance()->screen_name === $tab && 'cat_tag' === $sub_tab ) {
			wp_enqueue_script( 'jquery-ui-sortable' );
			// Category list table.
			$this->cat_tag_list = new CategoryRuleList();
			$this->cat_tag_list->prepare_items();
		}
	}

	/**
	 * Screen callback.
	 *
	 * @since 1.5.2.1
	 */
	public function render_callback() {
		global $plugin_page;

		// Display message once single rule and multiple rule remove.
		$message         = '';
		$style_attr      = 'none';
		$page_request    = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$tab_request     = ( ! empty( $_REQUEST['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_tab_request = ( ! empty( $_REQUEST['sub_tab'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['sub_tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$deleted_request = ( ! empty( $_REQUEST['deleted'] ) ) ? absint( $_REQUEST['deleted'] ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $page_request ) && 'bbapp-access-controls' === $page_request && ! empty( $tab_request ) && 'posts' === $tab_request && ! empty( $sub_tab_request ) && 'cat_tag' === $sub_tab_request && ! empty( $deleted_request ) ) {
			$style_attr = 'block';
			$message    = ( $deleted_request > 1 ) ? esc_html__( 'All access rules has been successfully removed.', 'buddyboss-app' ) : esc_html__( 'The access rule has been successfully removed.', 'buddyboss-app' );
		}
		?>
		<div class="bb-access-ctrls-rules-saved updated" style="dispaly:<?php echo esc_attr( $style_attr ); ?>">
			<p><?php echo esc_html( $message ); ?></p>
		</div>
		<div class="bbapp-access-control-categories">
			<?php
			$class = 'hidden';
			if ( true === Jobs::instance()->determine_process_running( 'bb_ac_term_calc' ) || true === Jobs::instance()->determine_process_running( 'bb_store_meta_by_ids' ) ) {
				$class = 'running';
			}
			?>
			<div class="bbapp-notice-box bbapp-notice-box--info bbapp-taxonomy-job <?php echo esc_attr( $class ); ?>">
				<div class="content-loading">
					<div class="bbapp_loading dark"></div>
					<?php echo esc_html__( 'Updating posts with new rules.....', 'buddyboss-app' ); ?>
				</div>
			</div>
			<div class="bbapp-access-control-categories-info"><?php esc_html_e( 'You can create rules to protect all posts that have a specific category, tag or custom taxonomy.', 'buddyboss-app' ); ?></div>
			<div class="bb-notice-box"></div>
			<div class="bbapp-access-filters flex">
				<div class="bbapp-categories-labels">
					<?php
					// Get the registered taxonomies with post type post.
					$taxonomies = get_object_taxonomies( Posts::instance()->post_type, 'objects' );
					$taxonomy   = '';
					$name       = '';
					if ( ! empty( $taxonomies ) ) {
						?>
						<select name="taxonomy" class="bbapp-categories-select">
							<?php
							foreach ( $taxonomies as $key => $value ) {
								// By pass post_format taxonomy.
								if ( 'post_format' === $key ) {
									continue;
								}
								// Set taxonomy if empty.
								if ( empty( $taxonomy ) ) {
									$taxonomy = esc_attr( $key );
								}
								// Set taxonomy name if empty.
								if ( empty( $name ) ) {
									$name = esc_attr( $value->labels->singular_name );
								}
								?>
								<option value="<?php echo esc_attr( $key ); ?>"
										data-slug="<?php echo esc_attr( $key ); ?>"
										data-item-type="<?php echo esc_attr( Posts::instance()->post_type ); ?>"
										data-name="<?php echo esc_attr( $value->labels->singular_name ); ?>">
									<?php echo esc_html( $value->labels->singular_name ); ?>
								</option>
								<?php
							}
							?>
						</select>
					<?php } ?>
				</div>
				<?php if ( ! empty( $taxonomies ) ) { ?>
					<div class="bbapp-categories-dropdown">
						<?php
						$allowed_html           = wp_kses_allowed_html( 'post' );
						$allowed_html['select'] = array(
							'class' => true,
							'id'    => true,
							'name'  => true,
							'value' => true,
							'type'  => true,
						);
						$allowed_html['option'] = array(
							'selected' => true,
							'value'    => true,
							'disabled' => true,
						);
						echo wp_kses( $this->bb_access_controls_get_taxonomies( $taxonomy, $name ), $allowed_html, array( 'javascript' ) );
						?>
					</div>
					<div class="bb-add-rule-btn-wrp">
						<button type="button" class="button bb-access-control-add-tax-rule"
								data-nonce="<?php echo esc_attr( wp_create_nonce( 'bb-add-cat-rule' ) ); ?>">
							<?php echo esc_html_x( '+ Add Rule', 'access controls category rules settings', 'buddyboss-app' ); ?>
						</button>
						<div class="bbapp_loading dark" style="display: none;"></div>
					</div>
				<?php } ?>
			</div>

			<div class="bb-cat-tag-list-wrp">
				<?php $this->category_rule_action_notices(); ?>
				<div class="bb-access-ctrls-rules-saved updated" style="display:none"><p></p></div>
				<form method="get" class="bbapp-access-rules-list bbapp-access-groups-form bbapp-cat-access-rule-list">
					<input type="hidden" name="page" value="<?php echo esc_attr( $plugin_page ); ?>"/>
					<input type="hidden" name="tab" value="posts">
					<input type="hidden" name="sub_tab" value="cat_tag">
					<input type="hidden" name="item_type" value="<?php echo esc_attr( $this->item_type ); ?>"/>
					<?php
					$this->cat_tag_list->get_views();
					$this->cat_tag_list->display();
					?>
				</form>
				<p class="description">
					<?php printf( '%1$s <strong>%2$s</strong> %3$s', esc_html__( '* Member requires access to', 'buddyboss-app' ), esc_html__( 'all', 'buddyboss-app' ), esc_html__( 'these access groups', 'buddyboss-app' ) ); ?>
				</p>
				<div class="bb-cat-footer-note"><?php printf( '<strong>%1$s</strong> %2$s', esc_html__( 'TIP:', 'buddyboss-app' ), esc_html__( 'Drag and drop the rules above to set the priority of which one should be followed if multiple rules apply to a single post.', 'buddyboss-app' ) ); ?></div>
			</div>
		</div>
		<?php
	}

	/**
	 * Function to insert condition data.
	 *
	 * @param string $taxonomy taxonomy slug.
	 * @param string $name     taxonomy name.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string
	 */
	public function bb_access_controls_get_taxonomies( $taxonomy = 'category', $name = 'Category' ) {
		$args = array(
			'taxonomy'   => $taxonomy,
			'fields'     => 'id=>name',
			'hide_empty' => false,
		);

		$term_query   = new \WP_Term_Query( $args );
		$stored_terms = $this->get_stored_category_access_rules( array( $taxonomy ) );
		$term_ids     = ( ! empty( $stored_terms ) ) ? wp_list_pluck( $stored_terms, 'term_id' ) : array();
		$html         = '';

		ob_start();
		?>
		<select name="cat" id="cat" class="postform">
			<option value="0">
				<?php
				/* translators: %1$s: Taxonomy name. */
				printf( esc_html__( '-- Select %1$s --', 'buddyboss-app' ), esc_html( $name ) );
				?>
			</option>
			<?php
			if ( ! empty( $term_query->get_terms() ) ) {
				foreach ( $term_query->get_terms() as $key => $term ) {
					$exist = in_array( $key, $term_ids, true );
					printf( '<option value="%1$s" %2$s>%3$s</option>', esc_attr( $key ), disabled( $exist, true, false ), esc_html( $term ) );
				}
			}
			?>
		</select>
		<?php
		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Modified result while saving ld category access rules.
	 *
	 * @param array  $result    Fetch result before modified.
	 * @param array  $item_ids  Id of the access rule.
	 * @param string $item_type Type of the access rule.
	 * @param array  $post      Post data when submit form.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	public function bb_prepare_result_data( $result, $item_ids, $item_type, $post ) {
		if ( $this->item_type === $item_type ) {
			$result['_list_entry_html']    = $this->get_category_rule_row( $item_ids[0], $item_type );
			$result['_list_tablenav_html'] = $this->prepare_list_tablenav(); // html entry for admin access rule category list.
		}

		return $result;
	}

	/**
	 * Function to add the category rule row.
	 *
	 * @param int|object $term_id   Item object.
	 * @param string     $item_type Item type.
	 *
	 * @since 1.5.2.1
	 * @return false|string
	 */
	public function get_category_rule_row( $term_id, $item_type ) {
		if ( ! $term_id || ! $item_type ) {
			return false;
		}

		$term_data = $this->get_category_data( $term_id );

		$category_list = new CategoryRuleList();
		$html          = '';
		ob_start();
		$category_list->single_row( $term_data );
		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Function to generate the Category nav.
	 *
	 * @since 1.5.5
	 *
	 * @return array
	 */
	public function prepare_list_tablenav() {
		$html          = array();
		$category_list = new CategoryRuleList();
		$category_list->prepare_items();

		ob_start();
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		echo wp_kses( $category_list->display_tablenav( 'top' ), $allowed_html, array( 'javascript' ) );
		$html['top'] = ob_get_clean();

		ob_start();
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		echo wp_kses( $category_list->display_tablenav( 'bottom' ), $allowed_html, array( 'javascript' ) );
		$html['bottom'] = ob_get_clean();

		return $html;
	}

	/**
	 * Get category data.
	 *
	 * @param int $term_id term id.
	 *
	 * @since 1.5.2.1
	 * @return null|array|false|\WP_Error|\WP_Term
	 */
	public function get_category_data( $term_id ) {
		if ( ! $term_id ) {
			return false;
		}

		$term_data = get_term( $term_id );

		return $term_data;
	}

	/**
	 * Function to show the category rule action messages.
	 *
	 * @since 1.5.2.1
	 */
	public function category_rule_action_notices() {
		$bulk_counts   = array(
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'deleted' => isset( $_REQUEST['deleted'] ) ? absint( $_REQUEST['deleted'] ) : 0,
		);
		$bulk_messages = array(
			/* translators: %s: Number of IAP Products. */
			'deleted' => _n( '%s rule permanently deleted.', '%s rules permanently deleted.', $bulk_counts['deleted'], 'buddyboss-app' ),
		);
		$bulk_counts   = array_filter( $bulk_counts );
		$messages      = array();

		foreach ( $bulk_counts as $message => $count ) {
			if ( isset( $bulk_messages[ $message ] ) ) {
				$messages[] = sprintf( $bulk_messages[ $message ], number_format_i18n( $count ) );
			}
		}

		if ( empty( $messages ) ) {
			$created_request = ( isset( $_REQUEST['created'] ) ) ? true : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

			if ( $created_request ) {
				$messages[] = esc_html__( 'The group has been created successfully.', 'buddyboss-app' );
			}

			$update_request = ( isset( $_REQUEST['updated'] ) ) ? true : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

			if ( $update_request ) {
				$messages[] = esc_html__( 'The group has been updated successfully.', 'buddyboss-app' );
			}
		}

		if ( $messages ) {
			echo '<div id="message" class="updated notice is-dismissible"><p>' . implode( ' ', array_map( 'esc_html', $messages ) ) . '</p></div>';
		}
	}

	/**
	 * Function to get the store category access rule.
	 *
	 * @param array $taxonomies Taxonomies array.
	 *
	 * @since 1.5.2.1
	 * @return int|int[]|string|string[]|\WP_Term[]
	 */
	public function get_stored_category_access_rules( $taxonomies = array() ) {
		if ( empty( $taxonomies ) ) {
			$taxonomies = $this->bb_get_taxonomy_list();
		}

		$access_rules = AccessRule::instance()->get_access_rules(
			array(
				'include_item_types' => $this->item_type,
			)
		);

		if ( ! empty( $access_rules['count'] ) ) {
			$item_ids = wp_list_pluck( $access_rules['results'], 'item_id' );
			$args     = array(
				'taxonomy'   => $taxonomies,
				'hide_empty' => false,
				'orderby'    => 'meta_value_num',
				'order'      => 'ASC',
				'include'    => $item_ids,
				'meta_query' => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'OR',
					array(
						'key'     => "bb_access_control_{$this->item_type}_menu_order",
						'value'   => '',
						'compare' => '!=',
					),
				),
			);
			$terms    = new \WP_Term_Query( $args );
			return $terms->get_terms();
		}

		return array();
	}

	/**
	 * Function to get taxonomy list.
	 *
	 * @since 1.6.3
	 * @return array
	 */
	public function bb_get_taxonomy_list() {
		$taxonomies     = array();
		$reg_taxonomies = get_object_taxonomies( Posts::instance()->post_type, 'objects' );
		if ( $reg_taxonomies ) {
			foreach ( $reg_taxonomies as $key => $taxonomy ) {
				// By pass post_format taxonomy.
				if ( 'post_format' === $key ) {
					continue;
				}
				$taxonomies[] = $key;
			}
		}

		return $taxonomies;
	}

	/**
	 * Function to get taxonomy dropdown.
	 *
	 * @since 1.5.2.1
	 */
	public function bb_get_taxonomies() {
		$result = array();
		$nonce  = ( ! empty( $_POST['nonce'] ) ) ? wp_unslash( $_POST['nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp_sort_nonce_' . get_current_user_id() ) ) {
			$result['message'] = __( 'Security check failed.', 'buddyboss-app' );

			wp_send_json_error( $result );
		}

		$item_type = ( ! empty( $_POST['item_type'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['item_type'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( Posts::instance()->post_type === $item_type ) {
			$taxonomy          = ! empty( $_POST['taxonomy'] ) ? bbapp_input_clean( wp_unslash( $_POST['taxonomy'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$name              = ! empty( $_POST['name'] ) ? bbapp_input_clean( wp_unslash( $_POST['name'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$result['message'] = '';
			$result['content'] = $this->bb_access_controls_get_taxonomies( $taxonomy, $name );

			wp_send_json_success( $result );
		}
	}

	/**
	 * Function to update category rule ajax.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function update_taxonomy_rules_order() {
		$screen_post = ( ! empty( $_POST['screen'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['screen'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $screen_post ) && Posts::instance()->screen_name . '_taxonomy_rules' === $screen_post ) {
			global $userdata;

			$result            = array();
			$result['message'] = __( 'Something went wrong please try again.', 'buddyboss-app' );
			$nonce             = ( ! empty( $_POST['nonce'] ) ) ? wp_unslash( $_POST['nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp_sort_nonce_' . $userdata->ID ) ) {
				$result['message'] = __( 'Security check failed.', 'buddyboss-app' );
				wp_send_json_error( $result );
			}

			$order_post = ( ! empty( $_POST['order'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['order'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$this->taxonomy_order_items( $order_post );
			$result['message']    = '';
			$result['is_job_set'] = ( Jobs::instance()->determine_process_running( 'bb_ac_term_calc' ) || Jobs::instance()->determine_process_running( 'bb_store_meta_by_ids' ) );
			wp_send_json_success( $result );
		}
	}

	/**
	 * Function to sort the category rules.
	 *
	 * @param string $taxonomy_order Rules order string.
	 *
	 * @since 1.5.2.1
	 *
	 * @return false|void
	 */
	public function taxonomy_order_items( $taxonomy_order = array() ) {
		parse_str( $taxonomy_order, $order_items );

		if ( ! empty( $order_items['cat-id'] ) ) {
			foreach ( $order_items['cat-id'] as $key => $order_item ) {
				update_term_meta( $order_item, "bb_access_control_{$this->item_type}_menu_order", absint( $key ) + ( 1 ) );
			}
		}

		/**
		 * Taxonomy item order update.
		 *
		 * Use the {@see 'bb_access_controls_taxonomy_items_order_update_after'} hook instead.
		 *
		 * @param string $user_login The user login name.
		 *
		 * @since      1.5.2.1
		 * @deprecated 1.6.3 New argument. Use {@see 'bb_access_controls_taxonomy_items_order_update_after'} hook instead.
		 */
		do_action_deprecated( 'bb_access_controls_taxonomy_order_items_after', array(), '1.6.3', 'bb_access_controls_taxonomy_items_order_update_after' );

		/**
		 * Taxonomy item order update.
		 *
		 * @param string $post_type Post type name.
		 * @param string $item_type Category item type.
		 *
		 * @since 1.6.3
		 */
		do_action( 'bb_access_controls_taxonomy_items_order_update_after', Posts::instance()->post_type, $this->item_type );
	}

	/**
	 * Function to add category rule order.
	 *
	 * @param int    $item_id   Taxonomy term.
	 * @param string $item_type item type name.
	 *
	 * @since 1.5.2.1
	 * @return bool
	 */
	public function add_category_rule_order( $item_id, $item_type ) {
		return bb_access_controls_add_category_rule_order( $item_id, $item_type );
	}
}
