<?php
/**
 * Handles table creation.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls\Core;

use BuddyBossApp\AccessControls\AccessRule;
use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Parent wp list table for custom ctp list.
 */
class List_Table extends \WP_List_Table {

	/**
	 * The type of view currently being displayed.
	 *
	 * E.g. "All", "Pending", "Approved", "Spam"...
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $view = 'all';

	/**
	 * Array of filters
	 *
	 * @since 1.5.2.1
	 * @var array $filters
	 */
	public $filters = array();

	/**
	 * Post type.
	 *
	 * @since 1.5.2.1
	 *
	 * @var string
	 */
	public $post_type = '';

	/**
	 * Current table
	 *
	 * @var mixed|string
	 */
	public $current_tab = '';

	/**
	 * Class constructor
	 *
	 * @param string $post_type Post type to create table.
	 * @param array  $args      Array of table arguments.
	 *
	 * @since 1.5.2.1
	 */
	public function __construct( $post_type, $args = array() ) {
		$this->post_type = $post_type;

		if ( $args['screen'] ) {
			$this->current_tab = $args['screen'];
			$args['screen']    = "bb_access_control_{$args['screen']}";
		}

		parent::__construct( $args );
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns['cb'] = '<input type="checkbox" />';

		if ( ! empty( $this->_args['column_slug'] ) && ! empty( $this->_args['column_title'] ) ) {
			$columns[ $this->_args['column_slug'] ] = sprintf( '<strong class="bb-bold-text">%1$s</strong>', $this->_args['column_title'] );
		}

		$columns['allow']          = sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Allow', 'access controls list table', 'buddyboss-app' ) );
		$columns['inherited_from'] = sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Inherited From', 'access controls list table', 'buddyboss-app' ) );
		$columns['manage']         = sprintf( '<strong class="bb-bold-text">%1$s</strong>', _x( 'Manage', 'access controls list table', 'buddyboss-app' ) );

		if ( isset( $this->_args['column_inherited_hide'] ) && true === $this->_args['column_inherited_hide'] ) {
			unset( $columns['inherited_from'] );
		}

		return $columns;
	}

	/**
	 * No rules found
	 *
	 * @since 1.5.2.1
	 */
	public function no_items() {
		esc_html_e( 'No access found.', 'buddyboss-app' );
	}

	/**
	 * App Pages column default values.
	 *
	 * @param object $item        Rule item.
	 * @param string $column_name Column name.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'inherited_from':
				echo 'inherited_from';
				break;
			case 'manage':
				$rule_data   = AccessRule::instance()->get_access_rule( $item->ID, $item->post_type );
				$row_actions = array(
					'add'   => array(
						'link'    => sprintf(
							'<a href="javascript:void(0);" class="bb-access-control-edit-rule" data-item-type="%1$s" data-item-id="%2$s">%3$s</a>',
							esc_attr( $item->post_type ),
							esc_attr( $item->ID ),
							esc_html_x( 'Add Rule', 'access controls list table', 'buddyboss-app' )
						),
						'is_hide' => ! empty( $rule_data ),
					),
					'edit'  => array(
						'link'    => sprintf(
							'<a href="javascript:void(0);" class="bb-access-control-edit-rule" data-item-type="%1$s" data-item-id="%2$s">%3$s</a>',
							esc_attr( $item->post_type ),
							esc_attr( $item->ID ),
							esc_html_x( 'Edit Rule', 'access controls list table', 'buddyboss-app' )
						),
						'is_hide' => empty( $rule_data ),
					),
					'trash' => array(
						'link'    => sprintf(
							'<a href="%1$s" class="submitdelete" onclick="return showNotice.warn();">%2$s</a>',
							add_query_arg(
								array(
									'page'      => 'bbapp-access-controls',
									'tab'       => $this->current_tab,
									'action'    => 'delete',
									"{$this->post_type}_ids" => $item->ID,
									'item_type' => $item->post_type,
									'_wpnonce'  => wp_create_nonce( 'bb-access-controls-delete-rule' ),
								),
								bbapp_get_admin_url( 'admin.php' )
							),
							esc_html_x( 'Remove Rule', 'access controls list table', 'buddyboss-app' )
						),
						'is_hide' => empty( $rule_data ),
					),
				);

				$action_count = count( $row_actions );
				$i            = 0;
				$output       = '<div class="row-actions visible">';

				foreach ( $row_actions as $action => $action_item ) {
					++ $i;
					( $i === $action_count ) ? $sep = '' : $sep = ' | ';
					$class                          = $action_item['is_hide'] ? 'hidden' : '';
					$sep                            = ( 'add' === $action ) ? '' : $sep;
					$output                        .= "<span class='{$action} {$class}'>{$action_item['link']}$sep</span>";
				}

				$output                      .= '</div>';
				$allowed_html                 = wp_kses_allowed_html( 'post' );
				$allowed_html['a']['onclick'] = true;
				echo wp_kses( $output, $allowed_html, array( 'javascript' ) );
				break;
			default:
				echo 'N/A';
				break;
		}
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Page item.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_allow( $item ) {
		$rule_data = AccessRule::instance()->get_access_rule( $item->ID, $item->post_type );
		$rule_data = ! empty( $rule_data['rule_data'] ) ? $rule_data['rule_data'] : array();

		echo wp_kses_post( bb_access_controls_rule_preview( $rule_data ) );
	}

	/**
	 * Display the table.
	 *
	 * @since 1.5.2.1
	 */
	public function display() {
		$singular = $this->_args['singular'];

		$this->display_tablenav( 'top' );

		$this->screen->render_screen_reader_content( 'heading_list' );
		?>
		<table class="wp-list-table <?php echo implode( ' ', array_map( 'esc_attr', $this->get_table_classes() ) ); ?>">
			<thead>
			<tr>
				<?php $this->print_column_headers(); ?>
			</tr>
			</thead>

			<tbody id="the-list"
				<?php
				if ( $singular ) {
					echo " data-wp-lists='list:" . esc_attr( $singular ) . "'";
				}
				?>
			>
			<?php $this->display_rows_or_placeholder(); ?>
			</tbody>

			<tfoot>
			<tr>
				<?php $this->print_column_headers( false ); ?>
			</tr>
			</tfoot>
		</table>
		<?php
		$this->display_tablenav( 'bottom' );
	}

	/**
	 * Function to change layout for the WP List table.
	 *
	 * @param object $item List table item.
	 *
	 * @since 1.5.2.1
	 */
	public function single_row( $item ) {
		$allowed_html                 = wp_kses_allowed_html( 'post' );
		$allowed_html['a']['onclick'] = true;
		$allowed_html['input']        = array(
			'type'    => array(),
			'name'    => array(),
			'value'   => array(),
			'checked' => array(),
		);
		echo "<tr data-rule-id='" . esc_attr( $item->ID ) . "'>";
		ob_start();
		$this->single_row_columns( $item );
		$row = ob_get_clean();
		echo wp_kses( $row, $allowed_html, array( 'javascript' ) );
		echo "</tr>\n";
	}

	/**
	 * Generate the table navigation above or below the table
	 *
	 * @param string $position Position to show the table navigation.
	 *
	 * @since 1.5.2.1
	 */
	protected function display_tablenav( $position ) {
		if ( 'top' === $position ) {
			wp_nonce_field( 'bulk-' . $this->_args['plural'] );
		}
		?>
		<div class="tablenav <?php echo esc_attr( $position ); ?>">
			<?php
			if ( 'top' === $position ) {
				if ( $this->has_items() ) {
					?>
					<div class="alignleft actions bulkactions">
						<?php $this->bulk_actions( $position ); ?>
					</div>
					<?php
				}
			}

			$this->extra_tablenav( $position );

			if ( 'top' !== $position ) {
				$this->pagination( $position );
			}
			?>
			<br class="clear"/>
		</div>
		<?php
	}

	/**
	 * Check table filters
	 *
	 * @since 1.5.2.1
	 */
	public function check_table_filters() {
		$this->filters = array();
		$search        = ( ! empty( $_GET['s'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['s'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ( ! empty( $search ) ) ) {
			$this->filters['search'] = esc_attr( $search );
		}
	}

	/**
	 * Whether the table has items to display or not
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function has_items() {
		return ! empty( $this->items );
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 *
	 * @since 1.5.2.1
	 */
	public function prepare_items() {
		$total_items  = 0;
		$post_type    = str_replace( '-', '_', $this->post_type );
		$per_page     = $this->get_items_per_page( "bb_access_controls_{$post_type}_per_page", 10 );
		$current_page = $this->get_pagenum();

		// Init column headers.
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		/** Process bulk action */
		$this->process_bulk_action();

		$post_query_args = array(
			'post_type'        => $this->post_type,
			'posts_per_page'   => $per_page,
			'paged'            => $current_page,
			'post_status'      => 'publish',
			'suppress_filters' => true,
		);

		if ( ! empty( $this->filters['search'] ) ) {
			$post_query_args['s'] = '"' . $this->filters['search'] . '"';
			add_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10, 2 );
		}

		$bbapp_post_query = new WP_Query( $post_query_args );

		if ( ! empty( $this->filters['search'] ) ) {
			remove_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10, 2 );
		}

		if ( ! empty( $bbapp_post_query->posts ) ) {
			$this->items = $bbapp_post_query->posts;
			$total_items = $bbapp_post_query->found_posts;
		}

		$this->set_pagination_args(
			array(
				'total_items' => intval( $total_items ),
				// We have to calculate the total number of items.
				'per_page'    => intval( $per_page ),
				// We have to determine how many items to show on a page.
				'total_pages' => ceil( intval( $total_items ) / intval( $per_page ) ),
				// We have to calculate the total number of pages.
			)
		);

		/** Process bulk action */
		$this->process_bulk_action();
	}

	/**
	 * Search filter by title
	 *
	 * @param string $search   Search SQL for WHERE clause.
	 * @param object $wp_query The current WP_Query object.
	 *
	 * @since 1.5.2.1
	 * @return array|string
	 */
	public function search_filter_by_title( $search, $wp_query ) {
		if ( ! empty( $search ) && ! empty( $wp_query->query_vars['search_terms'] ) ) {
			global $wpdb;

			$q      = $wp_query->query_vars;
			$n      = ! empty( $q['exact'] ) ? '' : '%';
			$search = array();

			foreach ( (array) $q['search_terms'] as $term ) {
				$search[] = $wpdb->prepare( "$wpdb->posts.post_title LIKE %s", $n . $term . $n );
			}

			if ( ! is_user_logged_in() ) {
				$search[] = "$wpdb->posts.post_password = ''";
			}

			$search = ' AND ' . implode( ' AND ', $search );
		}

		return $search;
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item List table item.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		if ( current_user_can( 'manage_options' ) ) {
			return sprintf( '<input type="checkbox" name="%1$s_ids[]" value="%2$s" />', $this->post_type, $item->ID );
		}

		return '';
	}

	/**
	 * Bulk action.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		return array(
			'bulk-edit'   => _x( 'Bulk edit', 'access controls list table', 'buddyboss-app' ),
			'bulk-delete' => _x( 'Remove all rules', 'access controls list table', 'buddyboss-app' ),
		);
	}

	/**
	 * Bulk action process.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function process_bulk_action() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'You don\'t have permission to access this page.', 'buddyboss-app' );
		}

		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$redirect_to = remove_query_arg(
			array(
				'action',
				'action2',
				's',
				'tab',
				'post_type',
				'item_type',
				"{$this->post_type}_ids",
				'deleted',
				'error',
				'updated',
				'success_new',
				'error_new',
				'success_modified',
				'error_modified',
				'_wpnonce',
				'_wp_http_referer',
				'paged',
			),
			$request_uri
		);
		$redirect_to = add_query_arg( array( 'tab' => $this->current_tab ), $redirect_to );
		$action      = $this->current_action();

		switch ( $action ) {
			case 'delete':
			case 'bulk-delete':
				if ( ! empty( $_REQUEST[ "{$this->post_type}_ids" ] ) ) {
					$nonce_key = ( 'bulk-delete' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'bb-access-controls-delete-rule';

					check_admin_referer( $nonce_key );

					$post_type_ids_key = "{$this->post_type}_ids";
					$item_ids_arr      = ( ! empty( $_REQUEST[ $post_type_ids_key ] ) ) ? ( is_array( $_REQUEST[ $post_type_ids_key ] ) ) ? array_map( 'absint', $_REQUEST[ $post_type_ids_key ] ) : absint( $_REQUEST[ $post_type_ids_key ] ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
					$item_ids          = wp_parse_id_list( $item_ids_arr );
					$item_type         = ( ! empty( $_REQUEST['item_type'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['item_type'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$access_rules      = AccessRule::instance()->get_access_rules(
						array(
							'include_item_types' => $item_type,
							'include_item_id'    => $item_ids,
						)
					);
					$count             = 0;

					if ( ! empty( $access_rules['count'] ) ) {
						foreach ( $access_rules['results'] as $access_rule ) {
							/**
							 * Before delete list item rule.
							 *
							 * @param string|int $item_id   Item id.
							 * @param string     $item_type Item type.
							 *
							 * @since 1.6.3
							 */
							do_action( 'bb_access_controls_delete_item_rule', $access_rule['item_id'], $access_rule['item_type'] );
							$deleted = AccessRule::instance()->delete_access_rule( $access_rule['id'] );
							if ( ! is_wp_error( $deleted ) ) {
								/**
								 * Before delete list item rule.
								 *
								 * @param string|int $item_id   Item id.
								 * @param string     $item_type Item type.
								 *
								 * @since 1.6.3
								 */
								do_action( 'bb_access_controls_deleted_item_rule', $access_rule['item_id'], $access_rule['item_type'] );
								$count ++;
							}
						}
					}

					$redirect_to = add_query_arg(
						array(
							'deleted' => $count,
							'ids'     => $item_ids,
						),
						$redirect_to
					);

					wp_safe_redirect( $redirect_to );
					exit();
				}
				break;
		}
	}
}
