<?php
/**
 * Handle core condtions
 *
 * @package BuddyBossApp\AccessControls\Core
 */

namespace BuddyBossApp\AccessControls\Core;

use BuddyBossApp\AccessControls\Integration_Abstract;
use BuddyBossApp\AccessControls\Core\Settings\AppPages;
use BuddyBossApp\AccessControls\Core\Settings\General;
use BuddyBossApp\AccessControls\Core\Settings\Pages;
use BuddyBossApp\AccessControls\Core\Settings\Posts;
use BuddyBossApp\Admin\SetupAdmin;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Contains the core Access Control Implementations.
 *
 * Class Core
 *
 * @package BuddyBossApp\AccessControls\Core
 */
class Core extends Integration_Abstract {

	/**
	 * Conditiona name to register.
	 *
	 * @var string $condition_name condition name.
	 */
	private $condition_name = 'wordpress-roles';

	/**
	 * Function to set up the condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function setup() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );

		// Register Condition.
		$this->register_condition(
			array(
				'condition'         => $this->condition_name,
				'items_callback'    => array( $this, 'wordpress_role_items_callback' ),
				'item_callback'     => array( $this, 'wordpress_role_item_callback' ),
				'users_callback'    => array( $this, 'wordpress_role_users_callback' ),
				'labels'            => array(
					'condition_name' => _x( 'WordPress Role', 'access control settings', 'buddyboss-app' ),
					'item_singular'  => _x( 'Role', 'access control settings', 'buddyboss-app' ),
				),
				'support_any_items' => false,
			)
		);

		// Register Rules Item Types.
		$this->register_rules_item_type( 'general' );
		$this->register_rules_item_type( 'default_ld_course' );
		$this->register_rules_item_type( 'default_post' );
		$this->register_rules_item_type( 'app_page' );
		$this->register_rules_item_type( 'page' );
		$this->register_rules_item_type( 'post' );
		$this->register_rules_item_type( 'post_term' );

		// Register Settings Screens.
		$this->register_screens();

		$this->load_hooks();
	}

	/**
	 * Load Access Controls.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function admin_init() {
		$page = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && ! empty( $page ) && 'bbapp-access-controls' === $page ) {
			$current_tab = $this->get_current_tab();

			if ( 'app-pages' === $current_tab && bbapp_is_admin_page() ) {
				AppPages::instance()->will_render( true );
			}

			if ( 'pages' === $current_tab && bbapp_is_admin_page() ) {
				Pages::instance()->will_render( true );
			}

			if ( 'posts' === $current_tab && bbapp_is_admin_page() ) {
				Posts::instance()->will_render( true );
			}
		}
	}

	/**
	 * Load screens settings classes.
	 *
	 * @since 1.5.2.1
	 */
	public function register_screens() {
		$this->register_settings_screen(
			array(
				'id'              => General::instance()->screen_name,
				'menu_title'      => _x( 'General', 'access control settings', 'buddyboss-app' ),
				'page_title'      => _x( 'General', 'access control settings', 'buddyboss-app' ),
				'render_callback' => array( General::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( 'admin.php?page=bbapp-help&article=125675' ),
				),
			)
		);
		$this->register_settings_screen(
			array(
				'id'              => AppPages::instance()->screen_name,
				'menu_title'      => _x( 'App Pages', 'access controls settings', 'buddyboss-app' ),
				'page_title'      => _x( 'App Pages', 'access controls settings', 'buddyboss-app' ),
				'render_callback' => array( AppPages::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( 'admin.php?page=bbapp-help&article=125675' ),
				),
			)
		);
		$this->register_settings_screen(
			array(
				'id'              => Pages::instance()->screen_name,
				'menu_title'      => _x( 'Pages', 'access controls settings', 'buddyboss-app' ),
				'page_title'      => _x( 'Pages', 'access controls settings', 'buddyboss-app' ),
				'render_callback' => array( Pages::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( 'admin.php?page=bbapp-help&article=125675' ),
				),
			)
		);
		$this->register_settings_screen(
			array(
				'id'              => Posts::instance()->screen_name,
				'menu_title'      => _x( 'Posts', 'access controls settings', 'buddyboss-app' ),
				'page_title'      => _x( 'Posts', 'access controls settings', 'buddyboss-app' ),
				'render_callback' => array( Posts::instance(), 'render_callback' ),
				'options'         => array(
					'tutorial_link' => admin_url( 'admin.php?page=bbapp-help&article=125675' ),
					'sub_tab'       => array(
						'posts'            => _x( 'Posts', 'access controls settings', 'buddyboss-app' ),
						'cat_tag'          => _x( 'Categories & Tags', 'access controls settings', 'buddyboss-app' ),
						'default_settings' => _x( 'Default Settings', 'access controls settings', 'buddyboss-app' ),
					),
				),
			)
		);
	}

	/**
	 * Function to load all hooks of this condition.
	 *
	 * @since 1.5.2.1
	 */
	public function load_hooks() {
		add_action( 'set_user_role', array( $this, 'set_user_role' ), 10, 3 );
		add_action( 'add_user_role', array( $this, 'add_user_role' ), 10, 2 );
		add_action( 'remove_user_role', array( $this, 'remove_user_role' ), 10, 2 );
		add_action( 'delete_user', array( $this, 'delete_user' ), 10 );
		add_action( 'wpmu_delete_user', array( $this, 'delete_user' ), 10 );
		add_action( 'bp_make_spam_user', array( $this, 'delete_user' ), 10 );
		add_action( 'bp_suspend_hide_user', array( $this, 'suspend_hide_user' ), 11, 1 );
	}

	/**
	 * Items callback method.
	 *
	 * @param string $search Search the condition.
	 * @param int    $page   Page number.
	 * @param int    $limit  Limit the items to be fetched.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string[]
	 */
	public function wordpress_role_items_callback( $search = '', $page = 1, $limit = 20 ) {
		$roles = $this->bb_get_editable_roles();
		$list  = array();

		if ( ! empty( $roles ) ) {
			foreach ( $roles as $key => $role ) {
				$list[ $key ] = array(
					'id'   => $key,
					'name' => translate_user_role( $role['name'] ),
					'link' => bbapp_get_admin_url( "users.php?role=$key" ),
				);
			}
		}

		return $this->paginate_items_list( $list, $page, $limit, $search );
	}

	/**
	 * Fetch a filtered list of user roles that the current user is
	 * allowed to edit.
	 *
	 * Simple function whose main purpose is to allow filtering of the
	 * list of roles in the $wp_roles object so that plugins can remove
	 * inappropriate ones depending on the situation or user making edits.
	 * Specifically because without filtering anyone with the edit_users
	 * capability can edit others to be administrators, even if they are
	 * only editors or authors. This filter allows admins to delegate
	 * user management.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array[] Array of arrays containing role information.
	 */
	public function bb_get_editable_roles() {
		$all_roles = wp_roles()->roles;

		/**
		 * Filters the list of editable roles.
		 *
		 * @since 2.8.0
		 *
		 * @param array[] $all_roles Array of arrays containing role information.
		 */
		$editable_roles = apply_filters( 'editable_roles', $all_roles );

		return $editable_roles;
	}

	/**
	 * Item callback method.
	 *
	 * @param string $item_value Item value of condition.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|false
	 */
	public function wordpress_role_item_callback( $item_value ) {
		$roles = $this->wordpress_role_items_callback( '', 1, - 1 );

		if ( empty( $roles ) || ! array_key_exists( $item_value, $roles ) ) {
			return false;
		}

		return $roles[ $item_value ];
	}

	/**
	 * Users callback method.
	 *
	 * @param array $data     condition data.
	 * @param int   $page     current page number.
	 * @param int   $per_page limit.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function wordpress_role_users_callback( $data, $page = 1, $per_page = 10 ) {
		$item_value = ( ! empty( $data['item_value'] ) ) ? $data['item_value'] : '';

		if ( empty( $item_value ) ) {
			return array();
		}

		$args = array(
			'role'   => $item_value,
			'number' => ( ! empty( $per_page ) ) ? $per_page : 10,
			'paged'  => ( ! empty( $page ) ) ? $page : 1,
			'fields' => 'ID',
		);

		$users = new \WP_User_Query( $args );

		return $this->return_users( $users->get_results() );
	}

	/**
	 * Function to update member role and update the member access.
	 *
	 * @param int      $user_id   The user ID.
	 * @param string   $role      The new role.
	 * @param string[] $old_roles An array of the user's previous roles.
	 *
	 * @since 1.5.2.1
	 */
	public function set_user_role( $user_id, $role, $old_roles ) {
		if ( empty( $user_id ) || empty( $role ) ) {
			return;
		}

		if ( ! empty( $old_roles ) ) {
			foreach ( $old_roles as $old_role ) {
				$this->condition_remove_user( $user_id, $this->condition_name, $old_role );
			}
		}

		$this->condition_add_user( $user_id, $this->condition_name, $role );
	}

	/**
	 * Function to update access when a role is added via add_role.
	 *
	 * @param int    $user_id The user ID.
	 * @param string $role    The role added.
	 *
	 * @since 2.6.0
	 */
	public function add_user_role( $user_id, $role ) {
		if ( empty( $user_id ) || empty( $role ) ) {
			return;
		}

		$this->condition_add_user( $user_id, $this->condition_name, $role );
	}

	/**
	 * Function to update access when a role is removed via remove_role.
	 *
	 * @param int    $user_id The user ID.
	 * @param string $role    The role removed.
	 *
	 * @since 2.6.0
	 */
	public function remove_user_role( $user_id, $role ) {
		if ( empty( $user_id ) || empty( $role ) ) {
			return;
		}

		$this->condition_remove_user( $user_id, $this->condition_name, $role );
	}

	/**
	 * Function to delete access of provided member.
	 *
	 * @param int $id ID of the user to delete.
	 *
	 * @since 1.5.2.1
	 */
	public function delete_user( $id ) {
		if ( empty( $id ) ) {
			return;
		}

		$user_roles = $this->bbapp_get_user_role( $id );

		foreach ( $user_roles as $user_role ) {
			$this->condition_remove_user( $id, $this->condition_name, $user_role );
		}
	}

	/**
	 * Function to delete access of provided member.
	 *
	 * @param int $item_id item id| user id.
	 *
	 * @since 1.5.2.1
	 */
	public function suspend_hide_user( $item_id ) {
		if ( empty( $item_id ) ) {
			return;
		}
		// Remove user when suspended.
		if ( function_exists( 'bp_moderation_is_user_suspended' ) && bp_moderation_is_user_suspended( $item_id ) ) {
			$user_roles = $this->bbapp_get_user_role( $item_id );

			foreach ( $user_roles as $user_role ) {
				$this->condition_remove_user( $item_id, $this->condition_name, $user_role );
			}
		}
	}

	/**
	 * This function to get user role by user id.
	 *
	 * @param int $user_id User id.
	 *
	 * @since 1.5.2.1
	 * @return string[]
	 */
	public function bbapp_get_user_role( $user_id ) {
		$user_meta = get_userdata( $user_id );

		return $user_meta->roles;
	}

	/**
	 * Get current tab.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|string
	 */
	public function get_current_tab() {
		$current_tab = ( ! empty( $_GET['tab'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['tab'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		return ! empty( $current_tab ) ? $current_tab : 'general';
	}
}
