<?php
/**
 * Used by Actions functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

use BuddyBossApp\Jobs;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Actions class.
 */
class Actions {

	/**
	 * Class instance.
	 *
	 * @since 1.5.2.1
	 * @var object
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Actions
	 * @since 1.5.2.1
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Add related class and hooks
	 *
	 * @since 1.5.2.1
	 */
	public function init() {
		add_filter( 'bb_access_controls_prepare_group_item', array( $this, 'item_value_status' ), 1, 2 );
		add_filter( 'bb_access_controls_prepare_group_item', array( $this, 'group_name' ), 10, 2 );
		add_filter( 'bb_access_controls_prepare_group_item', array( $this, 'group_member_count' ), 10, 2 );
		add_filter( 'posts_where', array( $this, 'posts_where' ), 999, 2 );
		// Access Group screen options.
		add_action( 'load-buddyboss-app_page_bbapp-access-controls', array( $this, 'get_screen_options' ) );
		add_filter( 'set-screen-option', array( $this, 'bbapp_admin_screen_options' ), 10, 3 );

		// Background job task.
		add_action( 'bbapp_queue_task_bb_ac_term_calc', array( $this, 'store_meta_ids' ), 10, 1 );
		add_action( 'bbapp_queue_task_bb_store_meta_by_ids', array( $this, 'store_meta_by_ids' ), 10, 1 );
		// Return the taoxonmy status.
		add_action( 'wp_ajax_bb_access_controls_get_taxonomy_job_status', array( $this, 'ajax_get_taxonomy_job_status' ) );
		add_action( 'deleted_user', array( $this, 'remove_user_from_condition' ), 10 );
	}

	/**
	 * Altering the post join logic to hide restricted posts.
	 *
	 * @param string $where    where condition string.
	 * @param object $wp_query WP_Query.
	 *
	 * @since 1.6.3
	 * @return string
	 */
	public function posts_where( $where, $wp_query ) {
		global $wpdb;

		if ( is_admin() ) {
			return $where;
		}

		$post_types = array();
		/**
		 * Filter to use for allowing post type.
		 *
		 * @param array $post_types Post type name in array.
		 *
		 * @since 1.6.3
		 */
		$allow_post_types = apply_filters( 'bb_access_controls_allow_post_types_filter', $post_types );

		$_post_type = $wp_query->query_vars['post_type'];
		if ( ! in_array( $_post_type, $allow_post_types, true ) ) {
			return $where;
		}

		if ( false === bbapp_is_app_request() ) {
			return $where;
		}

		if ( ! bbapp_is_rest() ) {
			return $where;
		}

		if ( empty( $wp_query->query_vars['access_control_filter'] ) || false === $wp_query->query_vars['access_control_filter'] ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return $where;
		}

		$default_rule = array();
		/**
		 * Access controls default rule.
		 *
		 * @param array $default_rule Default rule in array.
		 * @param string $_post_type   Post type.
		 *
		 * @since 1.6.3
		 */
		$default_rule_data    = apply_filters( 'bb_access_controls_default_rule', $default_rule, $_post_type );
		$can_access_all_posts = true;

		// Get the default posts rule.
		if ( isset( $default_rule_data['default_setting'] ) && isset( $default_rule_data['default_setting']['restrict_all'] ) && '1' === (string) $default_rule_data['default_setting']['restrict_all']['enabled'] ) {
			$can_access_all_posts = bb_access_controls_user_can_access_rule( $default_rule_data, get_current_user_id() );
		}

		$item_type = '';
		/**
		 * Access controls term item type.
		 *
		 * @param string $item_type  Default rule in array.
		 * @param string $_post_type Post type.
		 *
		 * @since 1.6.3
		 */
		$item_types       = apply_filters( 'bb_access_controls_get_term_item_type', $item_type, $_post_type );
		$term_rules       = AccessRule::instance()->get_access_rules( array( 'include_item_types' => $item_types ) );
		$restricted_terms = array();
		$allowed_terms    = array();
		foreach ( $term_rules['results'] as $term_rule ) {
			$can_access = bb_access_controls_user_can_access_rule( $term_rule['rule_data'], get_current_user_id() );
			if ( ! $can_access ) {
				$restricted_terms[] = $term_rule['item_id'];
			} else {
				$allowed_terms[] = $term_rule['item_id'];
			}
		}

		$in_statements     = array();
		$not_in_statements = array();

		if ( $can_access_all_posts ) {  // If user has access to all posts by default.

			// Exclude posts of restricted terms.
			if ( ! empty( $restricted_terms ) ) {
				$allowed_posts       = bb_access_controls_has_access_items_sql_query( array( 'item_type' => $_post_type ) );
				$in_placeholders     = implode( ', ', array_fill( 0, count( $restricted_terms ), '%d' ) );
				$not_in_statements[] = $wpdb->prepare( " {$wpdb->prefix}posts.ID NOT IN (SELECT object_id FROM `{$wpdb->prefix}term_relationships` WHERE `term_taxonomy_id` IN ({$in_placeholders}) AND object_id NOT IN ({$allowed_posts}) ) ", $restricted_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		} else { // when by default user has access to no posts.

			// We will allow the posts user has specially access to as priority.
			$allowed_posts   = bb_access_controls_has_access_items_sql_query( array( 'item_type' => $_post_type ) );
			$in_statements[] = " {$wpdb->prefix}posts.ID IN ({$allowed_posts}) ";

			// Allowed the posts from user allowed terms.
			if ( ! empty( $allowed_terms ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $allowed_terms ), '%d' ) );
				$in_statements[] = $wpdb->prepare( " {$wpdb->prefix}posts.ID IN (SELECT object_id FROM `{$wpdb->prefix}term_relationships` WHERE `term_taxonomy_id` IN ({$in_placeholders}))", $allowed_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Remove all items which user has no access to.
		$not_allowed_posts   = bb_access_controls_has_no_access_items_sql_query( array( 'item_type' => $_post_type ) );
		$not_in_statements[] = " {$wpdb->prefix}posts.ID NOT IN ({$not_allowed_posts}) ";

		$and_post_type = $wpdb->prepare( " AND {$wpdb->prefix}posts.post_type = %s", $_post_type ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$in_statements = implode( ' OR ', $in_statements );

		$post__not_in_query = '';
		if ( ! empty( $wp_query->query_vars['post__not_in'] ) ) {
			$post__not_in_placeholders = implode( ', ', array_fill( 0, count( $wp_query->query_vars['post__not_in'] ), '%d' ) );
			$post__not_in_query        = $wpdb->prepare( "AND acpm.post_id NOT IN ( {$post__not_in_placeholders} )", $wp_query->query_vars['post__not_in'] ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		$post__in_query = '';
		if ( ! empty( $wp_query->query_vars['post__in'] ) ) {
			$post__in_placeholders = implode( ', ', array_fill( 0, count( $wp_query->query_vars['post__in'] ), '%d' ) );
			$post__in_query        = $wpdb->prepare( "AND acpm.post_id IN ( {$post__in_placeholders} )", $wp_query->query_vars['post__in'] ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		if ( ! empty( $in_statements ) ) {
			$where .= " AND  ({$in_statements}";
			if ( ! empty( $restricted_terms ) ) {
				$or_in_placeholders = implode( ', ', array_fill( 0, count( $restricted_terms ), '%d' ) );
				$where             .= $wpdb->prepare( " AND {$wpdb->prefix}posts.ID NOT IN ( SELECT post_id FROM {$wpdb->prefix}postmeta AS acpm WHERE acpm.meta_key = 'bb_access_term_id' AND acpm.meta_value IN ({$or_in_placeholders}) {$post__in_query} {$post__not_in_query}) {$and_post_type}", $restricted_terms ); // //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
			$where .= ' )';
		}

		$not_in_statements = implode( ' AND ', $not_in_statements );
		if ( ! empty( $not_in_statements ) ) {
			$where .= " AND  ({$not_in_statements}";
			if ( ! empty( $allowed_terms ) ) {
				$or_in_placeholders = implode( ', ', array_fill( 0, count( $allowed_terms ), '%d' ) );
				$where             .= $wpdb->prepare( " OR {$wpdb->prefix}posts.ID  IN ( SELECT post_id FROM {$wpdb->prefix}postmeta AS acpm WHERE acpm.meta_key = 'bb_access_term_id' AND acpm.meta_value IN ({$or_in_placeholders}) {$post__in_query} {$post__not_in_query}) {$and_post_type}", $allowed_terms ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
			$where .= ' )';
		}

		return $where;
	}

	/**
	 * Register the screen options.
	 *
	 * @since 1.5.2.1
	 */
	public function get_screen_options() {
		$screen   = get_current_screen();
		$settings = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-access-controls' !== $screen->id || 'access-groups' !== $settings ) {
			return;
		}

		// Loop through all the options and add a screen option for each.
		add_screen_option(
			'per_page',
			array(
				'label'   => __( 'Number of access groups / group members per page:', 'buddyboss-app' ),
				'default' => 10,
				'option'  => 'bb_access_controls_groups_per_page',
			)
		);
	}

	/**
	 * Handle save/update of screen options for Access Groups.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @since 1.5.2.1
	 *
	 * @return string|int Option value. False to abandon update.
	 */
	public function bbapp_admin_screen_options( $value, $option, $new_value ) {
		if ( 'bb_access_controls_groups_per_page' !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;

		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}

	/**
	 * Add Item Value Status.
	 *
	 * @param array  $data  Group member data.
	 * @param object $group Group data object.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function item_value_status( $data, $group ) {
		$data['item_value_status'] = 'active';
		$data['item_value_data']   = false;
		$registered_conditions     = bb_access_controls_get_conditions();
		$condition_name            = $group->condition_name;

		if ( isset( $registered_conditions[ $condition_name ] ) ) {
			$condition = $registered_conditions[ $condition_name ];
			$item      = bb_access_controls_condition_item_callback_handler( $condition, $group->item_value );

			if ( false === $item ) {
				$data['item_value_status'] = 'deleted';
			} else {
				$data['item_value_data'] = $item;
			}
		} else {
			$data['item_value_status'] = 'disabled';
		}

		return $data;
	}

	/**
	 * Function to prepare group name.
	 *
	 * @param array  $data  Group data.
	 * @param object $group Group object.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function group_name( $data, $group ) {
		$group_name             = bb_access_controls_prepare_group_name( $data );
		$group_name_sign        = bb_access_controls_prepare_group_name( $data, ': ' );
		$group_name_by_status   = bb_access_controls_prepare_group_name( $data, ': ', true );
		$data['name']           = wp_strip_all_tags( $group_name );
		$data['name_rich']      = $group_name;
		$data['name_sign']      = $group_name_sign;
		$data['name_by_status'] = $group_name_by_status;

		return $data;
	}

	/**
	 * Get member count in group.
	 *
	 * @param array  $data  Group member data.
	 * @param object $group Group data object.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function group_member_count( $data, $group ) {
		$group_members         = bb_access_controls_get_group_members(
			array(
				'group_include' => $group->id,
				'only_count'    => true,
			)
		);
		$data['members_count'] = $group_members['count'];

		return $data;
	}


	/**
	 * Prepare function to store category/tag id to post meta.
	 *
	 * @param object $task Job task data.
	 *
	 * @since 1.6.3
	 */
	public function store_meta_ids( $task ) {
		$task_data            = maybe_unserialize( $task->data );
		$page                 = isset( $task_data['page'] ) ? $task_data['page'] : 1; // Initiate the recalculations.
		$post_type            = isset( $task_data['post_type'] ) ? $task_data['post_type'] : ''; // Initiate the recalculations.
		$bg_data['post_type'] = $post_type;
		/**
		 * If recalculation is requested reset the params.
		 */
		if ( ! empty( $task_data['status'] ) && 'start' === $task_data['status'] ) {
			$bg_data['status'] = 'running';
		}

		$args        = array(
			'post_type' => $post_type,
			'fields'    => 'ids',
			'paged'     => $page,
		);
		$found_posts = new \WP_Query( $args );
		if ( ! empty( $found_posts->have_posts() ) ) {
			$this->update_store_meta_ids( $found_posts->posts, $post_type );
			$bg_data['page'] = $page + 1;
			bb_access_controls_run_terms_calculation_job( $post_type, $bg_data );
		}
	}

	/**
	 * Prepare function to store category/tag id to post meta by post ids.
	 *
	 * @param object $task Job task data.
	 *
	 * @since 1.6.3
	 */
	public function store_meta_by_ids( $task ) {
		$task_data = maybe_unserialize( $task->data );
		$ids       = isset( $task_data['ids'] ) ? $task_data['ids'] : array();
		$extra     = isset( $task_data['extra'] ) ? $task_data['extra'] : array();
		$post_type = isset( $extra['post_type'] ) ? $extra['post_type'] : array();
		if ( ! empty( $ids ) ) {
			$this->update_store_meta_ids( $ids, $post_type, $extra );
		}
	}

	/**
	 * Update store meta ids.
	 *
	 * @param array  $post_ids     Post ids.
	 * @param string $post_type Post type.
	 * @param array  $extra     Extra array for adding extra process.
	 *
	 * @since 1.6.3
	 */
	public function update_store_meta_ids( $post_ids, $post_type, $extra = array() ) {
		foreach ( $post_ids as $post_id ) {

			/**
			 * Get access controls inherited from by filter.
			 *
			 * @param array  $post_ids  Post ids.
			 * @param string $post_type Post type.
			 * @param array  $extra     Extra array for adding extra process.
			 *
			 * @since 1.6.3
			 */
			$inherited_from = apply_filters( 'bb_access_controls_get_access_inherited_from', array(), $post_id, $post_type );

			if ( ! empty( $inherited_from ) ) {
				if ( 'taxonomy' === $inherited_from['type'] ) {
					$stored_term = $inherited_from['term'];
					if ( ! empty( $extra['item_id'] ) && $stored_term->term_id === (int) $extra['item_id'] ) {
						delete_post_meta( $post_id, 'bb_access_term_id' );
					} else {
						update_post_meta( $post_id, 'bb_access_term_id', $stored_term->term_id );
					}
				} else {
					delete_post_meta( $post_id, 'bb_access_term_id' );
				}
			}
		}
	}

	/**
	 * Function will run after every 30 sec if found any taxonomy job.
	 *
	 * @since 1.6.3
	 *
	 * @return void
	 */
	public function ajax_get_taxonomy_job_status() {
		check_ajax_referer( 'bbapp_sort_nonce_' . get_current_user_id(), 'sort_nonce' );

		$is_running = false;
		if ( true === Jobs::instance()->determine_process_running( 'bb_ac_term_calc' ) || true === Jobs::instance()->determine_process_running( 'bb_store_meta_by_ids' ) ) {
			$is_running = true;
		}

		wp_send_json_success(
			array(
				'is_running' => $is_running,
			)
		);
	}

	/**
	 * Remove user from condition when user deleted.
	 *
	 * @param int $user_id User id.
	 *
	 * @since 1.7.7
	 *
	 * @return void
	 */
	public function remove_user_from_condition( $user_id ) {
		if ( empty( $user_id ) ) {
			return;
		}

		$access_groups = bb_access_controls_get_groups( array( 'include_members' => array( $user_id ) ) );

		if ( ! empty( $access_groups['result'] ) ) {
			foreach ( $access_groups['result'] as $access_group ) {
				if ( ! empty( $access_group['condition_name'] ) && ! empty( $access_group['item_value'] ) ) {
					bb_access_controls_condition_remove_user( $user_id, $access_group['condition_name'], $access_group['item_value'] );
				}
			}
		}
	}
}
