<?php
/**
 * Used by Access rules functions.
 *
 * @package BuddyBossApp\AccessControls
 */

namespace BuddyBossApp\AccessControls;

/**
 * Used by access rule to process data.
 *
 * Class AccessRule
 *
 * @package BuddyBossApp\AccessControls
 */
class AccessRule {

	/**
	 * Access rules table name.
	 *
	 * @var string $table_access_rules Access rules table name.
	 */
	private $table_access_rules = 'bb_access_rules';

	/**
	 * Access rules groups table name.
	 *
	 * @var string $table_access_rules_groups Access rules groups table name.
	 */
	private $table_access_rules_groups = 'bb_access_rules_groups';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class construct
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return AccessRule
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Insert access rule into database.
	 *
	 * @param array $args      argument of access rules.
	 * @param array $group_ids argument of access groups ids.
	 *
	 * @since  1.5.2.1
	 * @return int|\WP_Error
	 */
	public function insert_access_rule( $args, $group_ids = array() ) {
		global $wpdb;

		$default_args = array(
			'item_type'       => '',
			'item_id'         => '',
			'group_condition' => '',
			'rule_condition'  => '',
			'data'            => '',
			'date'            => gmdate( 'Y-m-d H:i:s' ),
		);

		$args = wp_parse_args( $args, $default_args );

		if ( ! empty( $args['data'] ) ) {
			$args['data'] = array_merge( $args['data'], array( 'order_group_ids' => $group_ids ) );
			$args['data'] = maybe_serialize( $args['data'] );
		}

		$create = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			"{$wpdb->prefix}{$this->table_access_rules}",
			$args,
			array(
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
			)
		);

		if ( ! empty( $create ) ) {
			if ( ! empty( $group_ids ) ) {
				self::instance()->rule_groups_add_update_groups( $wpdb->insert_id, $group_ids );
			}

			return $wpdb->insert_id;
		}

		return new \WP_Error( 'error_insert_access_rules', __( 'There is a database error while adding access rule record.', 'buddyboss-app' ) );
	}

	/**
	 * Update access rule into database.
	 *
	 * @param int   $id        Access rule id.
	 * @param array $args      argument of access rules.
	 * @param array $group_ids argument of group ids.
	 *
	 * @since  1.5.2.1
	 * @return int|\WP_Error
	 */
	public function update_access_rule( $id, $args, $group_ids = array() ) {
		global $wpdb;

		$updated_args = array();

		if ( isset( $args['item_type'] ) ) {
			$updated_args['item_type'] = $args['item_type'];
		}

		if ( isset( $args['item_id'] ) ) {
			$updated_args['item_id'] = $args['item_id'];
		}

		if ( isset( $args['group_condition'] ) ) {
			$updated_args['group_condition'] = $args['group_condition'];
		}

		if ( isset( $args['rule_condition'] ) ) {
			$updated_args['rule_condition'] = $args['rule_condition'];
			if ( 'all-logged-in-members' === $args['rule_condition'] ) {
				$updated_args['group_condition'] = '';
			}
		}

		if ( isset( $args['data'] ) ) {
			$args['data']         = array_merge( $args['data'], array( 'order_group_ids' => $group_ids ) );
			$updated_args['data'] = maybe_serialize( $args['data'] );
		}

		$updated_args['date'] = ! empty( $args['date'] ) ? $args['date'] : gmdate( 'Y-m-d H:i:s' );
		$update               = $wpdb->update( "{$wpdb->prefix}{$this->table_access_rules}", $updated_args, array( 'id' => $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( ! empty( $update ) ) {
			if ( ! empty( $group_ids ) ) {
				self::instance()->rule_groups_add_update_groups( $id, $group_ids );
			} else {

				self::instance()->delete_access_rule_groups( $id );
			}

			return $update;
		}

		return new \WP_Error( 'error_update_access_rules', __( 'There is a database error while updating group record.', 'buddyboss-app' ) );
	}

	/**
	 * Function to use add or update user based on group and condition.
	 *
	 * @param int   $rule_id   access rule id.
	 * @param array $group_ids access rule groups ids.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function rule_groups_add_update_groups( $rule_id, $group_ids ) {
		global $wpdb;

		$rule_groups = $this->rule_groups_has_groups( $rule_id, $group_ids );

		if ( ! empty( $rule_groups['add'] ) ) {
			$values = array();

			foreach ( $rule_groups['add'] as $group_id ) {
				$values[] = $wpdb->prepare( '(%d,%d,%s)', $group_id, $rule_id, gmdate( 'Y-m-d H:i:s' ) );
			}

			if ( ! empty( $values ) ) {
				$query  = "INSERT INTO {$wpdb->prefix}{$this->table_access_rules_groups} (group_id, rule_id, date) VALUES ";
				$query .= implode( ",\n", $values );
				$wpdb->query( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			}
		}

		if ( ! empty( $rule_groups['update'] ) ) {
			$in_placeholders = implode( ', ', array_fill( 0, count( $rule_groups['update'] ), '%s' ) );
			$where_clause    = $wpdb->prepare( "id IN ({$in_placeholders})", $rule_groups['update'] ); // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$query           = $wpdb->prepare( "UPDATE {$wpdb->prefix}{$this->table_access_rules_groups} SET date=%s WHERE rule_id = %d AND {$where_clause}", gmdate( 'Y-m-d H:i:s' ), $rule_id ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$wpdb->query( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		}

		if ( ! empty( $rule_groups['delete'] ) ) {
			$ids = implode( ',', array_map( 'absint', $rule_groups['delete'] ) );
			$wpdb->query( "DELETE FROM {$wpdb->prefix}{$this->table_access_rules_groups} WHERE id IN({$ids})" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		return true;
	}

	/**
	 * Function to check member exists on provided group.
	 *
	 * @param int   $rule_id   Access Rules ID.
	 * @param array $group_ids Group ids.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void
	 */
	public function rule_groups_has_groups( $rule_id, $group_ids ) {
		global $wpdb;

		if ( empty( $group_ids ) ) {
			return array();
		}

		$include_implode    = implode( ',', $group_ids );
		$exists_rule_groups = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}{$this->table_access_rules_groups} WHERE rule_id = %d AND group_id IN ({$include_implode})", $rule_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$not_in_rule_groups = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}{$this->table_access_rules_groups} WHERE rule_id = %d AND group_id NOT IN ({$include_implode})", $rule_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( is_wp_error( $exists_rule_groups ) ) {
			$exists_rule_groups = array();
		}

		if ( is_wp_error( $not_in_rule_groups ) ) {
			$not_in_rule_groups = array();
		}

		$access_rule_groups = array(
			'add'    => array_diff( $group_ids, wp_list_pluck( $exists_rule_groups, 'group_id' ) ), // User ids.
			'update' => wp_list_pluck( $exists_rule_groups, 'id' ), // Group member ids.
			'delete' => wp_list_pluck( $not_in_rule_groups, 'id' ), // Group member ids.
		);

		/**
		 * Fires after checked a groups is exists on access rule groups.
		 *
		 * @param array $access_rule_groups Access rule groups.
		 *
		 * @since 1.5.2.1
		 *
		 * @return array (
		 *      'add' get the groups ids.
		 *      'update' get the rule group ids.
		 *      'delete' get the group ids.
		 * )
		 */
		return apply_filters( 'bb_access_controls_rule_group_has_groups', $access_rule_groups );
	}

	/**
	 * Function to get specific access rule.
	 *
	 * @param int    $item_id   item id.
	 * @param string $item_type item type.
	 *
	 * @since 1.5.2.1
	 *
	 * @return array|boolean
	 */
	public function get_access_rule( $item_id, $item_type ) {
		$rule_data = array();

		if ( empty( $item_id ) || empty( $item_type ) ) {
			return $rule_data;
		}

		$rule = $this->get_access_rules(
			array(
				'include_item_id'    => $item_id,
				'include_item_types' => $item_type,
			)
		);

		return ! empty( $rule['results'][0] ) ? $rule['results'][0] : false;
	}

	/**
	 * Select the access_rules from database based on args.
	 *
	 * @param array $args                       {.
	 *
	 * @type string   $exclude_duplicate          Providing a column will exclude all duplicates from results. Default: item_id Enum (item_id|item_secondary_id|false)
	 * @type int|bool $per_page                   Number of results per page. Default: false.
	 * @type int      $page                       Which page of results to fetch. Using page=1 without per_page will result
	 *                                            in no pagination. Default: 1.
	 * @type string   $order_by                   Column to order results by.
	 * @type string   $order                      ASC or DESC. Default: 'DESC'.
	 * @type array    $include_item_types         Array of items types OR comma separated items types to limit query by (IN). Default: false.
	 * @type array    $include_item_id            Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $include_group_condition    Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $exclude_item_types         Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type array    $exclude_item_id            Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type array    $exclude_group_condition    Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type boolean  $only_count                 True or false if you want to return only counts of selection. Default: false.
	 * @type array    $include_group_id           Array of group ids OR comma separated group ids to limit query by (IN). Default: false.
	 * @type array    $exclude_group_id           Array of group ids OR comma separated group ids to limit query by (NOT IN). Default: false.
	 *                                            }
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function get_access_rules( $args ) {
		global $wpdb;

		$default_args = array(
			'per_page'                => false,
			'page'                    => 1,
			'orderby'                 => 'id',
			'order'                   => 'desc',
			'include'                 => false,
			'include_item_types'      => false,
			'include_item_id'         => false,
			'include_group_condition' => false,
			'exclude'                 => false,
			'exclude_item_types'      => false,
			'exclude_item_id'         => false,
			'exclude_group_condition' => false,
			'exclude_duplicate'       => false,
			'include_group_id'        => false,
			'exclude_group_id'        => false,
			'only_count'              => false,
		);

		$args         = wp_parse_args( $args, $default_args );
		$where_clause = array();
		$join         = '';
		$arg_columns  = '';

		// Join access rule groups table if it's profile page.
		if ( false !== $args['include_group_id'] || false !== $args['exclude_group_id'] ) {
			$join       .= " LEFT JOIN {$wpdb->prefix}{$this->table_access_rules_groups} AS arg ON ( ar.id = arg.rule_id ) ";
			$arg_columns = ', arg.group_id';
		}

		// Include Rule id.
		if ( false !== $args['include'] ) {
			$include = $args['include'];

			if ( ! is_array( $include ) ) {
				$include = explode( ',', $include );
			}

			if ( ! empty( $include ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.id IN ({$in_placeholders})", $include ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include Types.
		if ( false !== $args['include_item_types'] ) {
			$include_types = $args['include_item_types'];
			if ( ! is_array( $include_types ) ) {
				$include_types = explode( ',', $include_types );
			}

			if ( ! empty( $include_types ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_types ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.item_type IN ({$in_placeholders})", $include_types ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include item_id.
		if ( false !== $args['include_item_id'] ) {
			$include_items = $args['include_item_id'];

			if ( ! is_array( $include_items ) ) {
				$include_items = explode( ',', $include_items );
			}

			if ( ! empty( $include_items ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_items ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.item_id IN ($in_placeholders)", $include_items ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include group_condition.
		if ( false !== $args['include_group_condition'] ) {
			$include_group_condition = $args['include_group_condition'];

			if ( ! is_array( $include_group_condition ) ) {
				$include_group_condition = explode( ',', $include_group_condition );
			}

			if ( ! empty( $include_group_condition ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_group_condition ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.group_condition IN ({$in_placeholders})", $include_group_condition ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude rule id.
		if ( false !== $args['exclude'] ) {
			$exclude = $args['exclude'];

			if ( ! is_array( $exclude ) ) {
				$exclude = explode( ',', $exclude );
			}

			if ( ! empty( $exclude ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.id NOT IN ({$in_placeholders})", $exclude ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude Item Type.
		if ( false !== $args['exclude_item_types'] ) {
			$exclude_type = $args['exclude_item_types'];

			if ( ! is_array( $exclude_type ) ) {
				$exclude_type = explode( ',', $exclude_type );
			}

			if ( ! empty( $exclude_type ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_type ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.item_type NOT IN ({$in_placeholders})", $exclude_type ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude Item id.
		if ( false !== $args['exclude_item_id'] ) {
			$exclude_items = $args['exclude_item_id'];

			if ( ! is_array( $exclude_items ) ) {
				$exclude_items = explode( ',', $exclude_items );
			}

			if ( ! empty( $exclude_items ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_items ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.item_id NOT IN ({$in_placeholders})", $exclude_items ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include group id.
		if ( false !== $args['include_group_id'] ) {
			$include_group_id = $args['include_group_id'];

			if ( ! is_array( $include_group_id ) ) {
				$include_group_id = explode( ',', $include_group_id );
			}

			if ( ! empty( $include_group_id ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_group_id ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "arg.group_id IN ({$in_placeholders})", $include_group_id ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude group id.
		if ( false !== $args['exclude_group_id'] ) {
			$exclude_items = $args['exclude_group_id'];

			if ( ! is_array( $exclude_items ) ) {
				$exclude_items = explode( ',', $exclude_items );
			}

			if ( ! empty( $exclude_items ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_items ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "arg.group_id NOT IN ({$in_placeholders})", $exclude_items ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude group_condition.
		if ( false !== $args['exclude_group_condition'] ) {
			$exclude_group_condition = $args['exclude_group_condition'];

			if ( ! is_array( $exclude_group_condition ) ) {
				$exclude_group_condition = explode( ',', $exclude_group_condition );
			}

			if ( ! empty( $exclude_group_condition ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_group_condition ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "ar.group_condition NOT IN ({$in_placeholders})", $exclude_group_condition ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		$where_conditions = ( ! empty( $where_clause ) ) ? 'WHERE ' . implode( ' AND ', $where_clause ) : '';
		$limit_clause     = '';

		if ( $args['per_page'] ) {
			$args['per_page'] = (int) $args['per_page'];
			$args['page']     = (int) $args['page'];

			$limit_clause  = 'LIMIT ' . $args['per_page'] . ' ';
			$limit_clause .= 'OFFSET ' . ( $args['page'] - 1 ) * $args['per_page'];
		}

		$order_clause = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order_clause .= 'ORDER BY ar.' . esc_sql( $args['orderby'] );
			$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$groupby      = '';
		$count_select = 'COUNT(ar.id)';

		if ( $args['exclude_duplicate'] ) {
			$groupby      = "group BY {$args["exclude_duplicate"]}";
			$count_select = "COUNT(DISTINCT {$args["exclude_duplicate"]})";
		}

		$access_rule_data = array();

		if ( false === $args['only_count'] ) {
			$sql              = "SELECT ar.* {$arg_columns} FROM {$wpdb->prefix}{$this->table_access_rules} AS ar {$join} {$where_conditions} {$groupby} {$order_clause} {$limit_clause}";
			$access_rule_data = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		}

		$sql    = "SELECT {$count_select} as count FROM {$wpdb->prefix}{$this->table_access_rules} AS ar {$join} {$where_conditions} {$order_clause}";
		$count  = $wpdb->get_row( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$retval = array();

		if ( ! empty( $access_rule_data ) ) {
			foreach ( $access_rule_data as $access_rule ) {
				$retval[] = $this->prepare_rule_data( $access_rule );
			}
		}

		return array(
			'results' => $retval,
			'count'   => isset( $count->count ) ? $count->count : null,
		);
	}

	/**
	 * Prepare additional information on access rule data object.
	 *
	 * @param object $rule Rule data object.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|void $data
	 */
	public function prepare_rule_data( $rule ) {
		$data = array(
			'id'        => ( ! empty( $rule->id ) ) ? $rule->id : 0,
			'item_type' => ( ! empty( $rule->item_type ) ) ? $rule->item_type : '',
			'item_id'   => ( ! empty( $rule->item_id ) ) ? $rule->item_id : '',
			'rule_data' => ( ! empty( $rule->data ) ) ? maybe_unserialize( $rule->data ) : array(),
			'date'      => ( ! empty( $rule->date ) ) ? $rule->date : '',
		);

		$data['rule_data']['group_condition'] = ( ! empty( $rule->group_condition ) ) ? $rule->group_condition : '';
		$data['rule_data']['rule_condition']  = ( ! empty( $rule->rule_condition ) ) ? $rule->rule_condition : '';

		$rule_groups = $this->get_access_rule_groups(
			array(
				'include_rule_id' => $rule->id,
			)
		);

		$data['rule_data']['group_ids'] = ! empty( $rule_groups['count'] ) ? wp_list_pluck( $rule_groups['results'], 'group_id' ) : array();

		/**
		 * Fires after prepared access rule data.
		 *
		 * @param array $data Prepare Access rule.
		 * @param array $rule Access rule.
		 *
		 * @since 1.5.2.1
		 *
		 * @return array
		 */
		return apply_filters( 'bb_access_controls_prepare_rule_data', $data, $rule );
	}

	/**
	 * Select the access_rules from database based on args.
	 *
	 * @param array $args                       {.
	 *
	 * @type string   $exclude_duplicate          Providing a column will exclude all duplicates from results. Default: item_id Enum (item_id|item_secondary_id|false)
	 * @type int|bool $per_page                   Number of results per page. Default: false.
	 * @type int      $page                       Which page of results to fetch. Using page=1 without per_page will result
	 *                                            in no pagination. Default: 1.
	 * @type string   $order_by                   Column to order results by.
	 * @type string   $order                      ASC or DESC. Default: 'DESC'.
	 * @type array    $include_item_types         Array of items types OR comma separated items types to limit query by (IN). Default: false.
	 * @type array    $include_item_id            Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $include_group_condition    Array of items ids OR comma separated items ids to limit query by (IN). Default: false.
	 * @type array    $exclude_item_types         Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type array    $exclude_item_id            Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type array    $exclude_group_condition    Array of items ids OR comma separated items ids to limit query by (NOT IN). Default: false.
	 * @type boolean  $only_count                 True or false if want to return only counts of selection. Default: false.
	 *                                            }
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function get_access_rule_groups( $args ) {
		global $wpdb;

		$default_args = array(
			'per_page'          => false,
			'page'              => 1,
			'orderby'           => 'id',
			'order'             => 'desc',
			'include_group_id'  => false,
			'include_rule_id'   => false,
			'exclude_group_id'  => false,
			'exclude_rule_id'   => false,
			'exclude_duplicate' => false,
			'only_count'        => false,
		);

		$args         = wp_parse_args( $args, $default_args );
		$where_clause = array();

		// Include group_id.
		if ( false !== $args['include_group_id'] ) {
			$include_id = $args['include_group_id'];

			if ( ! is_array( $include_id ) ) {
				$include_id = explode( ',', $include_id );
			}

			if ( ! empty( $include_id ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_id ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "group_id IN ({$in_placeholders})", $include_id ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Include rule_id.
		if ( false !== $args['include_rule_id'] ) {
			$include_id = $args['include_rule_id'];

			if ( ! is_array( $include_id ) ) {
				$include_id = explode( ',', $include_id );
			}

			if ( ! empty( $include_id ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $include_id ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "rule_id IN ({$in_placeholders})", $include_id ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude Item group_id.
		if ( false !== $args['exclude_group_id'] ) {
			$exclude_id = $args['exclude_group_id'];

			if ( ! is_array( $exclude_id ) ) {
				$exclude_id = explode( ',', $exclude_id );
			}

			if ( ! empty( $exclude_id ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_id ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "group_id NOT IN ({$in_placeholders})", $exclude_id ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		// Exclude rule_id.
		if ( false !== $args['exclude_rule_id'] ) {
			$exclude_id = $args['exclude_rule_id'];
			if ( ! is_array( $exclude_id ) ) {
				$exclude_id = explode( ',', $exclude_id );
			}

			if ( ! empty( $exclude_id ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $exclude_id ), '%s' ) );
				$where_clause[]  = $wpdb->prepare( "rule_id NOT IN ({$in_placeholders})", $exclude_id ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		$where_conditions = ( ! empty( $where_clause ) ) ? 'WHERE ' . implode( ' AND ', $where_clause ) : '';
		$limit_clause     = '';

		if ( $args['per_page'] ) {
			$args['per_page'] = (int) $args['per_page'];
			$args['page']     = (int) $args['page'];

			$limit_clause  = "LIMIT {$args['per_page']} ";
			$limit_clause .= 'OFFSET ' . ( $args['page'] - 1 ) * $args['per_page'];
		}

		$order_clause = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order_clause .= 'ORDER BY ' . esc_sql( $args['orderby'] );
			$order_clause .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$groupby      = '';
		$count_select = 'COUNT(id)';

		if ( $args['exclude_duplicate'] ) {
			$groupby      = "group BY {$args["exclude_duplicate"]}";
			$count_select = "COUNT(DISTINCT {$args["exclude_duplicate"]})";
		}

		$access_rule_groups = array();

		if ( false === $args['only_count'] ) {
			$sql                = "SELECT * FROM {$wpdb->prefix}{$this->table_access_rules_groups} {$where_conditions} {$groupby} {$order_clause} {$limit_clause}";
			$access_rule_groups = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
		}

		$sql   = "SELECT {$count_select} as count FROM {$wpdb->prefix}{$this->table_access_rules_groups} {$where_conditions} {$order_clause}";
		$count = $wpdb->get_row( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		return array(
			'results' => $access_rule_groups,
			'count'   => isset( $count->count ) ? $count->count : null,
		);
	}

	/**
	 * Function to delete access rule.
	 *
	 * @param int $rule_id Access rule id.
	 *
	 * @since 1.5.2.1
	 * @return bool|int|resource|\WP_Error
	 */
	public function delete_access_rule( $rule_id ) {
		global $wpdb;

		$delete_args  = array(
			'id' => $rule_id,
		);
		$access_rules = $this->get_access_rules( array( 'include' => $rule_id ) );

		if ( empty( $access_rules['count'] ) ) {
			return false;
		}

		$delete = $wpdb->delete( "{$wpdb->prefix}{$this->table_access_rules}", $delete_args );  //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( $delete ) {
			$item_id   = ! empty( $access_rules['results']['0']['item_id'] ) ? $access_rules['results']['0']['item_id'] : '';
			$item_type = ! empty( $access_rules['results']['0']['item_type'] ) ? $access_rules['results']['0']['item_type'] : '';

			/**
			 * Fires before access rule delete.
			 *
			 * @type array  $item_id   Array of item ids.
			 * @type string $item_type Item type.
			 */
			do_action( 'bb_access_controls_delete_access_rules', array( $item_id ), $item_type );

			$this->delete_access_rule_groups( $rule_id );

			return true;
		}

		return new \WP_Error( 'error_delete_access_rule', __( 'There is a database error while deleting access rule record.', 'buddyboss-app' ) );
	}

	/**
	 * Function to delete access rule groups.
	 *
	 * @param int $rule_id Access rule id.
	 *
	 * @since 1.5.2.1
	 * @return bool|int|resource|\WP_Error
	 */
	public function delete_access_rule_groups( $rule_id ) {
		global $wpdb;

		$delete_args = array(
			'rule_id' => $rule_id,
		);

		$delete = $wpdb->delete( "{$wpdb->prefix}{$this->table_access_rules_groups}", $delete_args );  //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( ! $delete ) {
			return new \WP_Error( 'error_delete_access_rule', __( 'There is a database error while deleting access rule record.', 'buddyboss-app' ) );
		}

		return $delete;
	}

	/**
	 * This function to save (add abd update) access rule data.
	 *
	 * @param int|array $item_ids  item id.
	 * @param string    $item_type item type.
	 * @param array     $settings  Rule settings.
	 *
	 * @since 1.5.2.1
	 *
	 * @return bool
	 */
	public function save_access_rules( $item_ids, $item_type, $settings ) {
		$saved = false;

		// Don't save if it's not a registered item type.
		if ( ! bb_access_controls_is_rules_item_type_registered( $item_type ) ) {
			return false;
		}

		$access_rules = $this->get_access_rules(
			array(
				'include_item_types' => $item_type,
				'include_item_id'    => $item_ids,
			)
		);
		$insert_rules = array(
			'update' => array(),
			'add'    => $item_ids,
		);

		if ( count( $item_ids ) !== $access_rules['count'] ) {
			$saved_item_ids = wp_list_pluck( $access_rules['results'], 'item_id' );
			$saved_ids      = wp_list_pluck( $access_rules['results'], 'id' );

			$new_rule_item_ids      = array_diff( $item_ids, $saved_item_ids );
			$insert_rules['update'] = $saved_ids;
			$insert_rules['add']    = $new_rule_item_ids;
		}

		if ( ! empty( $insert_rules['update'] ) ) {
			foreach ( $insert_rules['update'] as $rule_id ) {
				$update_args = array(
					'group_condition' => $settings['group_condition'],
					'data'            => $settings['data'],
					'date'            => gmdate( 'Y-m-d H:i:s' ),
				);

				if ( isset( $settings['rule_condition'] ) ) {
					$update_args['rule_condition'] = $settings['rule_condition'];
				}

				$update_rule = $this->update_access_rule( $rule_id, $update_args, $settings['group_ids'] );
			}

			if ( ! empty( $update_rule ) && ! is_wp_error( $update_rule ) ) {
				$saved = true;
			}
		}

		if ( ! empty( $insert_rules['add'] ) ) {
			foreach ( $insert_rules['add'] as $item_id ) {
				$items_args = array(
					'item_type'       => $item_type,
					'item_id'         => $item_id,
					'group_condition' => $settings['group_condition'],
					'data'            => $settings['data'],
					'date'            => gmdate( 'Y-m-d H:i:s' ),
				);

				if ( isset( $settings['rule_condition'] ) ) {
					$items_args['rule_condition'] = $settings['rule_condition'];
				}

				$rule_id = $this->insert_access_rule( $items_args, $settings['group_ids'] );

				/**
				 * Fires after rule data saved.
				 *
				 * @type int    $item_id   Rule item id.
				 * @type string $item_type Rule item type.
				 */
				do_action( "bb_access_controls_after_saved_rule_data_{$item_type}", $item_id, $item_type );
			}

			if ( ! empty( $rule_id ) && ! is_wp_error( $rule_id ) ) {
				$saved = true;
			}
		}

		/**
		 * Fires after access rules saved.
		 *
		 * @type array  $item_ids  Access rules item ids.
		 * @type string $item_type Access rules item type.
		 */
		do_action( 'bb_access_controls_save_access_rules', $item_ids, $item_type );

		return $saved;
	}
}
