<?php
/**
 * Holds Access Controls initialization functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\AccessControls\Helpers;
use BuddyBossApp\AccessControls\Core\Core;
use BuddyBossApp\AccessControls\Performance;
use BuddyBossApp\AccessControls\Segment;
use BuddyBossApp\AccessControls\Actions;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Access Controls main class.
 */
final class AccessControls {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Integration array.
	 *
	 * @var array $integration
	 */
	public $integration = array();

	/**
	 * Conditions array.
	 *
	 * @var array $conditions
	 */
	public $conditions = array();

	/**
	 * Rule item type.
	 *
	 * @var array $rules_item_types
	 */
	public $rules_item_types = array();

	/**
	 * Controller constructor.
	 *
	 * @since 1.5.2.1
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return AccessControls|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {

			// include functions(utils) file.
			require_once dirname( __FILE__ ) . '/AccessControls/functions.php';

			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->setup_globals();
			self::$instance->load_classes();
			self::$instance->setup_actions();
			self::$instance->setup_filters();
		}

		return self::$instance;
	}

	/**
	 * Setup Globals.
	 *
	 * @since 1.5.2.1
	 */
	protected function setup_globals() {
	}

	/**
	 * Initiate the required classes.
	 *
	 * @since 1.5.2.1
	 */
	protected function load_classes() {
	}

	/**
	 * Action hooks
	 *
	 * @since 1.5.2.1
	 */
	protected function setup_actions() {
		add_action( 'init', array( $this, 'load_integrations' ), 1 );
	}

	/**
	 * Define all filter hooks.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	protected function setup_filters() {
		// User segment for push notification.
		add_filter( 'bbapp_user_segment_integrations', array( $this, 'load_segment' ), 11 );
	}

	/**
	 * Load the integrations.
	 *
	 * @since 1.5.2.1
	 */
	public function load_integrations() {

		/**
		 * Load Heloer
		 */
		Helpers::instance();

		/**
		 * Load Actions
		 */
		Actions::instance();

		/**
		 * Load the Core Integration for Access Controls.
		 */
		Core::instance();

		if ( bbapp_is_active( 'performance' ) ) {
			Performance::instance();
		}

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			\BuddyBossApp\Integrations\BuddyBossPlatform\AccessControls::instance();
		}

		if ( bbapp_is_learndash_enabled() ) {
			\BuddyBossApp\Integrations\Learndash\AccessControls::instance(); // LearnDash Course Conditions.
		}

		if ( defined( 'MEPR_VERSION' ) && class_exists( '\MeprCtrlFactory' ) ) {
			\BuddyBossApp\Integrations\MemberPress\AccessControls::instance(); // MemberPress Conditions.
		}

		if ( function_exists( 'pmpro_changeMembershipLevel' ) ) {
			\BuddyBossApp\Integrations\PmProMembership\AccessControls::instance(); // Paid Membership Pro Conditions.
		}

		if ( class_exists( '\WishListMember' ) ) {
			\BuddyBossApp\Integrations\WishlistMember\AccessControls::instance(); // Wishlist Member Conditions.
		}

		if ( class_exists( '\WC_Memberships' ) ) {
			\BuddyBossApp\Integrations\WcMembership\AccessControls::instance(); // WooCommerce Membership Conditions.
		}

		if ( class_exists( '\RCP_Requirements_Check' ) ) {
			\BuddyBossApp\Integrations\RestrictContentPro\AccessControls::instance(); // Restrict Content Pro Conditions.
		}

		if ( defined( 'GAMIPRESS_VER' ) ) {
			\BuddyBossApp\Integrations\GamiPress\AccessControls::instance(); // GamiPress point and achievement condition.
		}

		if ( bbapp_is_tutor_lms_plugins_active() ) {
			\BuddyBossApp\Integrations\TutorLMS\AccessControls::instance(); // GamiPress point and achievement condition.
		}

		// Load the API.
		\BuddyBossApp\Api\AccessControls\Main::instance();
	}

	/**
	 * When BuddyBoss App plugin is activated
	 *
	 * @since 1.5.2.1
	 */
	public function on_activation() {

		global $wpdb;

		$db_prefix = $wpdb->prefix;

		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$access_groups_table_name = "{$db_prefix}bb_access_groups";

		$sql = "CREATE TABLE {$access_groups_table_name} (
			id bigint(20) unsigned NOT NULL auto_increment,
			status tinyint(1) DEFAULT 2,
			condition_name varchar(100) DEFAULT NULL,
			sub_condition_name varchar(50) DEFAULT NULL,
			item_value varchar(100) DEFAULT NULL,
			data text DEFAULT NULL,
			is_calculating tinyint(1) DEFAULT NULL,
			date_created datetime DEFAULT '0000-00-00 00:00:00' NULL,
			date_calculated datetime DEFAULT '0000-00-00 00:00:00' NULL,
			PRIMARY KEY (id),
			KEY status (status),
			KEY condition_name (condition_name),
			KEY sub_condition_name (sub_condition_name),
			KEY item_value (item_value),
			KEY is_calculating (is_calculating),
			KEY date_created (date_created),
			KEY date_calculated (date_calculated)
        ) {$charset_collate}";

		dbDelta( $sql );

		$access_groups_members_table_name = "{$db_prefix}bb_access_groups_members";

		$sql = "CREATE TABLE {$access_groups_members_table_name} (
			id bigint(20) unsigned NOT NULL auto_increment,
			group_id bigint(20) unsigned NOT NULL,
			user_id bigint(20) unsigned NOT NULL,
			date_added datetime DEFAULT '0000-00-00 00:00:00' NULL,
			status tinyint(1) DEFAULT 2,
			PRIMARY KEY (id),
			KEY group_id (group_id),
			KEY user_id (user_id),
			KEY date_added (date_added),
			KEY status (status)
		) {$charset_collate}";

		dbDelta( $sql );

		$access_group_temp_items_table_name = "{$db_prefix}bb_access_group_tmp_items";

		$sql = "CREATE TABLE {$access_group_temp_items_table_name} (
			id bigint(20) unsigned NOT NULL auto_increment,
			item_type varchar(50) DEFAULT NULL,
			item_id bigint(20) unsigned NOT NULL,
			item_secondary_id bigint(20) unsigned NOT NULL,
			date_added datetime DEFAULT '0000-00-00 00:00:00' NULL,
			PRIMARY KEY (id),
			KEY item_type (item_type),
			KEY item_id (item_id),
			KEY item_secondary_id (item_secondary_id),
			KEY date_added (date_added)
		) {$charset_collate}";

		dbDelta( $sql );

		$access_rules_table_name = "{$db_prefix}bb_access_rules";

		$sql = "CREATE TABLE {$access_rules_table_name} (
			id bigint(20) unsigned NOT NULL auto_increment,
			item_type varchar(100) DEFAULT NULL,
			item_id varchar(100) DEFAULT NULL,
			group_condition varchar(5) DEFAULT NULL,
			rule_condition varchar(30) DEFAULT NULL,
			data text DEFAULT NULL,
			date datetime DEFAULT '0000-00-00 00:00:00' NULL,
			PRIMARY KEY (id),
			KEY item_type (item_type),
			KEY item_id (item_id),
			KEY group_condition (group_condition),
			KEY date (date)
		) {$charset_collate}";

		dbDelta( $sql );

		$access_rules__groups_table_name = "{$db_prefix}bb_access_rules_groups";

		$sql = "CREATE TABLE {$access_rules__groups_table_name} (
			id bigint(20) unsigned NOT NULL auto_increment,
			group_id bigint(20) unsigned NOT NULL,
			rule_id bigint(20) unsigned NOT NULL,
			date datetime DEFAULT '0000-00-00 00:00:00' NULL,
			PRIMARY KEY (id),
			KEY group_id (group_id),
			KEY rule_id (rule_id),
			KEY date (date)
		) {$charset_collate}";

		dbDelta( $sql );
	}

	/**
	 * Access controll related user segments.
	 *
	 * @param array $integrations List of integration.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed|array
	 */
	public function load_segment( $integrations ) {

		$integrations['access_controls'] = new Segment();

		return $integrations;
	}
}
