<?php
/**
 * Plugin Name: BuddyBoss App
 * Description: The BuddyBoss App plugin allows you to sync your community and courses into a native mobile app.
 * Version: 2.10.1
 * Author: BuddyBoss
 * Author URI:  https://www.buddyboss.com
 * Text Domain: buddyboss-app
 * Domain Path: /languages
 *
 * @package BuddyBoss App
 */

// Make sure we don't expose any info if called directly.
if ( ! function_exists( 'add_action' ) ) {
	die( "Sorry, you can't access this directly - Security established" );
}

if ( ! defined( 'BBAPP_LOAD_MIGRATION_CODE' ) ) {
	define( 'BBAPP_LOAD_MIGRATION_CODE', true );
}

// Check Min php version.
if ( version_compare( PHP_VERSION, '7.2', '<' ) ) {
	/**
	 * Admin notice for not supported PHP version.
	 */
	function bbapp_min_supported_php_admin_notices() {
		if ( version_compare( PHP_VERSION, '7.2', '<' ) ) {
			/* translators: 1. div classes. 2. notice message. */
			printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( 'notice notice-error' ), esc_html__( 'The BuddyBoss App plugin requires a minimum PHP version 7.2.', 'buddyboss-app' ) );
		}
	}

	add_action( 'admin_notices', 'bbapp_min_supported_php_admin_notices' );

	return false;
}

$base_dir = dirname( __FILE__ );

/**
 * Always Load BuddyBossApp Instance First!.
 */
bbapp::instance( __FILE__ );

// Load WP Core Extended API.
\BuddyBossApp\Api\WpCore\Main::instance();

// Load Learndash API.
\BuddyBossApp\Api\LearnDash\Main::instance();

// Load TutorLMS API.
\BuddyBossApp\Api\TutorLMS\Main::instance();

\BuddyBossApp\Api\Core\Main::instance(); // if platform disable provide support.

// Deprecated rest feature provide support.
\BuddyBossApp\Api\Deprecated\Main::instance();

use BuddyBossApp\Admin\SetupAdmin;
use BuddyBossApp\Admin\Tools\Telemetry;
use BuddyBossApp\App\App;
use BuddyBossApp\AppSettings;
use BuddyBossApp\AppStores;
use BuddyBossApp\BuddyBossAppApi;
use BuddyBossApp\Build;
use BuddyBossApp\ClientApiPluginSupport;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\Common\IconPicker;
use BuddyBossApp\DeepLinking\DeepLinking;
use BuddyBossApp\HealthCheck;
use BuddyBossApp\Integrations\PmPro\PmProForumSupport;
use BuddyBossApp\Integrations\PmPro\PmProMembershipSupport;
use BuddyBossApp\ItemMeta\CourseMeta;
use BuddyBossApp\ItemMeta\LessonMeta;
use BuddyBossApp\ItemMeta\MetaApi;
use BuddyBossApp\ItemMeta\TopicMeta;
use BuddyBossApp\ManageApp;
use BuddyBossApp\NativeAppPage;
use BuddyBossApp\Network;
use BuddyBossApp\SmartBanner;
use BuddyBossApp\Styling;
use BuddyBossApp\Tools\Logger\BgProcessLog;

/**
 * BuddyBOss App main class.
 */
class bbapp { // phpcs:ignore

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Plugin URL.
	 *
	 * @var $plugin_url
	 */
	public $plugin_url;

	/**
	 * Plugin directory.
	 *
	 * @var $plugin_dir
	 */
	public $plugin_dir;

	/**
	 * Plugin prefix ( & _ ).
	 *
	 * @var $plugin_prefix
	 */
	public $plugin_prefix;

	/**
	 * Plugin version.
	 *
	 * @var $plugin_version
	 */
	public $plugin_version;

	/**
	 * It's used for language domain.
	 *
	 * @var $domain
	 */
	public $domain;

	/**
	 * Plugin root file.
	 *
	 * @var $root_file
	 */
	public $root_file;

	/**
	 * If plugin is network activated.
	 *
	 * @var bool $network_activated
	 */
	public $network_activated = false;

	/**
	 * Database version.
	 *
	 * @var string $db_version
	 */
	public $db_version = '32';

	/**
	 * Plugin aws url.
	 *
	 * @var string $plugin_aws_url
	 */
	public $plugin_aws_url;

	/**
	 * Transient Key.
	 *
	 * @var string
	 */
	public $transient_key = '';

	/**
	 * Class bbapp constructor.
	 */
	public function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Get the instance of the class.
	 *
	 * @param bool $root_file Root file.
	 *
	 * @return bbapp
	 */
	public static function instance( $root_file = false ) {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load( $root_file );
		}

		return self::$instance;
	}

	/**
	 * Function to load all essential components
	 *
	 * @param bool $root_file Root file.
	 */
	public function load( $root_file = false ) {
		$this->root_file      = $root_file;
		$this->plugin_version = '2.2.70';
		$this->plugin_dir     = plugin_dir_path( $this->root_file );
		$this->plugin_url     = plugin_dir_url( $this->root_file );
		$this->plugin_prefix  = 'buddyboss_app';
		$this->domain         = 'buddyboss-app'; // it's used for language domain.
		$this->is_network_activated();
		define( 'BUDDYBOSS_APP_PLUGIN_URL', $this->plugin_url );
		$this->transient_key  = 'bbapp-';
		$this->plugin_aws_url = 'https://buddyboss-app-files.s3.amazonaws.com/';

		// Include Composer.
		$this->require_composer();

		// Load Plugin Updater.
		new \BuddyBossApp\Updater( 'https://update.buddyboss.com/plugin', plugin_basename( __FILE__ ), 1133 );

		// Include functions.
		require_once trailingslashit( plugin_dir_path( __FILE__ ) ) . 'include/functions.php';

		// Install Cron Jobs.
		\BuddyBossApp\CronJobs::instance();
		\BuddyBossApp\Jobs::instance();

		// Background process log.
		BgProcessLog::instance();

		// Register activation & deactivation hooks.
		register_activation_hook( $root_file, array( $this, 'on_plugin_activate' ) );
		register_deactivation_hook( $root_file, array( $this, 'on_plugin_deactivate' ) );

		// Initiate required classes.
		\BuddyBossApp\Auth\Auth::instance();

		add_action( 'plugins_loaded', array( $this, 'load_classes' ), 0 );
	}

	/**
	 * Plugin load.
	 */
	public function load_classes() {
		$is_min_requirement_match = true;

		// Check if BuddyBoss Platform plugin installed and meet min requirement.
		if ( defined( 'BP_PLATFORM_VERSION' ) && version_compare( BP_PLATFORM_VERSION, '1.5.7.3', '<' ) ) {
			$is_min_requirement_match = false;
		}

		// Check if BuddyBoss Platform plugin installed and meet min requirement.
		if ( empty( $is_min_requirement_match ) ) {
			add_action( 'admin_notices', array( $this, 'min_supported_plugin_admin_notices' ) );

			// Delete the redirect transient.
			delete_transient( '_bbapp_activation_redirect' );

			return;
		}

		// Database Updater.
		\BuddyBossApp\DBUpdate\Main::instance();

		// Upgrade db version value.
		update_option( 'bbapp_db_version', $this->db_version );

		// Register the hooks.
		$this->hooks();

		/**
		 * Load BuddyBossApp Components.
		 */
		$this->load_components();

		\BuddyBossApp\RestCDN::instance();
		BuddyBossAppApi::instance();
		ClientCommon::instance();

		// App.
		App::instance();

		// Admin Setup.
		SetupAdmin::instance();

		// Tools.
		\BuddyBossApp\Admin\Tools::instance();

		new ClientApiPluginSupport();

		// Multi Site Network.
		Network::instance();

		\BuddyBossApp\Bookmark::instance();

		// Integrations.
		BuddyBossApp\Integrations\Memberium\Main::instance();
		BuddyBossApp\Integrations\IMember360\Main::instance();
		BuddyBossApp\Integrations\S2Member\Main::instance();
		BuddyBossApp\Integrations\BbpPrivateGroups\Main::instance();
		\BuddyBossApp\BrowserAuth::instance();
		\BuddyBossApp\Tools\Logger\ApiLogger::instance();
		BuddyBossApp\Integrations\ForceLogin\Main::instance();
		BuddyBossApp\Integrations\AdvancedRecaptcha\Main::instance();
		BuddyBossApp\Integrations\BuddyBossPlatform\Main::instance();
		BuddyBossApp\Integrations\BuddyBossTheme\Main::instance();
		BuddyBossApp\Integrations\GamiPress\Main::instance();
		BuddyBossApp\Integrations\BadgeOs\Main::instance();
		BuddyBossApp\Integrations\Vimeo\Main::instance();
		BuddyBossApp\Integrations\IThemeSecurity\Main::instance();

		BuddyBossApp\Integrations\Learndash\Main::instance();
		BuddyBossApp\Integrations\LearndashBBPress\Main::instance();
		BuddyBossApp\Integrations\LearndashBoss\Main::instance();
		BuddyBossApp\Integrations\LearndashBuddyPress\Main::instance();
		BuddyBossApp\Integrations\MemberPress\Main::instance();
		BuddyBossApp\Integrations\MemberPressCourses\Main::instance();
		BuddyBossApp\Integrations\WcMembership\Main::instance();
		BuddyBossApp\Integrations\WishlistMember\Main::instance();
		BuddyBossApp\Integrations\RestrictContentPro\Main::instance();
		BuddyBossApp\Integrations\DiviTheme\Main::instance();
		BuddyBossApp\Integrations\VerifiedMemberForBuddypress\Main::instance();

		// reCaptcha plugin support.
		BuddyBossApp\Integrations\GoogleCaptcha\Main::instance();
		BuddyBossApp\Integrations\SimpleGooglereCaptcha\Main::instance();
		BuddyBossApp\Integrations\Wordfence\Main::instance();

		// TutorLMS
		BuddyBossApp\Integrations\TutorLMS\Main::instance();

		// WPML Integration
		BuddyBossApp\Integrations\WPML\Main::instance();

		PmProForumSupport::instance();
		PmProMembershipSupport::instance();

		// AppStores API's.
		AppStores::instance();

		// DeepLinking.
		DeepLinking::instance();

		// Build.
		Build::instance();

		// Smart Banner.
		SmartBanner::instance();

		// IconPicker.
		IconPicker::instance();

		// ItemMeta
		// -----------------------------------------------------------------------------
		MetaApi::instance();
		CourseMeta::instance();
		LessonMeta::instance();
		TopicMeta::instance();

		if ( true === BBAPP_LOAD_MIGRATION_CODE ) {
			// ToDo: Temp Migration file. we will remove once we final launch buddyboss-app.
			\BuddyBossApp\Migration\Migrate::instance();
		}

		if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {
			// Load Telemetry.
			Telemetry::instance();
		} else {
			// Clear the cron if scheduled.
			if ( wp_next_scheduled( 'bbapp_telemetry_report_cron_event' ) ) {
				wp_clear_scheduled_hook( 'bbapp_telemetry_report_cron_event' );
			}
		}

		add_action( 'admin_notices', array( $this, 'app_disable_admin_notices' ) );
	}

	/**
	 * Include composer auto loader.
	 */
	public function require_composer() {
		require trailingslashit( $this->plugin_dir ) . 'vendor/autoload.php';
	}

	/**
	 * Check if the plugin is activated network wide(in multisite)
	 *
	 * @return boolean
	 */
	public function is_network_activated() {
		if ( ! $this->network_activated ) {
			$this->network_activated = false;

			if ( is_multisite() ) {
				if ( ! function_exists( 'is_plugin_active_for_network' ) ) {
					require_once ABSPATH . '/wp-admin/includes/plugin.php';
				}

				if ( is_plugin_active_for_network( 'buddyboss-app/buddyboss-app.php' ) ) {
					$this->network_activated = true;
				}
			}
		}

		return $this->network_activated;
	}

	/**
	 * Register all hooks.
	 *
	 * @since 1.0,0
	 */
	public function hooks() {
		add_action( 'admin_menu', 'register_bbapp_menu_page' );

		/**
		 * Make Menu Available on WP Network if Plugin enabled in Network Mode.
		 */
		if ( bbapp()->is_network_activated() ) {
			add_action( 'network_admin_menu', 'register_bbapp_menu_page' );
		}

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'block_editor_assets' ), 9999 );
		add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
		add_action( 'wp_footer', array( $this, 'wp_footer' ), 1 );
		add_filter( 'safe_style_css', array( $this, 'safe_style_css' ) );
		add_action( 'upgrader_process_complete', array( $this, 'bbapp_plugin_upgrade_function_callback' ), 10, 2 );
		add_action( 'init', array( $this, 'sync_with_app_and_platform_settings' ), 10 );
	}

	/**
	 * To add additional element in footer so app can get(post) title without any modification by third-party plugin
	 */
	public function wp_footer() {
		global $post;

		if ( ! empty( $post->post_title ) ) {
			printf( '<span id="bbapp-title" style="display: none;">%1$s</span>', wp_kses_post( $post->post_title ) );
		}
	}

	/**
	 * Function will run after plugin successfully update.
	 *
	 * @since 1.6.3
	 *
	 * @param object $upgrader_object WP_Upgrader instance.
	 * @param array  $options         Array of bulk item update data.
	 */
	public function bbapp_plugin_upgrade_function_callback( $upgrader_object, $options ) {
		$show_display_popup = false;
		// The path to our plugin's main file.
		$our_plugin = 'buddyboss-app/buddyboss-app.php';

		if ( ! empty( $options ) && 'update' === $options['action'] && 'plugin' === $options['type'] && isset( $options['plugins'] ) ) {
			foreach ( $options['plugins'] as $plugin ) {
				if ( ! empty( $plugin ) && $plugin === $our_plugin ) {
					update_option( '_bbapp_is_update', $show_display_popup );
					flush_rewrite_rules(); // Flush rewrite rules when update the Buddyboss platform plugin.
				}
			}
		}
	}

	/**
	 * Sync with app and platform settings.
	 *
	 * @since 1.7.90
	 */
	public function sync_with_app_and_platform_settings() {

		$sync_notification_fields = array(
			'notification_app_push'          => 'manual_push_notification',
			'bb_posts_new_comment_reply_app' => 'bbapp_new_comment_reply',
		);

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {

			// Platform to app setting sync.
			add_action(
				'updated_user_meta',
				function ( $meta_id, $object_id, $meta_key, $_meta_value ) use ( $sync_notification_fields ) {

					if ( ! isset( $sync_notification_fields[ $meta_key ] ) ) {
						return;
					}

					$app_push_settings = get_user_meta( $object_id, 'push_notification_settings', true );

					if ( isset( $app_push_settings[ $sync_notification_fields[ $meta_key ] ] ) ) {
						$app_push_settings[ $sync_notification_fields[ $meta_key ] ] = 'yes' === $_meta_value ? 1 : 0;
					}

					update_user_meta( $object_id, 'push_notification_settings', $app_push_settings );

				},
				10,
				4
			);

		} else {
			// App to platform setting sync.
			add_action(
				'updated_user_meta',
				function ( $meta_id, $object_id, $meta_key, $_meta_value ) use ( $sync_notification_fields ) {

					if ( 'push_notification_settings' !== $meta_key ) {
						return;
					}

					foreach ( $sync_notification_fields as $key => $notification_field ) {

						$platform_field_setting = get_user_meta( $object_id, $key, true );

						if ( isset( $platform_field_setting ) && isset( $_meta_value[ $notification_field ] ) ) {

							$platform_field_setting = 1 === (int) $_meta_value[ $notification_field ] ? 'yes' : 'no';
							update_user_meta( $object_id, $key, $platform_field_setting );
						}
					}

				},
				10,
				4
			);
		}
	}

	/**
	 * Returns defaults app setting values.
	 * These defaults get added as default when key's are not exists on db.
	 * These settings get apply to individual app settings,
	 *
	 * @return array
	 */
	public function app_settings_defaults() {
		$defaults = array(
			'app_auth.enable_signup'       => true,
			'app_auth.signup_form'         => \BuddyBossApp\Auth\Auth::instance()->is_app_registration_form(),
			// Pass default registration form.
			'app_subscription.enabled'     => false,
			'learndash_course_downloading' => 0,
			'learndash_author_visible'     => 1,
			'learndash_date_visible'       => 1,
			'app_telemetry'                => 'anonymous',
		);

		return $defaults;
	}

	/**
	 * Defaults for Auth Settings
	 * Auth Settings are global over all apps..
	 *
	 * @return array
	 */
	public function bbapp_default_settings() {
		$defaults = array(
			'app_auth.email_activation_body' => __( 'From a mobile device, enter the following code <b>( {{KEY_CODE}} )</b> in the app, or tap this link: <a href="{{KEY_CODE_LINK}}">Click here</a> to be sent to the app.', 'buddyboss-app' ),
		);

		return $defaults;
	}

	/**
	 * Load Components.
	 */
	public function load_components() {
		\BuddyBossApp\Components\AppPages\AppPages::instance();
		\BuddyBossApp\Components\PushNotification\PushNotification::instance();
		\BuddyBossApp\Components\InAppPurchases\InAppPurchases::instance();
		\BuddyBossApp\Components\AccessControls\AccessControls::instance();
		\BuddyBossApp\Components\Performance\Performance::instance();

		/**
		 * All Components load the code on this hook at 99.
		 * If you want to load any component before that please use priority less then 99.
		 */
		do_action( 'bbapp_components_loaded' );
	}

	/**
	 * Register admin assets.
	 *
	 * @since 1.0.0
	 */
	public function admin_enqueue_assets() {
		global $post_type, $pagenow;

		$suffix       = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		$get_page_var = ( isset( $_GET['page'] ) ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$is_bbapp     = false !== strpos( $get_page_var, 'bbapp' );

		// All BuddyBossApp Settings and Other Screens. (App Editor).
		if ( $is_bbapp || // All bbApp Settings and Other Screens.
			in_array(
				$pagenow,
				array(
					'post-new.php',
					'edit.php',
					'post.php',
					'edit.php',
				),
				true
			)  // Allow on all CPT edit pages. (App Editor).
		) {
			// load select2 library version 4.0.5.
			wp_enqueue_script( 'bbapp_select2', $this->plugin_url . 'assets/libs/select2/select2.full' . $suffix . '.js', array( 'jquery' ), '4.0.5', true );
			wp_enqueue_style( 'bbapp_select2', $this->plugin_url . 'assets/libs/select2/select2' . $suffix . '.css', array(), '4.0.5' );

			// load croppie library version 2.6.4.
			wp_register_script( 'bbapp_croppie', $this->plugin_url . 'assets/libs/croppie/croppie' . $suffix . '.js', array( 'jquery' ), '2.6.4', true );
			wp_register_style( 'bbapp_croppie', $this->plugin_url . 'assets/libs/croppie/croppie' . $suffix . '.css', array(), '2.6.4' );

			// load listjs library version 2.3.1.
			wp_register_script( 'bbapp-list', $this->plugin_url . 'assets/libs/list' . $suffix . '.js', array( 'jquery' ), '2.3.1', true );

			// load fancybox library.
			wp_enqueue_script( 'jquery-fancybox', bbapp()->plugin_url . 'assets/libs/fancybox/jquery.fancybox' . $suffix . '.js', array( 'jquery' ), '3.0.47', true );
			wp_enqueue_style( 'jquery-fancybox', bbapp()->plugin_url . 'assets/libs/fancybox/jquery.fancybox' . $suffix . '.css', array(), '3.0.47' );

			// Main js file @ admin backend.
			wp_enqueue_script(
				'bbapp-script',
				$this->plugin_url . 'assets/js/admin.js',
				array(
					'jquery',
					'jquery-ui-dialog',
					'wp-i18n',
					'wp-util',
					'jquery-ui-autocomplete',
					'clipboard'
				),
				$this->plugin_version,
				true
			);

			/**
			 * Wp Color Picker.
			 */
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_script( 'wp-color-picker' );
			wp_register_script( 'wp-color-picker-alpha', $this->plugin_url . '/assets/libs/wp-color-picker-alpha' . $suffix . '.js', array( 'wp-color-picker', 'iris' ), '3.0.2', true );
			wp_enqueue_script( 'wp-color-picker-alpha' );

			// Register Tipsy version 1.0.3.
			wp_register_script( 'bbapp-tipsy', 'https://cdnjs.cloudflare.com/ajax/libs/jquery.tipsy/1.0.3/jquery.tipsy.js', array( 'jquery' ), '1.0.3', false );
			wp_register_style( 'bbapp-tipsy', 'https://cdnjs.cloudflare.com/ajax/libs/jquery.tipsy/1.0.3/jquery.tipsy.min.css', array(), '1.0.3' );

			// Register Bootstrap Date Picker version 1.0.3.
			wp_register_script( 'bbapp-bootstrap-datepicker', 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js', array( 'jquery' ), '1.9.0', false );
			wp_register_style( 'bbapp-bootstrap-datepicker', 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker3.standalone.min.css', array(), '1.9.0' );

			$icons_path    = IconPicker::instance()->icon_picker_app_icon_url();
			$bb_icons_path = IconPicker::instance()->icon_picker_bb_icon_url();

			$is_app_page = false;
			if ( 'app_page' === $post_type ) {
				$is_app_page = true;
			}

			// Help URL.
			$bbapp_help_base_url = bbapp_get_super_admin_url( 'admin.php?page=bbapp-help' );

			// Localize.
			wp_localize_script(
				'bbapp-script',
				'bbappAjax',
				array(
					'ajaxurl'               => admin_url( 'admin-ajax.php' ),
					'siteurl'               => site_url(),
					'resturl'               => get_rest_url(),
					'rest_nonce'            => wp_create_nonce( 'wp_rest' ),
					'sort_nonce'            => wp_create_nonce( 'bbapp_sort_nonce_' . get_current_user_id() ),
					'icon_path'             => $icons_path,
					'bb_icon_path'          => $bb_icons_path,
					'is_app_page'           => $is_app_page,
					'bbapp_help_url'        => $bbapp_help_base_url,
					'bbapp_help_title'      => __( 'Docs', 'buddyboss-app' ),
					'bbapp_help_no_network' => __( '<strong>You are offline.</strong> Documentation requires internet access.', 'buddyboss-app' ),
				)
			);

			wp_localize_script( 'bbapp-script', 'bbappListMultilanguages', bbapp_get_multi_languages_list() );

			if ( realpath( bbapp()->plugin_dir . '/include/Admin/JSTranslationsStrings.php' ) ) {
				wp_localize_script( 'bbapp-script', 'bbapp_translations_string', include_once bbapp()->plugin_dir . '/include/Admin/JSTranslationsStrings.php' );
			}
		}

		// Main js file @ admin backend.
		wp_enqueue_script(
			'bbapp-about-script',
			$this->plugin_url . 'assets/js/about.js',
			array(
				'jquery',
				'jquery-ui-dialog',
				'wp-i18n',
				'wp-util',
				'jquery-ui-autocomplete',
			),
			$this->plugin_version,
			true
		);
		wp_localize_script(
			'bbapp-about-script',
			'BBAPP_ABOUT',
			array(
				'bbapp_display_auto_popup' => get_option( '_bbapp_is_update' ),
			)
		);

		$rtl_css = is_rtl() ? '.rtl' : '';

		// Main css file @ admin backend.
		wp_enqueue_style( 'bbapp_style', $this->plugin_url . 'assets/css/admin' . $rtl_css . '.css', array( 'wp-jquery-ui-dialog' ), $this->plugin_version );

		// BB Icons css file @ admin backend.
		// BB icon version.
		$bb_icon_version = IconPicker::instance()->bb_icon_font_map( 'version' );
		$bb_icon_version = ! empty( $bb_icon_version ) ? $bb_icon_version : $this->plugin_version;
		wp_enqueue_style( 'bbapp_icons_style', $this->plugin_url . 'assets/icons/bb-icons/css/bb-icons' . $suffix . '.css', array(), $bb_icon_version );
	}

	/**
	 * Gutenberg Editor Block assets.
	 */
	public function block_editor_assets() {
		global $post;

		if ( is_admin() ) {
			$post_type = ! empty( $post->post_type ) ? $post->post_type : '';

			if ( 'app_page' === $post_type ) {

				if ( wp_style_is( 'et-builder-gutenberg' ) ) {
					wp_dequeue_style( 'et-builder-gutenberg' );
				}

				if ( wp_script_is( 'et-builder-gutenberg' ) ) {
					wp_dequeue_script( 'et-builder-gutenberg' );
				}
			}

			wp_enqueue_script(
				$this->plugin_prefix . 'gutenberg',
				$this->plugin_url . 'assets/js/gutenberg.js',
				array(
					'wp-blocks',
					'wp-editor',
					'wp-i18n',
					'wp-element',
					'wp-components',
					'wp-api',
					'wp-data',
					'wp-edit-post',
					'lodash',
				),
				$this->plugin_version,
				true
			);

			$styling              = Styling::instance()->get_styling_colors();
			$header_text_color    = ! empty( $styling['colors'] ) && ! empty( $styling['colors']['headingsColor'] ) ? $styling['colors']['headingsColor'] : '#000000';
			$is_learndash_enabled = bbapp_is_learndash_enabled();
			wp_localize_script(
				$this->plugin_prefix . 'gutenberg',
				'bbappGutenberg',
				array(
					'is_bbpress_enabled'                => function_exists( 'bbpress' ),
					'is_buddypress_enabled'             => class_exists( 'BuddyPress' ),
					'is_learndash_enabled'              => $is_learndash_enabled,
					'is_tutor_lms_enabled'              => ( function_exists( 'bbapp_is_tutor_lms_plugins_active' ) && bbapp_is_tutor_lms_plugins_active() ),
					'is_activity_component_enabled'     => class_exists( 'BuddyPress' ) && bp_is_active( 'activity' ),
					'is_notification_component_enabled' => class_exists( 'BuddyPress' ) && bp_is_active( 'notifications' ) || bbapp_is_active( 'push_notification' ),
					'is_groups_component_enabled'       => class_exists( 'BuddyPress' ) && bp_is_active( 'groups' ),
					'is_members_component_enabled'      => class_exists( 'BuddyPress' ) && bp_is_active( 'members' ),
					'is_h5p_enabled'                    => class_exists( 'H5P_Plugin' ),
					'header_text_color'                 => $header_text_color,
					'is_ld_cat_enabled'                 => ( $is_learndash_enabled && method_exists( 'LearnDash_Settings_Section', 'get_section_setting' ) && 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_category' ) ),
					'is_ld_tag_enabled'                 => ( $is_learndash_enabled && method_exists( 'LearnDash_Settings_Section', 'get_section_setting' ) && 'yes' === LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Courses_Taxonomies', 'ld_course_tag' ) ),
					'is_mp_lms_enabled'                 => bbapp_is_memberpress_lms_enabled(),
					'is_lite_license'                   => bbapp_is_lite_app(),
				)
			);

			if ( function_exists( 'wp_set_script_translations' ) ) {
				wp_set_script_translations( $this->plugin_prefix . 'action', 'buddyboss-app' );
			}
		}
	}

	/**
	 * Register admin assets.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_scripts() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_enqueue_style( $this->plugin_prefix . 'style', $this->plugin_url . 'assets/css/main.css', array(), $this->plugin_version );

		// Main js file @ frontend backend.
		wp_enqueue_script( $this->plugin_prefix . 'action', $this->plugin_url . 'assets/js/main.js', array( 'jquery' ), $this->plugin_version, true );

		// Smart banner.
		wp_register_script( bbapp()->plugin_prefix . '_jquery.detectIncognito', bbapp()->plugin_url . 'assets/libs/detectIncognito' . $suffix . '.js', array( 'jquery' ), '1.2.0', true );
		wp_register_script( bbapp()->plugin_prefix . '_jquery.smartbanner', bbapp()->plugin_url . 'assets/libs/jquery.smartbanner.js', array( 'jquery' ), '1.0.0', true );
	}

	/**
	 * Loads text domain.
	 *
	 * @since 1.0.0
	 */
	public function load_textdomain() {
		/**
		 * Filters plugin locale.
		 *
		 * @param string $locale The current locale.
		 * @param string $domain Plugin domain.
		 */
		$locale = apply_filters( 'plugin_locale', get_locale(), $this->domain );

		// First try to load from wp-content/languages/plugins/ directory.
		load_textdomain( $this->domain, WP_LANG_DIR . '/plugins/' . $this->domain . '-' . $locale . '.mo' );

		// If not found, then load from plugin_folder/languages directory.
		load_plugin_textdomain( $this->domain, false, basename( $this->domain ) . '/languages' );
	}

	/**
	 * When plugin activates we do some required updates.
	 *
	 * @param bool $network_wide If the plugin is network active.
	 */
	public function on_plugin_activate( $network_wide ) {
		$this->on_database_update( $network_wide );

		/**
		 * Add About Redirect Trigger
		 */
		set_transient( '_bbapp_activation_redirect', true, 60 * 2 );

		/**
		 * End Update structure of app info data
		 * --------------------------------------------------------------------
		 */

		/**
		 * Fires after BuddyBoss App Plugin activate.
		 */
		do_action( "{$this->plugin_prefix}_on_plugin_activate" );

		/**
		 * General hook fires after plugin activate.
		 */
		do_action( 'bbapp_on_plugin_activate' );
	}

	/**
	 * Fires on database update.
	 *
	 * @param bool $network_wide If plugin is network active.
	 */
	public function on_database_update( $network_wide ) {
		// Only way to know if plugin is activated network wide.
		$this->network_activated = $network_wide;

		global $wpdb;
		if ( bbapp()->is_network_activated() ) {
			// Get all blogs in the network and activate plugin on each one.
			$blog_ids = $wpdb->get_col( "SELECT blog_id FROM $wpdb->blogs" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

			foreach ( $blog_ids as $blog_id ) {
				switch_to_blog( $blog_id );

				// Create Gutenberg Default Page for BuddyBossApp Editor.
				bbapp_get_default_app_page();

				// Network Wide Tables.
				if ( 1 === (int) $blog_id ) {
					$this->create_network_table();
				}

				// Create default App Pages.
				NativeAppPage::instance()->create_default_apppage();

				restore_current_blog();
			}
		} else {
			$this->create_network_table();

			// Create Gutenberg Default Page.
			bbapp_get_default_app_page();

			// Create default App Pages.
			NativeAppPage::instance()->create_default_apppage();
		}

		// Install/Upgrade Defaults Settings.
		ClientCommon::instance()->merge_default_app_settings();

		// Install other Default Settings.
		ClientCommon::instance()->merge_default_auth_settings();
	}

	/**
	 * Fires on plugin deactivation.
	 */
	public function on_plugin_deactivate() {
		do_action( "{$this->plugin_prefix}_on_plugin_deactivate" );
		do_action( 'bbapp_on_plugin_deactivate' );
	}

	/**
	 * Create Table function
	 */
	private function create_network_table() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$sql = "CREATE TABLE {$wpdb->prefix}bbapp_user_devices (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            user_id bigint(20) NOT NULL,
            device_token varchar(500) NOT NULL,
            device_token_hash varchar(70) NOT NULL,
            auth_token_hash varchar(70) NOT NULL,
            platform varchar(10) NOT NULL,
            app_ver varchar(12) NOT NULL,
            push_type varchar(20) NOT NULL,
            device_id varchar(100) NOT NULL,
            device_model varchar(100) NOT NULL,
            data longtext DEFAULT NULL,
            date_registered datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_updated datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_active datetime DEFAULT '0000-00-00 00:00:00' NULL,
            uuid varchar(36) NULL,
            KEY device_token_hash (device_token_hash),
            KEY auth_token_hash (auth_token_hash),
            KEY user_id (user_id),
            KEY date_registered (date_registered),
            KEY date_updated (date_updated),
            KEY date_active (date_active),
            KEY app_ver (app_ver),
            KEY push_type (push_type),
            UNIQUE KEY uuid (uuid)
        ) {$charset_collate}";

		dbDelta( $sql );

		// Upgrade Table Data.
		bbapp_table_fix_wp_bbapp_user_devices();

		// Store all notification for backend purpose.
		$sql2 = "CREATE TABLE {$wpdb->prefix}bbapp_push_notifications (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            action varchar(30) NOT NULL,
            primary_text varchar(500) NOT NULL,
            secondary_text varchar(500) DEFAULT NULL,
            target_type varchar(30) DEFAULT NULL,
            target bigint(20) DEFAULT NULL,
            sent_as bigint(20) DEFAULT 0,
            data longtext DEFAULT NULL,
            agent varchar(100) DEFAULT 'event' NULL,
            status varchar(100) DEFAULT 'sent' NULL,
            is_schedule tinyint(1) DEFAULT '0' NULL,
            date_schedule datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_expire datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_updated datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_created datetime DEFAULT '0000-00-00 00:00:00' NULL,
            blog_id bigint(20) NOT NULL,
            KEY target (target),
            KEY date_created (date_created),
            KEY date_updated (date_updated),
            KEY is_schedule (is_schedule),
            KEY date_expire (date_expire),
            KEY sent_as (sent_as),
            KEY agent (agent)
        ) {$charset_collate}";

		dbDelta( $sql2 );

		// Stores all notification for notification endpoint purpose.
		$sql3 = "CREATE TABLE {$wpdb->prefix}bbapp_notifications (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            date_notified datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            action varchar(20) DEFAULT '' NOT NULL,
            namespace varchar(20) DEFAULT '' NOT NULL,
            primary_text varchar(500) DEFAULT '' NOT NULL,
            secondary_text varchar(500) DEFAULT '' DEFAULT NULL,
            component_name varchar(75) DEFAULT '' NOT NULL,
            component_action varchar(75) DEFAULT '' NOT NULL,
            user_id bigint(20) DEFAULT 0 NOT NULL,
            author_id bigint(20) DEFAULT 0 NOT NULL,
            item_id bigint(20) DEFAULT 0 NOT NULL,
            secondary_item_id bigint(20) DEFAULT 0 NOT NULL,
            blog_id bigint(20) DEFAULT 0 NOT NULL,
            unread tinyint(0) DEFAULT 1 NOT NULL,
            version tinyint(0) DEFAULT 1 NOT NULL,
            KEY action (action),
            KEY user_id (user_id),
            KEY item_id (item_id),
            KEY secondary_item_id (secondary_item_id),
            KEY blog_id (blog_id),
            KEY unread (unread),
            KEY component_name (component_name),
            KEY component_action (component_action),
            KEY version (version)
        ) $charset_collate;";

		if ( $wpdb->query( "SELECT * FROM information_schema.tables WHERE table_schema = '{$wpdb->dbname}' AND table_name = '{$wpdb->prefix}bbapp_notifications' LIMIT 1;" ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			// Set Default to Deprecated Columns.
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN primary_text SET DEFAULT ''" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN secondary_text SET DEFAULT ''" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN namespace SET DEFAULT ''" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN action SET DEFAULT ''" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN author_id SET DEFAULT 0" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$wpdb->query( "ALTER TABLE {$wpdb->prefix}bbapp_notifications ALTER COLUMN blog_id SET DEFAULT 0" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
		}

		dbDelta( $sql3 );

		// Stores the queue.
		$sql4 = "CREATE TABLE {$wpdb->prefix}bbapp_queue (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            created datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            modified datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            type varchar(20) NOT NULL,
            data text DEFAULT NULL,
            priority tinyint(2) NOT NULL,
            blog_id bigint(20) NOT NULL,
            KEY created (created),
            KEY modified (modified),
            KEY type (type),
            KEY priority (priority),
            KEY blog_id (blog_id)
        ) $charset_collate;";

		dbDelta( $sql4 );

		// Stores the push queue.
		$sql5 = "CREATE TABLE {$wpdb->prefix}bbapp_push_queue (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            n_id bigint(20) NOT NULL,
            created datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            device text DEFAULT NULL,
            unique_hash varchar(100) NOT NULL,
            user_id bigint(20) NOT NULL,
            type varchar(50) NOT NULL,
            agent varchar(50) NOT NULL,
            data text DEFAULT NULL,
            priority tinyint(2) NOT NULL,
            sent tinyint(1) NOT NULL,
            KEY created (created),
            KEY unique_hash (unique_hash),
            KEY user_id (user_id),
            KEY type (type),
            KEY agent (agent),
            KEY priority (priority),
            KEY sent (sent)
        ) $charset_collate;";

		dbDelta( $sql5 );

		// Stores the Logs.
		$sql6 = "CREATE TABLE {$wpdb->prefix}bbapp_logs (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            ref_id bigint(20) NOT NULL,
            created datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            type varchar(20) NOT NULL,
            blog_id bigint(20) NOT NULL,
            text text DEFAULT NULL,
            KEY ref_id (ref_id),
            KEY blog_id (blog_id),
            KEY created (created),
            KEY type (type)
        ) $charset_collate;";

		dbDelta( $sql6 );

		// UserSegment.
		$sql7 = "CREATE TABLE {$wpdb->prefix}bbapp_user_segment (
			id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            user_id bigint(20) NOT NULL,
            segment_group varchar(30) NOT NULL,
            segment_id varchar(30) NOT NULL,
            created datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            expire datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            KEY user_id (user_id),
            KEY segment_group (segment_group),
            KEY created (created),
            KEY expire (expire),
            KEY segment_id (segment_id)
        ) $charset_collate;";

		dbDelta( $sql7 );

		// Push notification history.
		$sql9 = "CREATE TABLE {$wpdb->prefix}bbapp_push_notifications_history (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            type varchar(100) DEFAULT 'custom',
            status int(1) DEFAULT 0,
            data longtext DEFAULT NULL,
            user_id bigint(20) DEFAULT NULL,
            processed_at datetime DEFAULT '0000-00-00 00:00:00',
            item_id bigint(20) DEFAULT 0,
            secondary_id bigint(20) DEFAULT 0,
            device varchar(20) DEFAULT NULL,
            platform varchar(10) DEFAULT NULL,
            blog_id bigint(20) NOT NULL,
            KEY type (type),
            KEY status (status),
            KEY user_id (user_id),
            KEY processed_at (processed_at),
            KEY item_id (item_id)
        ) {$charset_collate};";
		dbDelta( $sql9 );

		// Create Auth Related Tables.
		\BuddyBossApp\Auth\Auth::instance()->on_activation();

		// Create IAP Related Tables.
		\BuddyBossApp\InAppPurchases\Controller::instance()->on_activation();
		\BuddyBossApp\InAppPurchases\Orders::instance()->on_activation();

		// Create Meta API Related Tables.
		MetaApi::instance()->on_activation();

		// Create Access Controls related tables.
		\BuddyBossApp\AccessControls::instance()->on_activation();
		\BuddyBossApp\Bookmark::instance()->on_activation();

		// Load Performance Class and call on_activation.
		if ( ! class_exists( 'BuddyBoss\Performance\Performance' ) ) {
			// The class doesn't exist, so we need to load it and then call on_activation.
			require_once trailingslashit( bbapp()->plugin_dir ) . 'include/Performance/class-performance.php';
		}
		\BuddyBoss\Performance\Performance::instance()->on_activation();

		BgProcessLog::instance()->create_table();

		// Create default menus.
		BuddyBossApp\Menu::instance()->on_activation();

		// Create App Languages related tables.
		BuddyBossApp\AppLanguages::instance()->on_activation();
	}

	/**
	 * Add admin notice if non supported activated
	 */
	public function min_supported_plugin_admin_notices() {
		if ( defined( 'BP_PLATFORM_VERSION' ) && version_compare( BP_PLATFORM_VERSION, '1.5.7.3', '<' ) ) {
			/* translators: 1. div classes. 2. notice message. 3. Admin update page url. 4. link text.  */
			printf( '<div class="%1$s"><p>%2$s <a href="%3$s">%4$s</a></p></div>', esc_attr( 'notice notice-error' ), esc_html__( 'The BuddyBoss App plugin requires a minimum BuddyBoss Platform version 1.5.7.3.', 'buddyboss-app' ), esc_url( admin_url( 'update-core.php' ) ), esc_html__( 'Please update BuddyBoss Platform.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Add admin notice if app center in app disabled.
	 */
	public function app_disable_admin_notices() {
		if ( ! \BuddyBossApp\ManageApp::instance()->is_app_disabled() ) {
			return;
		}
		echo "<div class='notice notice-error is-dismissible'>";
		include __DIR__ . '/views/settings/app-is-disabled.php';
		echo '</div>';
	}

	/**
	 * Add safe style.
	 *
	 * @param array $styles safe style list.
	 *
	 * @since 1.6.8
	 *
	 * @return mixed
	 */
	public function safe_style_css( $styles ) {
		$styles[] = 'display';

		return $styles;
	}
}

/**
 * Easy to call function.
 *
 * @return bbapp
 */
function bbapp() {
	return bbapp::instance();
}

/**
 * Easy to call function alias for bbapp() func.
 *
 * @return bbapp
 */
function bbapp_client() {
	return bbapp();
}


/**
 * Register BuddyBossApp Menu.
 */
function register_bbapp_menu_page() {

	// Set position with odd number to avoid conflict with other plugin/theme.
	add_menu_page(
		'BuddyBoss App',
		__( 'BuddyBoss App', 'buddyboss-app' ),
		'manage_options',
		'bbapp-settings',
		'',
		'',
		4
	);

	// To remove empty parent menu item.
	add_submenu_page(
		'bbapp-settings',
		__( 'BuddyBoss App', 'buddyboss-app' ),
		__( 'BuddyBoss App', 'buddyboss-app' ),
		'manage_options',
		'bbapp-settings'
	);

	remove_submenu_page( 'bbapp-settings', 'bbapp-settings' );

}
