<?php
/*
Plugin Name: BuddyBoss Complete Analytics
Description: Comprehensive Analytics Dashboard with All Requested Metrics
Version: 3.1
Author: mobinteg
*/

if (!defined('ABSPATH')) {
    exit;
}

class BuddyBoss_Complete_Analytics {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
    }

    public function add_admin_menu() {
        add_menu_page(
            'Complete Analytics',
            'Complete Analytics',
            'manage_options',
            'buddyboss-analytics',
            array($this, 'display_full_dashboard'),
            'dashicons-chart-bar',
            6
        );
    }

    public function enqueue_assets($hook) {
        if ($hook === 'toplevel_page_buddyboss-analytics') {
            // CSS
            wp_enqueue_style('bb-analytics-css', plugins_url('assets/css/analytics.css', __FILE__));
            
            // JS
            wp_enqueue_script('chart-js', 'https://cdn.jsdelivr.net/npm/chart.js');
            wp_enqueue_script('bb-analytics-js', plugins_url('assets/js/analytics.js', __FILE__), array('jquery', 'chart-js'), null, true);
            
            // Localize data for JS
            $members_data = $this->get_members_growth_data();
            wp_localize_script('bb-analytics-js', 'bbAnalyticsData', array(
                'membersGrowth' => $members_data,
                'ajaxurl' => admin_url('admin-ajax.php')
            ));
        }
    }

    public function display_full_dashboard() {
        echo '<div class="wrap bb-analytics-wrap">';
        echo '<h1>BuddyBoss Community Analytics</h1>';
        
        // Dashboard summary row
        echo '<div class="bb-dashboard-summary">';
        $this->display_summary_metrics();
        echo '</div>';
        
        // Main content tabs
        echo '<div class="bb-analytics-tabs">';
        echo '<nav class="nav-tab-wrapper">';
        echo '<a href="#members" class="nav-tab nav-tab-active">Members</a>';
        echo '<a href="#activity" class="nav-tab">Activity</a>';
        echo '<a href="#groups" class="nav-tab">Groups</a>';
        echo '<a href="#network" class="nav-tab">Network</a>';
        echo '<a href="#events" class="nav-tab">Events</a>';
        echo '<a href="#discussions" class="nav-tab">Discussions</a>';
        echo '<a href="#searches" class="nav-tab">Searches</a>';
        echo '<a href="#messages" class="nav-tab">Messages</a>';

        echo '</nav>';
        
        // Tab content
        echo '<div id="members" class="bb-tab-content active">';
        $this->display_enhanced_members_section();
        $this->display_popular_members();
        echo '</div>';
        
        echo '<div id="activity" class="bb-tab-content">';
        $this->display_activity_section();
        echo '</div>';
        
        echo '<div id="groups" class="bb-tab-content">';
        $this->display_groups_section();
        echo '</div>';
        
        echo '<div id="network" class="bb-tab-content">';
        $this->display_network_section();
        echo '</div>';

        echo '<div id="events" class="bb-tab-content">';
        $this->display_events_section();
        echo '</div>';

        echo '<div id="discussions" class="bb-tab-content">';
        $this->display_discussions_section();
        echo '</div>';

        echo '<div id="searches" class="bb-tab-content">';
        $this->display_searches_section();
        echo '</div>';

        echo '<div id="messages" class="bb-tab-content">';
        $this->display_messages_section();
        echo '</div>';
        
        echo '</div>'; // Close tabs
        
        echo '</div>'; // Close wrap
    }

    private function display_summary_metrics() {
        global $wpdb;
        
        $total_members = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->users}");
        $active_members = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} 
             WHERE meta_key = 'last_activity' AND meta_value >= %s",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        $total_groups = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_groups");
        $total_activity = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity");
        
        echo '<div class="bb-summary-container">';
        
        echo '<div class="bb-summary-card">';
        echo '<div class="bb-summary-value">'.number_format($total_members).'</div>';
        echo '<div class="bb-summary-label">Total Members</div>';
        echo '</div>';
        
        echo '<div class="bb-summary-card">';
        echo '<div class="bb-summary-value">'.number_format($active_members).'</div>';
        echo '<div class="bb-summary-label">Active Members (3mo)</div>';
        echo '</div>';
        
        echo '<div class="bb-summary-card">';
        echo '<div class="bb-summary-value">'.number_format($total_groups).'</div>';
        echo '<div class="bb-summary-label">Total Groups</div>';
        echo '</div>';
        
        echo '<div class="bb-summary-card">';
        echo '<div class="bb-summary-value">'.number_format($total_activity).'</div>';
        echo '<div class="bb-summary-label">Activities</div>';
        echo '</div>';
        
        echo '</div>';
    }

    private function get_members_growth_data() {
        global $wpdb;
        
        $data = $wpdb->get_results("
            SELECT 
                DATE_FORMAT(user_registered, '%Y-%m') as month, 
                COUNT(*) as count,
                (SELECT COUNT(*) FROM {$wpdb->users} WHERE DATE_FORMAT(user_registered, '%Y-%m') <= month) as running_total
            FROM {$wpdb->users}
            GROUP BY month
            ORDER BY month
        ");
        
        return $data;
    }

    private function display_enhanced_members_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-groups"></i> Member Analytics</h2>';
        
        // 1. Growth Chart
        echo '<div class="bb-subsection">';
        echo '<h3>Member Growth Over Time</h3>';
        echo '<div class="bb-chart-container"><canvas id="membersGrowthChart"></canvas></div>';
        echo '</div>';
        
        // 2. Current Stats
        echo '<div class="bb-subsection">';
        echo '<h3>Current Member Statistics</h3>';
        
        $total_members = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->users}");
        $active_members = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} 
             WHERE meta_key = 'last_activity' AND meta_value >= %s",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        
        echo '<div class="bb-stats-grid">';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">'.number_format($total_members).'</div>';
        echo '<div class="bb-stat-label">Total Members</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">'.number_format($active_members).'</div>';
        echo '<div class="bb-stat-label">Active Members (3mo)</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">'.round(($active_members/$total_members)*100, 1).'%</div>';
        echo '<div class="bb-stat-label">Engagement Rate</div>';
        echo '</div>';
        
        echo '</div>'; // Close grid
        echo '</div>'; // Close subsection
        
        // 3. Member Types
        echo '<div class="bb-subsection">';
        echo '<h3>Member Types</h3>';
        
        $member_types = $wpdb->get_results("
            SELECT 
                IFNULL(meta_value, 'Not Specified') as member_type, 
                COUNT(*) as count
            FROM {$wpdb->usermeta}
            WHERE meta_key = 'member_type' OR meta_key IS NULL
            GROUP BY meta_value
            ORDER BY count DESC
        ");
        
        echo '<div class="bb-table-responsive">';
        echo '<table class="bb-analytics-table">';
        echo '<thead><tr><th>Type</th><th>Count</th><th>% of Total</th></tr></thead>';
        echo '<tbody>';
        foreach ($member_types as $type) {
            $percentage = round(($type->count / $total_members) * 100, 2);
            echo '<tr>';
            echo '<td>'.$type->member_type.'</td>';
            echo '<td>'.number_format($type->count).'</td>';
            echo '<td>'.$percentage.'%</td>';
            echo '</tr>';
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        echo '</div>';
        
        // 4. Active Members by Type
        echo '<div class="bb-subsection">';
        echo '<h3>Active Members by Type (Last 3 Months)</h3>';
        
        $active_by_type = $wpdb->get_results($wpdb->prepare(
            "SELECT 
                IFNULL(um2.meta_value, 'Not Specified') as member_type, 
                COUNT(DISTINCT um.user_id) as count
             FROM {$wpdb->usermeta} um
             LEFT JOIN {$wpdb->usermeta} um2 ON um.user_id = um2.user_id AND um2.meta_key = 'member_type'
             WHERE um.meta_key = 'last_activity'
             AND um.meta_value >= %s
             GROUP BY um2.meta_value
             ORDER BY count DESC",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        
        echo '<div class="bb-table-responsive">';
        echo '<table class="bb-analytics-table">';
        echo '<thead><tr><th>Type</th><th>Active Count</th><th>% of Active</th><th>Engagement Rate</th></tr></thead>';
        echo '<tbody>';
        foreach ($active_by_type as $type) {
            $percent_of_active = round(($type->count / $active_members) * 100, 2);
            
            $total_of_type = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->usermeta} 
                 WHERE (meta_key = 'member_type' AND meta_value = %s) OR (%s = 'Not Specified' AND (meta_key IS NULL OR meta_value IS NULL))",
                $type->member_type, $type->member_type
            ));
            
            $engagement_rate = $total_of_type > 0 ? round(($type->count / $total_of_type) * 100, 2) : 0;
            
            echo '<tr>';
            echo '<td>'.$type->member_type.'</td>';
            echo '<td>'.number_format($type->count).'</td>';
            echo '<td>'.$percent_of_active.'%</td>';
            echo '<td>'.$engagement_rate.'%</td>';
            echo '</tr>';
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        echo '</div>';
        
        // 5. User Locations
        echo '<div class="bb-subsection">';
        echo '<h3>User Locations</h3>';
        
        $user_locations = $wpdb->get_results("
            SELECT 
                meta_value as location, 
                COUNT(*) as count
            FROM {$wpdb->usermeta}
            WHERE meta_key = 'location' AND meta_value != ''
            GROUP BY meta_value
            ORDER BY count DESC
            LIMIT 50
        ");
        
        echo '<div class="bb-table-responsive">';
        echo '<table class="bb-analytics-table">';
        echo '<thead><tr><th>Location</th><th>Users</th><th>% of Total</th></tr></thead>';
        echo '<tbody>';
        foreach ($user_locations as $location) {
            $percentage = round(($location->count / $total_members) * 100, 2);
            echo '<tr>';
            echo '<td>'.$location->location.'</td>';
            echo '<td>'.number_format($location->count).'</td>';
            echo '<td>'.$percentage.'%</td>';
            echo '</tr>';
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        echo '</div>';
        
        echo '</div>'; // Close section
    }

    private function display_popular_members() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-awards"></i> Top Members</h2>';
        
        $popular_members = $wpdb->get_results("
            SELECT 
                u.ID, 
                u.display_name, 
                um.meta_value as member_type,
                (SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE user_id = u.ID) as activity_count,
                (SELECT COUNT(*) FROM {$wpdb->prefix}bp_follow WHERE followee_id = u.ID) as follower_count,
                (SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity_meta am 
                 JOIN {$wpdb->prefix}bp_activity a ON am.activity_id = a.id 
                 WHERE a.user_id = u.ID AND am.meta_key = 'favorite_count') as like_count
            FROM {$wpdb->users} u
            LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'member_type'
            ORDER BY activity_count DESC
            LIMIT 25
        ");
        
        echo '<div class="bb-table-responsive">';
        echo '<table class="bb-analytics-table">';
        echo '<thead><tr><th>Member</th><th>Type</th><th>Activities</th><th>Followers</th><th>Likes</th><th>Score</th></tr></thead>';
        echo '<tbody>';
        foreach ($popular_members as $member) {
            $score = ($member->activity_count * 1) + ($member->follower_count * 2) + ($member->like_count * 0.5);
            echo '<tr>';
            echo '<td><a href="'.get_edit_user_link($member->ID).'">'.$member->display_name.'</a></td>';
            echo '<td>'.$member->member_type.'</td>';
            echo '<td>'.number_format($member->activity_count).'</td>';
            echo '<td>'.number_format($member->follower_count).'</td>';
            echo '<td>'.number_format($member->like_count).'</td>';
            echo '<td>'.number_format($score).'</td>';
            echo '</tr>';
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        
        echo '</div>';
    }

    // [Previous methods for other sections (groups, activity, network) would be maintained here]
    // For brevity, I'm not including them again since you said they were working fine
    
    private function display_groups_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Groups Analytics</h2>';
        
        // Total groups
        $total_groups = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_groups");
        echo '<p><strong>Total Groups:</strong> '.number_format($total_groups).'</p>';
        
        // Groups by type
        echo '<h3>Groups by Type</h3>';
        $groups_by_type = $wpdb->get_results("
            SELECT status, type, COUNT(*) as count
            FROM {$wpdb->prefix}bp_groups
            GROUP BY status, type
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Privacy</th><th>Type</th><th>Count</th></tr></thead><tbody>';
        foreach ($groups_by_type as $group) {
            echo '<tr>';
            echo '<td>'.$group->status.'</td>';
            echo '<td>'.$group->type.'</td>';
            echo '<td>'.number_format($group->count).'</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        // Active groups (last 3 months)
        $active_groups = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT g.id)
             FROM {$wpdb->prefix}bp_groups g
             JOIN {$wpdb->prefix}bp_activity a ON g.id = a.item_id
             WHERE a.component = 'groups'
             AND a.date_recorded >= %s",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        echo '<p><strong>Active Groups (Last 3 Months):</strong> '.number_format($active_groups).'</p>';
        
        // Active groups by type (last 3 months)
        echo '<h3>Active Groups by Type (Last 3 Months)</h3>';
        $active_groups_by_type = $wpdb->get_results($wpdb->prepare(
            "SELECT g.status, g.type, COUNT(DISTINCT g.id) as count
             FROM {$wpdb->prefix}bp_groups g
             JOIN {$wpdb->prefix}bp_activity a ON g.id = a.item_id
             WHERE a.component = 'groups'
             AND a.date_recorded >= %s
             GROUP BY g.status, g.type",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Privacy</th><th>Type</th><th>Active Count</th></tr></thead><tbody>';
        foreach ($active_groups_by_type as $group) {
            echo '<tr>';
            echo '<td>'.$group->status.'</td>';
            echo '<td>'.$group->type.'</td>';
            echo '<td>'.number_format($group->count).'</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        echo '</div>';
    }

    private function display_activity_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Activity/Post Analytics</h2>';
        
        // Total posts per month (historical)
        echo '<h3>Post Activity Over Time</h3>';
        $posts_history = $wpdb->get_results("
            SELECT DATE_FORMAT(date_recorded, '%Y-%m') as month, 
            COUNT(*) as count
            FROM {$wpdb->prefix}bp_activity
            WHERE type = 'activity_update'
            GROUP BY month
            ORDER BY month
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Month</th><th>Posts</th></tr></thead><tbody>';
        foreach ($posts_history as $month) {
            echo '<tr><td>'.$month->month.'</td><td>'.number_format($month->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        // Posts by media type
        echo '<h3>Posts by Media Type</h3>';
        $media_types = array(
            'Video' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE content LIKE '%<video%'"),
            'Image' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE content LIKE '%<img%'"),
            'Document' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE content LIKE '%.pdf%' OR content LIKE '%.doc%' OR content LIKE '%.xls%'"),
            'Website' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE content LIKE '%<a href=%'"),
            'Text' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE type = 'activity_update' AND 
                                      content NOT LIKE '%<video%' AND content NOT LIKE '%<img%' AND 
                                      content NOT LIKE '%.pdf%' AND content NOT LIKE '%.doc%' AND 
                                      content NOT LIKE '%.xls%' AND content NOT LIKE '%<a href=%'")
        );
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Media Type</th><th>Count</th><th>Percentage</th></tr></thead><tbody>';
        $total_posts = array_sum($media_types);
        foreach ($media_types as $type => $count) {
            $percentage = ($total_posts > 0) ? round(($count / $total_posts) * 100, 2) : 0;
            echo '<tr>';
            echo '<td>'.$type.'</td>';
            echo '<td>'.number_format($count).'</td>';
            echo '<td>'.$percentage.'%</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        // Downloads per month (historical)
        echo '<h3>Downloads Over Time</h3>';
        $downloads_history = $wpdb->get_results("
            SELECT DATE_FORMAT(date_logged, '%Y-%m') as month, 
            COUNT(*) as count
            FROM {$wpdb->prefix}download_log
            GROUP BY month
            ORDER BY month
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Month</th><th>Downloads</th></tr></thead><tbody>';
        foreach ($downloads_history as $month) {
            echo '<tr><td>'.$month->month.'</td><td>'.number_format($month->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        // Active contributors (last 3 months)
        $active_contributors = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT user_id)
             FROM {$wpdb->prefix}bp_activity
             WHERE type = 'activity_update'
             AND date_recorded >= %s",
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        echo '<p><strong>Active Contributors (Last 3 Months):</strong> '.number_format($active_contributors).'</p>';
        
        // Passive users (last 3 months)
        $passive_users = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT um.user_id)
             FROM {$wpdb->usermeta} um
             LEFT JOIN {$wpdb->prefix}bp_activity a ON um.user_id = a.user_id AND a.type = 'activity_update' AND a.date_recorded >= %s
             WHERE um.meta_key = 'last_activity'
             AND um.meta_value >= %s
             AND a.id IS NULL",
            date('Y-m-d H:i:s', strtotime('-3 months')),
            date('Y-m-d H:i:s', strtotime('-3 months'))
        ));
        echo '<p><strong>Passive Users (Last 3 Months):</strong> '.number_format($passive_users).'</p>';
        
        // Engagement metrics
        echo '<h3>Engagement Metrics</h3>';
        $engagement_metrics = array(
            'Average posts read per session' => $wpdb->get_var("SELECT AVG(posts_viewed) FROM {$wpdb->prefix}user_sessions"),
            'Average session duration (minutes)' => $wpdb->get_var("SELECT AVG(duration)/60 FROM {$wpdb->prefix}user_sessions"),
            'Average post viewing time (seconds)' => $wpdb->get_var("SELECT AVG(view_time) FROM {$wpdb->prefix}post_views"),
            'Average sessions per user per month' => $wpdb->get_var("SELECT AVG(session_count) FROM {$wpdb->prefix}user_monthly_stats"),
            'Average likes per user per month' => $wpdb->get_var("SELECT AVG(like_count) FROM {$wpdb->prefix}user_monthly_stats"),
            'Average comments per user per month' => $wpdb->get_var("SELECT AVG(comment_count) FROM {$wpdb->prefix}user_monthly_stats")
        );
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Metric</th><th>Value</th></tr></thead><tbody>';
        foreach ($engagement_metrics as $metric => $value) {
            echo '<tr>';
            echo '<td>'.$metric.'</td>';
            echo '<td>'.round($value, 2).'</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        // Redirection clicks
        echo '<h3>Redirection Clicks</h3>';
        $redirections = $wpdb->get_results("
            SELECT redirection_name, COUNT(*) as count
            FROM {$wpdb->prefix}redirection_clicks
            GROUP BY redirection_name
            ORDER BY count DESC
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Redirection</th><th>Clicks</th></tr></thead><tbody>';
        foreach ($redirections as $redirect) {
            echo '<tr><td>'.$redirect->redirection_name.'</td><td>'.number_format($redirect->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        echo '</div>';
    }

    private function display_network_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Network Analytics</h2>';
        
        // New connections
        echo '<h3>New Connections</h3>';
        $new_connections = $wpdb->get_results("
            SELECT 
            SUM(CASE WHEN type = 'add' THEN 1 ELSE 0 END) as added,
            SUM(CASE WHEN type = 'follow' THEN 1 ELSE 0 END) as followed
            FROM {$wpdb->prefix}connections
            WHERE date_created >= DATE_SUB(NOW(), INTERVAL 3 MONTH)
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Connection Type</th><th>Count (Last 3 Months)</th></tr></thead><tbody>';
        echo '<tr><td>Added</td><td>'.number_format($new_connections[0]->added).'</td></tr>';
        echo '<tr><td>Followed</td><td>'.number_format($new_connections[0]->followed).'</td></tr>';
        echo '</tbody></table>';
        
        // Average connections
        $avg_connections = $wpdb->get_row("
            SELECT 
            AVG(added_count) as avg_added,
            AVG(followed_count) as avg_followed
            FROM (
                SELECT 
                user_id,
                SUM(CASE WHEN type = 'add' THEN 1 ELSE 0 END) as added_count,
                SUM(CASE WHEN type = 'follow' THEN 1 ELSE 0 END) as followed_count
                FROM {$wpdb->prefix}connections
                GROUP BY user_id
            ) as connection_counts
        ");
        
        echo '<h3>Average Connections Per User</h3>';
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Connection Type</th><th>Average Count</th></tr></thead><tbody>';
        echo '<tr><td>Added</td><td>'.round($avg_connections->avg_added, 2).'</td></tr>';
        echo '<tr><td>Followed</td><td>'.round($avg_connections->avg_followed, 2).'</td></tr>';
        echo '</tbody></table>';
        
        echo '</div>';
    }

    private function display_events_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Events Analytics</h2>';
        
        // Events added per month
        echo '<h3>Events Added Over Time</h3>';
        $events_history = $wpdb->get_results("
            SELECT DATE_FORMAT(date_created, '%Y-%m') as month, 
            COUNT(*) as count
            FROM {$wpdb->prefix}events
            GROUP BY month
            ORDER BY month
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Month</th><th>Events Added</th></tr></thead><tbody>';
        foreach ($events_history as $month) {
            echo '<tr><td>'.$month->month.'</td><td>'.number_format($month->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        // Average events per user
        $avg_events = $wpdb->get_var("
            SELECT AVG(event_count)
            FROM (
                SELECT user_id, COUNT(*) as event_count
                FROM {$wpdb->prefix}events
                GROUP BY user_id
            ) as user_events
        ");
        echo '<p><strong>Average Events Added to Calendar Per User:</strong> '.round($avg_events, 2).'</p>';
        
        echo '</div>';
    }

    private function display_discussions_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Discussions Analytics</h2>';
        
        // Total discussions
        $total_discussions = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}discussions");
        echo '<p><strong>Total Discussions:</strong> '.number_format($total_discussions).'</p>';
        
        // Discussions by type
        echo '<h3>Discussions by Type</h3>';
        $discussions_by_type = $wpdb->get_results("
            SELECT is_private, COUNT(*) as count
            FROM {$wpdb->prefix}discussions
            GROUP BY is_private
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Type</th><th>Count</th></tr></thead><tbody>';
        foreach ($discussions_by_type as $discussion) {
            $type = $discussion->is_private ? 'Private' : 'Public';
            echo '<tr><td>'.$type.'</td><td>'.number_format($discussion->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        // Discussion metrics
        echo '<h3>Discussion Engagement</h3>';
        $discussion_metrics = array(
            'Average replies per discussion' => $wpdb->get_var("SELECT AVG(reply_count) FROM {$wpdb->prefix}discussion_stats"),
            'Average members per discussion' => $wpdb->get_var("SELECT AVG(member_count) FROM {$wpdb->prefix}discussion_stats"),
            'Average discussions per group' => $wpdb->get_var("SELECT AVG(discussion_count) FROM {$wpdb->prefix}group_discussion_stats")
        );
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Metric</th><th>Value</th></tr></thead><tbody>';
        foreach ($discussion_metrics as $metric => $value) {
            echo '<tr>';
            echo '<td>'.$metric.'</td>';
            echo '<td>'.round($value, 2).'</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        echo '</div>';
    }

    private function display_searches_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Search Analytics</h2>';
        
        // Searches per month
        echo '<h3>Search Activity Over Time</h3>';
        $searches_history = $wpdb->get_results("
            SELECT DATE_FORMAT(search_date, '%Y-%m') as month, 
            COUNT(*) as count
            FROM {$wpdb->prefix}search_log
            GROUP BY month
            ORDER BY month
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Month</th><th>Searches</th></tr></thead><tbody>';
        foreach ($searches_history as $month) {
            echo '<tr><td>'.$month->month.'</td><td>'.number_format($month->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        // Top keywords
        echo '<h3>Top 20 Search Keywords</h3>';
        $top_keywords = $wpdb->get_results("
            SELECT search_term, COUNT(*) as count
            FROM {$wpdb->prefix}search_log
            GROUP BY search_term
            ORDER BY count DESC
            LIMIT 20
        ");
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Keyword</th><th>Searches</th></tr></thead><tbody>';
        foreach ($top_keywords as $keyword) {
            echo '<tr><td>'.$keyword->search_term.'</td><td>'.number_format($keyword->count).'</td></tr>';
        }
        echo '</tbody></table>';
        
        echo '</div>';
    }

    private function display_messages_section() {
        global $wpdb;
        
        echo '<div class="analytics-section">';
        echo '<h2>Messages Analytics</h2>';
        
        // Message metrics
        echo '<h3>Message Statistics</h3>';
        $message_metrics = array(
            'Average messages received per user' => $wpdb->get_var("SELECT AVG(message_count) FROM {$wpdb->prefix}user_message_stats"),
            'Average conversations per user' => $wpdb->get_var("SELECT AVG(conversation_count) FROM {$wpdb->prefix}user_message_stats")
        );
        
        echo '<table class="analytics-table">';
        echo '<thead><tr><th>Metric</th><th>Value</th></tr></thead><tbody>';
        foreach ($message_metrics as $metric => $value) {
            echo '<tr>';
            echo '<td>'.$metric.'</td>';
            echo '<td>'.round($value, 2).'</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        
        echo '</div>';
    }
}

new BuddyBoss_Complete_Analytics();